<?php

/**
 * main file class
 */
class file
{

    public $errorMsg = null;

    function __construct()
    {
        $this->errorMsg = null;
    }

    public function download($forceDownload = true, $doPluginIncludes = true)
    {
        // remove session
        if (isset($_SESSION['showDownload']))
        {
            $clearSession = true;
            
            // fixes android snag which requests files twice
            if(deviceIsAndroid())
            {
                if(!isset($_SESSION['showDownloadFirstRun']))
                {
                    $_SESSION['showDownloadFirstRun'] = true;
                    $clearSession = false;
                }
                else
                {
                    $_SESSION['showDownloadFirstRun'] = null;
                    unset($_SESSION['showDownloadFirstRun']);
                }
            }
            
            if($clearSession == true)
            {
                // reset session variable for next time
                $_SESSION['showDownload'] = null;
                unset($_SESSION['showDownload']);
                session_write_close();
            }
        }

        // get user
        $Auth = Auth::getAuth();

        // check for concurrent downloads for paid users
        if ((int) SITE_CONFIG_PAID_USER_MAX_DOWNLOAD_THREADS > 0)
        {
			// get database
            $db = Database::getDatabase(true);
			
            $sQL          = "SELECT COUNT(download_tracker.id) AS total_threads ";
            $sQL .= "FROM download_tracker ";
            $sQL .= "WHERE download_tracker.status='downloading' AND download_tracker.download_username = " . $db->quote(getUsersIPAddress()) . " ";
            $sQL .= "GROUP BY download_tracker.download_username ";
            $totalThreads = (int) $db->getValue($sQL);
            if ($totalThreads >= (int) SITE_CONFIG_PAID_USER_MAX_DOWNLOAD_THREADS)
            {
                // fail
                header("HTTP/1.0 403 Forbidden");
                exit;
            }
        }

        // php script timeout for long downloads (2 days!)
        set_time_limit(60 * 60 * 24 * 2);

        // load the server the file is on
        $storageType         = 'local';
        $storageLocation     = _CONFIG_FILE_STORAGE_PATH;
        $uploadServerDetails = $this->loadServer();
        if ($uploadServerDetails != false)
        {
            $storageLocation = $uploadServerDetails['storagePath'];
            $storageType     = $uploadServerDetails['serverType'];

            // if no storage path set & local, use system default
            if ((strlen($storageLocation) == 0) && ($storageType == 'local'))
            {
                $storageLocation = _CONFIG_FILE_STORAGE_PATH;
            }
        }

        // get file path
        $fullPath = $this->getFullFilePath($storageLocation);

        // open file - via ftp
        if ($storageType == 'ftp')
        {
            // connect via ftp
            $conn_id = ftp_connect($uploadServerDetails['ipAddress'], $uploadServerDetails['ftpPort'], 30);
            if ($conn_id === false)
            {
                $this->errorMsg = 'Could not connect to ' . $uploadServerDetails['ipAddress'] . ' to upload file.';
                return false;
            }

            // authenticate
            $login_result = ftp_login($conn_id, $uploadServerDetails['ftpUsername'], $uploadServerDetails['ftpPassword']);
            if ($login_result === false)
            {
                $this->errorMsg = 'Could not login to ' . $uploadServerDetails['ipAddress'] . ' with supplied credentials.';
                return false;
            }

            // prepare the stream of data
            $pipes = stream_socket_pair(STREAM_PF_UNIX, STREAM_SOCK_STREAM, STREAM_IPPROTO_IP);
            if ($pipes === false)
            {
                $this->errorMsg = 'Could not create stream to download file on ' . $uploadServerDetails['ipAddress'];
                return false;
            }

            stream_set_write_buffer($pipes[0], 10000);
            stream_set_timeout($pipes[1], 10);
            stream_set_blocking($pipes[1], 0);

            $fail = false;
        }
        elseif($storageType == 'direct')
        {
            $fullPath = _CONFIG_SCRIPT_ROOT.'/'.$fullPath;
        }

        // download speed
        $_SESSION['speed'] = 0;

        // if free/non user
        if (($Auth->loggedIn == false) || ($Auth->level == 'free user'))
        {
            $_SESSION['speed'] = (int) SITE_CONFIG_FREE_USER_MAX_DOWNLOAD_SPEED;
        }
        else
        {
            $_SESSION['speed'] = (int) SITE_CONFIG_PREMIUM_USER_MAX_DOWNLOAD_SPEED;
        }

        if ($forceDownload == true)
        {
            // include any plugin includes
            if ($doPluginIncludes == true)
            {
                pluginHelper::includeAppends('class_file_download.php');
            }
        }
        else
        {
            $_SESSION['speed'] = 0;
        }

        // do we need to throttle the speed?
        if ($_SESSION['speed'] > 0)
        {
            // create new throttle config
            $config = new ThrottleConfig();

            // set standard transfer rate (in bytes/second)
            $config->burstLimit = $_SESSION['speed'];
            $config->rateLimit  = $_SESSION['speed'];

            // enable module (this is a default value)
            $config->enabled = true;

            // start throttling
            $x = new Throttle($config);
        }

        // handle where to start in the download, support for resumed downloads
        $seekStart = 0;
        $seekEnd   = -1;
        if (isset($_SERVER['HTTP_RANGE']) || isset($HTTP_SERVER_VARS['HTTP_RANGE']))
        {
            if (isset($HTTP_SERVER_VARS['HTTP_RANGE']))
            {
                $seekRange = substr($HTTP_SERVER_VARS['HTTP_RANGE'], strlen('bytes='));
            }
            else
            {
                $seekRange = substr($_SERVER['HTTP_RANGE'], strlen('bytes='));
            }

            $range = explode('-', $seekRange);
            if ((int) $range[0] > 0)
            {
                $seekStart = intval($range[0]);
            }

            if ((int) $range[1] > 0)
            {
                $seekEnd = intval($range[1]);
            }
        }

        if ($forceDownload == true)
        {
            // output some headers
            header("Expires: 0");
            header("Cache-Control: must-revalidate, post-check=0, pre-check=0");
            header("Content-type: " . $this->fileType);
            header("Pragma: public");
            header("Content-Disposition: attachment; filename=\"" . str_replace("\"", "", $this->originalFilename) . "\"");
            header("Content-Description: File Transfer");

            if ($seekStart > 0)
            {
                header("HTTP/1.0 206 Partial Content");
                header("Status: 206 Partial Content");
                header('Accept-Ranges: bytes');
                header("Content-Range: bytes $seekStart-$seekEnd/$this->fileSize");
                header("Content-Length: " . ($seekEnd - $seekStart + 1));
            }
            else
            {
                header("Content-Length: " . $this->fileSize);
            }
        }

        if (SITE_CONFIG_DOWNLOADS_TRACK_CURRENT_DOWNLOADS == 'yes')
        {
            // track downloads
            $downloadTracker = new downloadTracker($this);
            $downloadTracker->create($seekStart, $seekEnd);
        }

        // for returning the file contents
        if ($forceDownload == false)
        {
            $fileContent = '';
        }

        // output file - via ftp
        $timeTracker = time();
        $length      = 0;
        if ($storageType == 'ftp')
        {
            $ret = ftp_nb_fget($conn_id, $pipes[0], $fullPath, FTP_BINARY, $seekStart);
            while ($ret == FTP_MOREDATA)
            {
                $contents = stream_get_contents($pipes[1], $seekEnd);
                if ($contents !== false)
                {
                    if ($forceDownload == true)
                    {
                        echo $contents;
                    }
                    else
                    {
                        $fileContent .= $contents;
                    }
                    $length = $length + strlen($contents);
                    flush();
                }

                $ret = ftp_nb_continue($conn_id);

                // update download status every DOWNLOAD_TRACKER_UPDATE_FREQUENCY seconds
                if (($timeTracker + DOWNLOAD_TRACKER_UPDATE_FREQUENCY) < time())
                {
                    $timeTracker = time();
                    if (SITE_CONFIG_DOWNLOADS_TRACK_CURRENT_DOWNLOADS == 'yes')
                    {
                        $downloadTracker->update();
                    }
                }
            }

            fclose($pipes[0]);
            fclose($pipes[1]);
        }
        // output file - local
        else
        {
            // open file - locally
            $handle = @fopen($fullPath, "r");
            if (!$handle)
            {
                $this->errorMsg = 'Could not open file for reading.';
                return false;
            }

            // move to starting position
            fseek($handle, $seekStart);

            while (($buffer = fgets($handle, 4096)) !== false)
            {
                if ($forceDownload == true)
                {
                    echo $buffer;
                }
                else
                {
                    $fileContent .= $buffer;
                }
                $length = $length + strlen($buffer);

                // update download status every DOWNLOAD_TRACKER_UPDATE_FREQUENCY seconds
                if (($timeTracker + DOWNLOAD_TRACKER_UPDATE_FREQUENCY) < time())
                {
                    $timeTracker = time();
                    if (SITE_CONFIG_DOWNLOADS_TRACK_CURRENT_DOWNLOADS == 'yes')
                    {
                        $downloadTracker->update();
                    }
                }
            }
            fclose($handle);
        }

        if ($forceDownload == true)
        {
            // stats
            if (($Auth->loggedIn == true) && ($Auth->id != $this->userId) && ($Auth->level == 'admin'))
            {
                // dont update stats, this was triggered by an admin user
            }
            else
            {
                // update stats
                $rs = Stats::track($this, $this->id);
                if ($rs)
                {
                    $this->updateLastAccessed();
                }
            }
        }

		// finish off any plugins
        pluginHelper::includeAppends('class_file_download_complete.php', array('forceDownload'    => $forceDownload, 'Auth'             => $Auth, 'file'             => $this, 'doPluginIncludes' => $doPluginIncludes));
		
        if (SITE_CONFIG_DOWNLOADS_TRACK_CURRENT_DOWNLOADS == 'yes')
        {
            // close download
            $downloadTracker->finish();
        }

        // return file content
        if ($forceDownload == false)
        {
            return $fileContent;
        }

        exit();
    }

    public function loadServer()
    {
        // load the server the file is on
        if ((int) $this->serverId)
        {
            // load from the db
            $db                  = Database::getDatabase(true);
            $uploadServerDetails = $db->getRow('SELECT * FROM file_server WHERE id = ' . $db->quote((int) $this->serverId));
            $db->close();
            if (!$uploadServerDetails)
            {
                return false;
            }

            return $uploadServerDetails;
        }

        return false;
    }

    public function getFullFilePath($prePath = '')
    {
        if (substr($prePath, strlen($prePath) - 1, 1) == '/')
        {
            $prePath = substr($prePath, 0, strlen($prePath) - 1);
        }

        return $prePath . '/' . $this->localFilePath;
    }

    /**
     * Get full short url path
     *
     * @return string
     */
    public function getFullShortUrl()
    {
        if (SITE_CONFIG_FILE_URL_SHOW_FILENAME == 'yes')
        {
            return $this->getFullLongUrl();
        }

        return $this->getShortUrlPath();
    }

    public function getShortUrlPath()
    {
        $fileServerPath = file::getFileDomainAndPath($this->id, $this->serverId);

        return _CONFIG_SITE_PROTOCOL . '://' . $fileServerPath . '/' . $this->shortUrl;
    }

    public function getStatisticsUrl()
    {
        return $this->getShortUrlPath() . '~s';
    }

    public function getDeleteUrl()
    {
        return $this->getShortUrlPath() . '~d?' . $this->deleteHash;
    }

    public function getInfoUrl()
    {
        return $this->getShortUrlPath() . '~i?' . $this->deleteHash;
    }

    public function getShortInfoUrl()
    {
        return $this->getShortUrlPath() . '~i';
    }

    /**
     * Get full long url including the original filename
     *
     * @return string
     */
    public function getFullLongUrl()
    {
        return $this->getShortUrlPath() . '/' . str_replace(array(" ", "\"", "'", ";"), "_", strip_tags($this->originalFilename));
    }

    /**
     * Method to increment visitors
     */
    public function updateVisitors()
    {
        $db = Database::getDatabase(true);
        $this->visits++;
        $db->query('UPDATE file SET visits = :visits WHERE id = :id', array('visits' => $this->visits, 'id'     => $this->id));
    }

    /**
     * Method to update last accessed
     */
    public function updateLastAccessed()
    {
        $db = Database::getDatabase(true);
        $db->query('UPDATE file SET lastAccessed = NOW() WHERE id = :id', array('id' => $this->id));
    }

    /**
     * Method to set folder
     */
    public function updateFolder($folderId = '')
    {
        $db       = Database::getDatabase(true);
        $folderId = (int) $folderId;
        if ($folderId == 0)
        {
            $folderId = '';
        }
        $db->query('UPDATE file SET folderId = :folderId WHERE id = :id', array('folderId' => $folderId, 'id'       => $this->id));
    }

    /**
     * Remove by user
     */
    public function removeByUser()
    {
        // load the server the file is on
        $storageType         = 'local';
        $storageLocation     = _CONFIG_FILE_STORAGE_PATH;
        $uploadServerDetails = $this->loadServer();
        if ($uploadServerDetails != false)
        {
            $storageLocation = $uploadServerDetails['storagePath'];
            $storageType     = $uploadServerDetails['serverType'];
            if ((strlen($uploadServerDetails['storagePath']) == 0) && ($storageType == 'local'))
            {
                $storageLocation = _CONFIG_FILE_STORAGE_PATH;
            }
        }

        // file path
        $filePath = $this->getFullFilePath($storageLocation);

        // remote - ftp
        if ($storageType == 'ftp')
        {
            // connect via ftp
            $conn_id = ftp_connect($uploadServerDetails['ipAddress'], $uploadServerDetails['ftpPort'], 30);
            if ($conn_id === false)
            {
                $this->errorMsg = 'Could not connect to ' . $uploadServerDetails['ipAddress'] . ' to upload file.';
                return false;
            }

            // authenticate
            $login_result = ftp_login($conn_id, $uploadServerDetails['ftpUsername'], $uploadServerDetails['ftpPassword']);
            if ($login_result === false)
            {
                $this->errorMsg = 'Could not login to ' . $uploadServerDetails['ipAddress'] . ' with supplied credentials.';
                return false;
            }

            // remove file
            if (!ftp_delete($conn_id, $filePath))
            {
                $this->errorMsg = 'Could not remove file on ' . $uploadServerDetails['ipAddress'];
                return false;
            }
        }
        // enable removal of 'direct' stored files
        elseif ($storageType == 'direct')
        {
            if(!file_exists($fullPath))
            {
                $fullPath = _CONFIG_SCRIPT_ROOT . '/' . $fullPath;
            }
            
            unlink($filePath);
        }
        // local
        else
        {
            // delete file from server
            unlink($filePath);
        }

        // update db
        $db = Database::getDatabase(true);
        $db->query('UPDATE file SET statusId = 2 WHERE id = :id', array('id' => $this->id));
    }

    /**
     * Remove by system
     */
    public function removeBySystem()
    {
        // load the server the file is on
        $storageType         = 'local';
        $storageLocation     = _CONFIG_FILE_STORAGE_PATH;
        $uploadServerDetails = $this->loadServer();
        if ($uploadServerDetails != false)
        {
            $storageLocation = $uploadServerDetails['storagePath'];
            $storageType     = $uploadServerDetails['serverType'];
            if ((strlen($uploadServerDetails['storagePath']) == 0) && ($storageType == 'local'))
            {
                $storageLocation = _CONFIG_FILE_STORAGE_PATH;
            }
        }

        // file path
        $filePath = $this->getFullFilePath($storageLocation);

        // remote - ftp
        if ($storageType == 'ftp')
        {
            // connect via ftp
            $conn_id = ftp_connect($uploadServerDetails['ipAddress'], $uploadServerDetails['ftpPort'], 30);
            if ($conn_id === false)
            {
                $this->errorMsg = 'Could not connect to ' . $uploadServerDetails['ipAddress'] . ' to upload file.';
                return false;
            }

            // authenticate
            $login_result = ftp_login($conn_id, $uploadServerDetails['ftpUsername'], $uploadServerDetails['ftpPassword']);
            if ($login_result === false)
            {
                $this->errorMsg = 'Could not login to ' . $uploadServerDetails['ipAddress'] . ' with supplied credentials.';
                return false;
            }

            // remove file
            if (!ftp_delete($conn_id, $filePath))
            {
                $this->errorMsg = 'Could not remove file on ' . $uploadServerDetails['ipAddress'];
                return false;
            }
        }
        // enable removal of 'direct' stored files
        elseif ($storageType == 'direct')
        {
            if(!file_exists($fullPath))
            {
                $fullPath = _CONFIG_SCRIPT_ROOT . '/' . $fullPath;
            }
            
            unlink($filePath);
        }
        // local
        else
        {
            // delete file from server
            unlink($filePath);
        }

        // update db
        $db = Database::getDatabase(true);
        $db->query('UPDATE file SET statusId = 5 WHERE id = :id', array('id' => $this->id));
    }

    public function getLargeIconPath()
    {
        $fileTypePath = DOC_ROOT . '/themes/' . SITE_CONFIG_SITE_THEME . '/images/file_icons/512px/' . strtolower($this->extension) . '.png';
        if (!file_exists($fileTypePath))
        {
            return false;
        }

        return SITE_IMAGE_PATH . '/file_icons/512px/' . strtolower($this->extension) . '.png';
    }

    public function getFilenameExcExtension()
    {
        $filename = $this->originalFilename;

        return basename($filename, '.' . $this->extension);
    }
    
    /**
     * Method to set password
     */
    public function updatePassword($password = '')
    {
        $db       = Database::getDatabase(true);
        $md5Password = '';
        if (strlen($password))
        {
            $md5Password = md5($password);
        }

        $db->query('UPDATE file SET accessPassword = :accessPassword WHERE id = :id', array('accessPassword' => $md5Password, 'id' => $this->id));
    }

    /**
     * Load by short url
     *
     * @param string $shortUrl
     * @return file
     */
    static function loadByShortUrl($shortUrl)
    {
        $db  = Database::getDatabase(true);
        $row = $db->getRow('SELECT * FROM file WHERE shortUrl = ' . $db->quote($shortUrl));
        if (!is_array($row))
        {
            return false;
        }

        $fileObj = new file();
        foreach ($row AS $k => $v)
        {
            $fileObj->$k = $v;
        }

        return $fileObj;
    }

    /**
     * Load by delete hash
     *
     * @param string $deleteHash
     * @return file
     */
    static function loadByDeleteHash($deleteHash)
    {
        $db  = Database::getDatabase(true);
        $row = $db->getRow('SELECT * FROM file WHERE deleteHash = ' . $db->quote($deleteHash));
        if (!is_array($row))
        {
            return false;
        }

        $fileObj = new file();
        foreach ($row AS $k => $v)
        {
            $fileObj->$k = $v;
        }

        return $fileObj;
    }

    /**
     * Load by id
     *
     * @param integer $shortUrl
     * @return file
     */
    static function loadById($id)
    {
        $db  = Database::getDatabase(true);
        $row = $db->getRow('SELECT * FROM file WHERE id = ' . (int) $id);
        if (!is_array($row))
        {
            return false;
        }

        $fileObj = new file();
        foreach ($row AS $k => $v)
        {
            $fileObj->$k = $v;
        }

        return $fileObj;
    }

    /**
     * Create short url
     *
     * @param integer $in
     * @return string
     */
    static function createShortUrlPart($in)
    {
        // note: no need to check whether it already exists as it's handled by the code which calls this
        switch (SITE_CONFIG_GENERATE_UPLOAD_URL_TYPE)
        {
            case 'Medium Hash':
                return substr(MD5($in . microtime()), 0, 16);
                break;
            case 'Long Hash':
                return MD5($in . microtime());
                break;
        }

        // Shortest
        $codeset  = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
        $cbLength = strlen($codeset);
        $out      = '';
        while ((int) $in > $cbLength - 1)
        {
            $out = $codeset[fmod($in, $cbLength)] . $out;
            $in  = floor($in / $cbLength);
        }

        return $codeset[$in] . $out;
    }

    /**
     * Update short url for file
     *
     * @param integer $id
     * @param string $shortUrl
     */
    static function updateShortUrl($id, $shortUrl = '')
    {
        $db = Database::getDatabase(true);
        $db->query('UPDATE file SET shortUrl = :shorturl WHERE id = :id', array('shorturl' => $shortUrl, 'id'       => $id));
    }

    /**
     * Load all by account id
     *
     * @param integer $accountId
     * @return array
     */
    static function loadAllByAccount($accountId)
    {
        $db = Database::getDatabase(true);
        $rs = $db->getRows('SELECT * FROM file WHERE userId = ' . $db->quote($accountId) . ' ORDER BY originalFilename');
        if (!is_array($rs))
        {
            return false;
        }

        return $rs;
    }

    /**
     * Load all active by folder id
     *
     * @param integer $folderId
     * @return array
     */
    static function loadAllActiveByFolderId($folderId)
    {
        $db = Database::getDatabase(true);
        $rs = $db->getRows('SELECT * FROM file WHERE folderId = ' . $db->quote($folderId) . ' AND statusId = 1 ORDER BY originalFilename');
        if (!is_array($rs))
        {
            return false;
        }

        return $rs;
    }

    /**
     * Load all active by account id
     *
     * @param integer $accountId
     * @return array
     */
    static function loadAllActiveByAccount($accountId)
    {
        $db = Database::getDatabase(true);
        $rs = $db->getRows('SELECT * FROM file WHERE userId = ' . $db->quote($accountId) . ' AND statusId = 1 ORDER BY originalFilename');
        if (!is_array($rs))
        {
            return false;
        }

        return $rs;
    }

    /**
     * Load recent files based on account id
     *
     * @param integer $accountId
     * @return array
     */
    static function loadAllRecentByAccount($accountId, $activeOnly = false)
    {
        $db = Database::getDatabase(true);
        $rs = $db->getRows('SELECT * FROM file WHERE userId = ' . $db->quote($accountId) . ($activeOnly === true ? ' AND statusId=1' : '') . ' ORDER BY uploadedDate DESC LIMIT 10');
        if (!is_array($rs))
        {
            return false;
        }

        return $rs;
    }

    /**
     * Load recent files based on IP address
     *
     * @param string $ip
     * @return array
     */
    static function loadAllRecentByIp($ip, $activeOnly = false)
    {
        $db = Database::getDatabase(true);
        $rs = $db->getRows('SELECT * FROM file WHERE uploadedIP = ' . $db->quote($ip) . ($activeOnly === true ? ' AND statusId=1' : '') . ' AND userId IS NULL ORDER BY uploadedDate DESC LIMIT 10');
        if (!is_array($rs))
        {
            return false;
        }

        return $rs;
    }

    /**
     * Get status label
     *
     * @param integer $statusId
     * @return string
     */
    static function getStatusLabel($statusId)
    {
        $db  = Database::getDatabase(true);
        $row = $db->getRow('SELECT label FROM file_status WHERE id = ' . (int) $statusId);
        if (!is_array($row))
        {
            return 'unknown';
        }

        return $row['label'];
    }

    static function getUploadUrl()
    {
        // get available file server
        $db            = Database::getDatabase(true);
        $fileServerId  = getAvailableServerId();
        $sQL           = "SELECT serverType, fileServerDomainName, scriptPath FROM file_server WHERE id = " . (int) $fileServerId . " LIMIT 1";
        $serverDetails = $db->getRow($sQL);
        if ($serverDetails)
        {
            if ($serverDetails['serverType'] == 'direct')
            {
                $url = $serverDetails['fileServerDomainName'] . $serverDetails['scriptPath'];
                if (substr($url, strlen($url) - 1, 1) == '/')
                {
                    $url = substr($url, 0, strlen($url) - 1);
                }

                return _CONFIG_SITE_PROTOCOL . "://" . $url;
            }
        }

        return WEB_ROOT;
    }

    /*
     * Get all direct file servers
     */

    static function getDirectFileServers()
    {
        $db  = Database::getDatabase(true);
        $sQL = "SELECT id, serverType, fileServerDomainName, scriptPath FROM file_server WHERE serverType='direct' ORDER BY fileServerDomainName";

        return $db->getRows($sQL);
    }

    static function getValidReferrers($formatted = false)
    {
        $pre = '';
        if ($formatted == true)
        {
            $pre = _CONFIG_SITE_PROTOCOL . '://';
        }

        $validUrls                               = array();
        $validUrls[$pre . _CONFIG_SITE_HOST_URL] = $pre . _CONFIG_SITE_HOST_URL;
        $directFileServers                       = self::getDirectFileServers();
        if (COUNT($directFileServers))
        {
            foreach ($directFileServers AS $directFileServer)
            {
                $validUrls[$pre . $directFileServer{'fileServerDomainName'}] = $pre . $directFileServer['fileServerDomainName'];
            }
        }

        return $validUrls;
    }

    static function getFileDomainAndPath($fileId, $fileServerId = null)
    {
        // get database connection
        $db = Database::getDatabase(true);
        if ($fileServerId == null)
        {
            $fileServerId = $db->getValue('SELECT serverId FROM file WHERE id = ' . (int) $fileId . ' LIMIT 1');
        }

        if ((int) $fileServerId)
        {
            // use caching for better database performance
            if(!cache::cacheExists('FILE_SERVERS'))
            {
                $fileServers = $db->getRows('SELECT id, fileServerDomainName, scriptPath FROM file_server');
                foreach($fileServers AS $fileServer)
                {
                    $rs[$fileServer{'id'}] = $fileServer;
                }
                cache::setCache('FILE_SERVERS', $rs);
            }
            
            $fileServers = cache::getCache('FILE_SERVERS');
            $fileServer = $fileServers[$fileServerId];
            if ($fileServer)
            {
                if (strlen($fileServer['fileServerDomainName']))
                {
                    $path = $fileServer['fileServerDomainName'] . $fileServer['scriptPath'];
                    if (substr($path, strlen($path) - 1, 1) == '/')
                    {
                        $path = substr($path, 0, strlen($path) - 1);
                    }

                    return $path;
                }
            }
        }

        return _CONFIG_SITE_FILE_DOMAIN;
    }

    static function getFileUrl($fileId, $file = null)
    {
        if (!$file)
        {
            $file = file::loadById((int) $fileId);
        }

        if (!$file)
        {
            return false;
        }

        return $file->getFullShortUrl();
    }
    
    static function getFileStatisticsUrl($fileId, $file = null)
    {
        if (!$file)
        {
            $file = file::loadById((int) $fileId);
        }

        if (!$file)
        {
            return false;
        }

        return $file->getStatisticsUrl();
    }
    
    static function getFileDeleteUrl($fileId, $file = null)
    {
        if (!$file)
        {
            $file = file::loadById((int) $fileId);
        }

        if (!$file)
        {
            return false;
        }

        return $file->getDeleteUrl();
    }
    
    static function getFileInfoUrl($fileId, $file = null)
    {
        if (!$file)
        {
            $file = file::loadById((int) $fileId);
        }

        if (!$file)
        {
            return false;
        }

        return $file->getInfoUrl();
    }
    
    static function getFileShortInfoUrl($fileId, $file = null)
    {
        if (!$file)
        {
            $file = file::loadById((int) $fileId);
        }

        if (!$file)
        {
            return false;
        }

        return $file->getShortInfoUrl();
    }
}
