<?php

class Brivium_AttachmentEssential_Model_AttachmentEssential extends XenForo_Model
{
	const FETCH_USER     	= 0x01;
	const FETCH_USER_FULL	= 0x02;

	public function getDownloadById($downloadId, array $fetchOptions = array())
	{
		if (empty($downloadId))
		{
			return false;
		}

		$joinOptions = $this->prepareDownloadFetchOptions($fetchOptions);

		return $this->_getDb()->fetchRow('
			SELECT download.*
				' . $joinOptions['selectFields'] . '
			FROM xf_attachment_essential AS download
			' . $joinOptions['joinTables'] . '
			WHERE download.download_id = ?
		', $downloadId);
	}

	public function getLimitById($userId, array $fetchOptions = array())
	{
		if (empty($userId))
		{
			return false;
		}

		$joinOptions = $this->prepareDownloadFetchOptions($fetchOptions);

		return $this->_getDb()->fetchRow('
			SELECT download.*
				' . $joinOptions['selectFields'] . '
			FROM xf_attachment_essential_limit AS download
			' . $joinOptions['joinTables'] . '
			WHERE download.user_id = ?
		', $userId);
	}

	public function getDownloads(array $conditions, array $fetchOptions = array())
	{
		$whereClause = $this->prepareDownloadConditions($conditions, $fetchOptions);
		$joinOptions = $this->prepareDownloadFetchOptions($fetchOptions);
		$orderClause = $this->prepareDownloadOrderOptions($fetchOptions, 'download.download_date');

		$query = ('
			SELECT download.*
				' . $joinOptions['selectFields'] . '
			FROM xf_attachment_essential AS download
			' . $joinOptions['joinTables'] . '
			WHERE ' . $whereClause . '
			' . $orderClause . '
		');

		if (!empty($fetchOptions['fetch']))
		{
			if ($fetchOptions['fetch'] == 'allKeyed')
			{
				return $this->fetchAllKeyed($query, 'download_id');
			}
		}
		else
		{
			return $this->_getDb()->fetchRow($query);
		}
	}

	public function getMostDownload(array $conditions = array(), array $fetchOptions = array())
	{
		$joinOptions = $this->prepareDownloadFetchOptions($fetchOptions);
		$orderClause = $this->prepareDownloadOrderOptions($fetchOptions, 'total_download');
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);

		return $this->_getDb()->fetchAll($this->limitQueryResults('
				SELECT SUM(download_count) AS total_download
					' . $joinOptions['selectFields'] . '
				FROM xf_attachment_essential AS download
				' . $joinOptions['joinTables'] . '
				GROUP BY download.user_id
				' . $orderClause . '
				', $limitOptions['limit'], $limitOptions['offset']
		));
	}

	public function prepareDownloadConditions(array $conditions, array &$fetchOptions)
	{
		$db = $this->_getDb();
		$sqlConditions = array();

		if (!empty($conditions['download_id']))
		{
			if (is_array($conditions['download_id']))
			{
				$sqlConditions[] = 'download.download_id IN(' . $db->quote($conditions['download_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'download.download_id = ' . $db->quote($conditions['download_id']);
			}
		}

		if (!empty($conditions['attachment_id']))
		{
			if (is_array($conditions['attachment_id']))
			{
				$sqlConditions[] = 'download.attachment_id IN(' . $db->quote($conditions['attachment_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'download.attachment_id = ' . $db->quote($conditions['attachment_id']);
			}
		}

		if (!empty($conditions['user_id']))
		{
			if (is_array($conditions['user_id']))
			{
				$sqlConditions[] = 'download.user_id IN(' . $db->quote($conditions['user_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'download.user_id = ' . $db->quote($conditions['user_id']);
			}
		}

		return $this->getConditionsForClause($sqlConditions);
	}

	public function prepareDownloadFetchOptions(array &$fetchOptions, $defaultOrderSql = '')
	{
		$selectFields = '';
		$joinTables = '';

		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_USER)
			{
				$selectFields .= ',
					user.*';
				$joinTables .= '
					INNER JOIN xf_user AS user ON
						(download.user_id = user.user_id)';
			}
			if ($fetchOptions['join'] & self::FETCH_USER_FULL)
			{
				$selectFields .= ',
					user.*,
					user_profile.*,
					user_option.*';
				$joinTables .= '
					INNER JOIN xf_user AS user ON
						(download.user_id = user.user_id)
					INNER JOIN xf_user_profile AS user_profile ON
						(download.user_id = user_profile.user_id)
					INNER JOIN xf_user_option AS user_option ON
						(download.user_id = user_option.user_id)';
			}
		}

		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables
		);
	}

	public function prepareDownloadOrderOptions(array &$fetchOptions, $defaultOrderSql = '')
	{
		$choices = array(
			'username' => 'user.username',
			'download_count' => 'download.download_count'
		);
		return $this->getOrderByClause($choices, $fetchOptions, $defaultOrderSql);
	}

	public function prepareDownload(array $download)
	{
		$download['attachment'] = XenForo_Model::create('XenForo_Model_Attachment')->getAttachmentById($download['attachment_id']);

		return $download;
	}

	public function prepareDownloads(array $downloads)
	{
		foreach ($downloads AS &$download) 
		{
			$download = $this->prepareDownload($download);
		}

		return $downloads;
	}

	public function getDiskSpaceUsage()
	{
		return $this->_getDb()->fetchOne('
			SELECT SUM(file_size) AS disk_usage
			FROM xf_attachment_data
		');
	}

	public function getTotalDownload()
	{
		return $this->_getDb()->fetchOne('
			SELECT SUM(view_count) AS download_count
			FROM xf_attachment
		');
	}

	public function getAttachmentTotalsCounter()
	{
		$output = array(
			'attachments' => $this->getModelFromCache('XenForo_Model_Attachment')->countAttachments(),
			'disk_usage' => $this->getDiskSpaceUsage(),
			'download_count' => $this->getTotalDownload()
		);

		return $output;
	}

	public function rebuildAttachmentTotalsCounter()
	{
		$counter = $this->getAttachmentTotalsCounter();

		$this->getModelFromCache('XenForo_Model_DataRegistry')->set('attachmentTotals', $counter);
		return $counter;
	}

	public function isDownloaded(array $attachment, $userId) 
	{
		$found = $this->_getDb()->fetchOne('
			SELECT user_id
			FROM xf_attachment_essential 
			WHERE attachment_id = ? AND user_id = ?'
		, array($attachment['attachment_id'], $userId));
		
		return $found == $userId;
	}

	public function logAttachmentTracker(array $attachment, $userId)
	{
		if (empty($attachment) || empty($userId))
		{
			return false;
		}

		$dw = XenForo_DataWriter::create('Brivium_AttachmentEssential_DataWriter_Attachment');
		
		$dw->set('user_id', $userId);
		$dw->set('attachment_id', $attachment['attachment_id']);
		$dw->save();
	}

	public function updateAttachmentTracker(array $attachment, $userId)
	{
		if (empty($attachment) || empty($userId))
		{
			return false;
		}
		
		$this->_getDb()->query('
			UPDATE xf_attachment_essential 
			SET download_date = ?, download_count = download_count + 1
			WHERE attachment_id = ? AND user_id = ?
		', array(XenForo_Application::$time, $attachment['attachment_id'], $userId));
	}

	public function logDownloadLimit($user)
	{
		if (is_array($user))
		{
			if (!isset($user['user_id']))
			{
				return '';
			}

			$user = $user['user_id'];
		}

		$user = intval($user);

		$dw = XenForo_DataWriter::create('Brivium_AttachmentEssential_DataWriter_Limit');
		
		$dw->set('user_id', $user);
		$dw->save();
	}

	public function updateDownloadLimit($user, $date = null)
	{
		if (is_array($user))
		{
			if (!isset($user['user_id']))
			{
				return '';
			}

			$user = $user['user_id'];
		}

		$user = intval($user);
		
		if ($date)
		{
			$this->_getDb()->query('
				UPDATE xf_attachment_essential_limit
				SET download_limit = 1, download_date = ' . $date . '
				WHERE user_id = ?
			', array($user));
		}
		else
		{
			$this->_getDb()->query('
				UPDATE xf_attachment_essential_limit
				SET download_limit = download_limit + 1
				WHERE user_id = ?
			', array($user));
		}
	}

	public function getAttachmentPermissions(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		$options = XenForo_Application::get('options');

		$time = $this->getTimeLimit() > 0 ? $this->getTimeLimit() : false;
		$download = $this->getMaxDownloadCounts() > 0 ? $this->getMaxDownloadCounts() : false;
		// Revert to default attachments option if permission is unlimited or 0
		$maxFileSize = $this->getMaxFileSize() > 0 ? $this->getMaxFileSize() * 1024 : $options->attachmentMaxFileSize * 1024;
		$maxWidth = $this->getMaxWidth() > 0 ? $this->getMaxWidth() : $options->attachmentMaxDimensions['width'];
		$maxHeight = $this->getMaxHeight() > 0 ? $this->getMaxHeight() : $options->attachmentMaxDimensions['height'];
		return array(
			'time' => $time,
			'download' => $download,
			'maxFileSize' => $maxFileSize,
			'maxWidth' => $maxWidth,
			'maxHeight' => $maxHeight
		);		
	}

	public function canViewMyAttachments(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		return XenForo_Permission::hasPermission($user['permissions'], 'AttachmentEssential', 'viewOwn');		
	}

	public function canDeleteMyAttachments(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		return XenForo_Permission::hasPermission($user['permissions'], 'AttachmentEssential', 'deleteOwn');		
	}

	public function canViewDownloadTracker(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		return XenForo_Permission::hasPermission($user['permissions'], 'AttachmentEssential', 'viewTracker');		
	}

	public function getTimeLimit(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		return XenForo_Permission::hasPermission($user['permissions'], 'AttachmentEssential', 'timeLimit');		
	}

	public function getMaxDownloadCounts(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		return XenForo_Permission::hasPermission($user['permissions'], 'AttachmentEssential', 'downloadCounts');		
	}

	public function getMaxFileSize(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		return XenForo_Permission::hasPermission($user['permissions'], 'AttachmentEssential', 'maxFileSize');		
	}

	public function getMaxWidth(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		return XenForo_Permission::hasPermission($user['permissions'], 'AttachmentEssential', 'maxWidth');		
	}

	public function getMaxHeight(array $user = null)
	{
		$this->standardizeViewingUserReference($user);

		return XenForo_Permission::hasPermission($user['permissions'], 'AttachmentEssential', 'maxHeight');		
	}
}