<?php

class Brivium_Pokes_Model_Poke extends XenForo_Model
{
	const FETCH_OWNER    			= 0x01;
	const FETCH_POKER   			= 0x02;
	const FETCH_FULL    		= 0x07;
	/**
	 * Get all pokes , in their relative display order.
	 *
	 * @return array Format: [] => poke info
	 */
	public function getAllPokes()
	{
		return $this->fetchAllKeyed('
				SELECT *
				FROM xf_brivium_poke
				ORDER BY poke_date
			', 'poke_id');
	}
	
	/**
	 * Returns poke records based on poke_id.
	 *
	 * @param string $pokeId
	 *
	 * @return array|false
	 */
	public function getPokeById($pokeId = 0)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xf_brivium_poke
			WHERE  poke_id = ?
		', array( $pokeId));
	}
	public function getPokeByIds(array $pokeIds)
	{
		if (!$pokeIds)
		{
			return array();
		}

		return $this->fetchAllKeyed('
			SELECT *
			FROM xf_brivium_poke
			WHERE poke_id IN (' . $this->_getDb()->quote($pokeIds) . ')
		', 'poke_id');
	}
	
	public function getNoResponsePoke($userId, $pokeUserId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xf_brivium_poke
			WHERE  user_id = ? AND poke_user_id = ? AND response = ?
		', array($userId, $pokeUserId, ''));
	}
	/**
	 * Prepares a collection of poke fetching related conditions into an SQL clause
	 *
	 * @param array $conditions List of conditions
	 * @param array $fetchOptions Modifiable set of fetch options (may have joins pushed on to it)
	 *
	 * @return string SQL clause (at least 1=1)
	 */
	public function preparePokeConditions(array $conditions, array &$fetchOptions)
	{
		$sqlConditions = array();
		$db = $this->_getDb();

		if (!empty($conditions['poke_id']))
		{
			if (is_array($conditions['poke_id']))
			{
				$sqlConditions[] = 'poke.poke_id IN (' . $db->quote($conditions['poke_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'poke.poke_id = ' . $db->quote($conditions['poke_id']);
			}
		}
		if (!empty($conditions['user_id']))
		{
			if (is_array($conditions['user_id']))
			{
				$sqlConditions[] = 'poke.user_id IN (' . $db->quote($conditions['user_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'poke.user_id = ' . $db->quote($conditions['user_id']);
			}
		}
		if (!empty($conditions['all_user_id']))
		{
			if (is_array($conditions['all_user_id']))
			{
				$sqlConditions[] = '(poke.poke_user_id IN (' . $db->quote($conditions['all_user_id']) . ') OR poke.user_id IN (' . $db->quote($conditions['all_user_id']) . ') )';
			}
			else
			{
				$sqlConditions[] = '(poke.poke_user_id = ' . $db->quote($conditions['all_user_id']) . ' OR ' . 'poke.user_id = ' . $db->quote($conditions['all_user_id']).' )';
			}
		}
		if (!empty($conditions['poke_user_id']))
		{
			if (is_array($conditions['poke_user_id']))
			{
				$sqlConditions[] = 'poke.poke_user_id IN (' . $db->quote($conditions['poke_user_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'poke.poke_user_id = ' . $db->quote($conditions['poke_user_id']);
			}
		}
		if (!empty($conditions['response']))
		{
			$sqlConditions[] = 'poke.response = ' . $db->quote($conditions['response']);
		}
		if (!empty($conditions['poke_date']) && is_array($conditions['poke_date']))
		{
			list($operator, $cutOff) = $conditions['poke_date'];

			$this->assertValidCutOffOperator($operator);
			$sqlConditions[] = "poke.poke_date $operator " . $db->quote($cutOff);
		}
		if (!empty($conditions['response_date']) && is_array($conditions['response_date']))
		{
			list($operator, $cutOff) = $conditions['response_date'];

			$this->assertValidCutOffOperator($operator);
			$sqlConditions[] = "poke.response_date $operator " . $db->quote($cutOff);
		}

		return $this->getConditionsForClause($sqlConditions);
	}
	public function preparePokeFetchOptions(array $fetchOptions)
	{
		$selectFields = '';
		$joinTables = '';
		$orderBy = '';
		$orderBySecondary = '';
		if (!empty($fetchOptions['order']))
		{
			switch ($fetchOptions['order'])
			{
				case 'poke_date':
					$orderBy = 'poke.' . $fetchOptions['order'];
					break;
				case 'response_date':
					$orderBy = 'poke.' . $fetchOptions['order'];
					$orderBySecondary = ', poke.poke_date DESC';
					break;
				default:
					$orderBy = 'poke.poke_date';	
			}
			if (!isset($fetchOptions['orderDirection']) || $fetchOptions['orderDirection'] == 'desc')
			{
				$orderBy .= ' DESC';
			}
			else
			{
				$orderBy .= ' ASC';
			}
			$orderBy .= $orderBySecondary;
		}

		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_OWNER)
			{
				$selectFields .= ', user.username AS username';
				$joinTables .= '
					LEFT JOIN xf_user AS user ON
						(user.user_id = poke.user_id)';
			}
			if ($fetchOptions['join'] & self::FETCH_POKER)
			{
				$selectFields .= ', user_receive.username AS receive_username';
				$joinTables .= '
					LEFT JOIN xf_user AS user_receive ON
						(user_receive.user_id = poke.poke_user_id)';
			}
			
		}
		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables,
			'orderClause'  => ($orderBy ? "ORDER BY $orderBy" : '')
		);
	}
	
	/**
	 * Gets pokes that match the given conditions.
	 *
	 * @param array $conditions Conditions to apply to the fetching
	 * @param array $fetchOptions Collection of options that relate to fetching
	 *
	 * @return array Format: [poke id] => info
	 */
	public function getPokes(array $conditions, array $fetchOptions = array())
	{
		$whereConditions = $this->preparePokeConditions($conditions, $fetchOptions);

		$sqlClauses = $this->preparePokeFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		
		return $this->fetchAllKeyed($this->limitQueryResults(			'
				SELECT poke.*
					' . $sqlClauses['selectFields'] . '
				FROM xf_brivium_poke AS poke
				' . $sqlClauses['joinTables'] . '
				WHERE ' . $whereConditions . '
				' . $sqlClauses['orderClause'] . '
			', $limitOptions['limit'], $limitOptions['offset']
		), 'poke_id');
	}
	
	/**
	 * Gets the count of pokes with the specified criteria.
	 *
	 * @param array $conditions Conditions to apply to the fetching
	 *
	 * @return integer
	 */
	public function countPokes(array $conditions)
	{
		$fetchOptions = array();
		$whereConditions = $this->preparePokeConditions($conditions, $fetchOptions);
		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM xf_brivium_poke AS poke
			WHERE ' . $whereConditions . '
		');
	}
	public function canPoke(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'BR_poke', 'poke'))
		{
			return true;
		}

		return false;
	}
	public function createAlert($alertUserId, $userId, $username, $pokeId, $action = 'poke',array $extraData = null)
	{
		XenForo_Model_Alert::alert(
			$alertUserId,
			$userId,
			$username,
			'pokes',
			$pokeId,
			$action,
			$extraData
		);
	}
	public function preparePokes(array $pokes)
	{
		if(!$pokes) return array();
		foreach($pokes AS &$poke){
			$poke = $this->preparePoke($poke);
		}
		return $pokes;
	}
	public function preparePoke(array $poke)
	{
		
		
		if(!$poke) return array();
		$userId = XenForo_Visitor::getUserId();
		
		$poke['user'] = $userId==$poke['poke_user_id']?$this->_getUserModel()->getUserById($poke['user_id']):$this->_getUserModel()->getUserById($poke['poke_user_id']);
		return $poke;
	}
	protected function _getUserModel()
	{
		return $this->getModelFromCache('XenForo_Model_User');
	}

}