<?php

/**
 * ControllerPublic class for displaying participant groups
 *
 * @package ConvEss
 */
class ConvEss_ControllerPublic_Account extends XFCP_ConvEss_ControllerPublic_Account
{
	/**
	 * Displays a list of visitor's participant groups.
	 *
	 * @return XenForo_ControllerResponse_Base
	 */
	public function actionConversationGroups()
	{
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		if (!$this->_getConversationModel()->canCreateParticipantGroups($viewingUser))
		{
			return $this->responseNoPermission();
		}
		
		$maxRecipients = XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'maxRecipients');
		$conversationModel = $this->_getConversationModel();
		$groups = $conversationModel->getParticipantGroups($viewingUser['user_id']);
		
		$firstGroupName = '';
		foreach ($groups AS &$group)
		{
			$group = $conversationModel->processParticipantGroupData($group, $maxRecipients);
			if (!$firstGroupName)
			{
				$firstGroupName = $group['group_name'];
			}
		}
		
		$numGroups = count($groups);//$conversationModel->countParticipantGroups($viewingUser['user_id']);
		$maxGroups = XenForo_Application::get('options')->convess_max_groups;
		
		$viewParams = array(
			'groups' => $groups,
			'maxRecipients' => $maxRecipients,
			'numGroups' => $numGroups,
			'maxGroups' => $maxGroups,
			'firstGroupName' => $firstGroupName
		);
		
		return $this->_getWrapper(
			'account', 'conversation-groups',
			$this->responseView('XenForo_ViewPublic_Base', 'convess_participant_groups', $viewParams)
		);	
	}
	
	/**
	 * Displays a form to add a conversation auto response message
	 *
	 * @return XenForo_ControllerResponse_Base
	 */
	public function actionConversationAutoResponse()
	{
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		if (!$this->_getConversationModel()->canAutoRespond($viewingUser))
		{
			return $this->responseNoPermission();
		}
		
		$state = ''; // not yet set				
		$conversationModel = $this->_getConversationModel();
		$autoResponse = $conversationModel->getAutoResponse($viewingUser['user_id']);
				
		// some info is available, so find out if response is active or inactive
		if (!empty($autoResponse['message']))
		{
			$autoResponse['message'] = XenForo_Helper_String::censorString($autoResponse['message']);
			$autoResponse['dates'] = unserialize($autoResponse['dates']);

			$state = 'expired';
			if ($autoResponse['active'])
			{
				$state = 'inactive'; // awaiting start time
				$dates = $autoResponse['dates'];
				
				// START DATE
				$startDate = $dates['start']['ymd'];
				$startHour = $dates['start']['hh'];
				$startMinute = $dates['start']['mm'];
				$datetimeStart = new DateTime("$startDate $startHour:$startMinute", new DateTimeZone($viewingUser['timezone']));

				// END DATE
				$endDate = $dates['end']['ymd'];
				$endHour = $dates['end']['hh'];
				$endMinute = $dates['end']['mm'];
				$datetimeEnd = new DateTime("$endDate $endHour:$endMinute", new DateTimeZone($viewingUser['timezone']));
						
				$timeNow = XenForo_Application::$time;
				if ($timeNow >= $datetimeStart->format('U')) 
				{
					$state = 'active';
				}
				if ($timeNow > $datetimeEnd->format('U')) 
				{
					$state = 'expired';
					
					// set state as inactive
					$conversationModel->autoResponseSetActiveState($viewingUser['user_id'], 0, $errorKey);
				}
			}
		}
		else
			$autoResponse = array();
		
		$hours = array();

		for ($i = 0; $i < 24; $i++)
		{
			$hh = str_pad($i, 2, '0', STR_PAD_LEFT);
			$hours[$hh] = $hh;
		}
		
		$minutes = array();
		
		for ($i = 0; $i < 60; $i += 5)
		{
			$mm = str_pad($i, 2, '0', STR_PAD_LEFT);
			$minutes[$mm] = $mm;
		}
		
		$autoResponse['hours'] = $hours;
		$autoResponse['minutes'] = $minutes;
											
		$viewParams = array(
			'autoResponse' => $autoResponse,
			'state' => $state
		);
		
		return $this->_getWrapper(
			'account', 'conversation-auto-response',
			$this->responseView('ConvEss_ViewPublic_Account_AutoResponse', 'convess_auto_response', $viewParams)
		);		
	}

	/**
	 * Save Auto Response
	 *
	 * @return XenForo_ControllerResponse_Redirect
	 */
	public function actionAutoResponseSave()
	{
		$this->_assertPostOnly();
		
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		if (!$this->_getConversationModel()->canAutoRespond($viewingUser))
		{
			return $this->responseNoPermission();
		}
		
		$autoResponse = $this->_input->filterSingle('autoResponse', XenForo_Input::ARRAY_SIMPLE);
		$dates = $autoResponse['dates'];
		
		// START DATE
		$startDate = $dates['start']['ymd'];
		$startHour = $dates['start']['hh'];
		$startMinute = $dates['start']['mm'];
			
		// END DATE
		$endDate = $dates['end']['ymd'];
		$endHour = $dates['end']['hh'];
		$endMinute = $dates['end']['mm'];
		
		// MESSAGE
		
		$message = $this->getHelper('Editor')->getMessageText('autoResponseMessage', $this->_input);
		
		$message = XenForo_Helper_String::autoLinkBbCode($message);
		$message = XenForo_Helper_String::censorString($message);
		$strippedMessage = XenForo_Helper_String::bbCodeStrip($message);
			
		// date fields are empty - assume user is trying to reset the form
		if (!$startDate && !$endDate)
		{
			if (!$strippedMessage) // message box is also empty
			{
				$dates['start']['hh'] = '00';
				$dates['start']['mm'] = '00';
				$dates['end']['hh'] = '00';
				$dates['end']['mm'] = '00';
				
				$message = ''; // ensure no empty BB Code tags were left over
			}
			else
			{
				return $this->responseError(new XenForo_Phrase('convess_auto_response_dates_must_be_set_when_message'));
			}
		}
		else
		{
			if (!$strippedMessage) // ensure message has content
				return $this->responseError(new XenForo_Phrase('convess_auto_response_message_cannot_be_empty'));
			
			// ensure start date is in the futute
			$datetimeStart = new DateTime("$startDate $startHour:$startMinute", new DateTimeZone($viewingUser['timezone']));
			if (XenForo_Application::$time >= $datetimeStart->format('U'))
			{
				return $this->responseError(new XenForo_Phrase('convess_auto_response_start_date_must_be_in_future'));
			}
			// ensure end date is after start date
			$datetimeEnd = new DateTime("$endDate $endHour:$endMinute", new DateTimeZone($viewingUser['timezone']));
			if ($datetimeEnd->format('U') <= $datetimeStart->format('U'))
			{
				return $this->responseError(new XenForo_Phrase('convess_auto_response_end_date_after_start_date'));
			}		
		}
		
		// SAVE
		$errorKey = '';
		
		$conversationModel = $this->_getConversationModel();
		$exclude = isset($autoResponse['exclude']) ? 1 : 0;
		
		// use $message and not $strippedMessage which was just for checking the contents
		$conversationModel->autoResponseSave($viewingUser['user_id'], serialize($dates), $message, $exclude, $errorKey);
		
		if ($errorKey)
		{
			$errorKey = $errorKey instanceof XenForo_Phrase ? $errorKey : new XenForo_Phrase($errorKey);
			return $this->responseError(new XenForo_Phrase($errorKey));
		}
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('account/conversation-auto-response')
		);
	}
	
	/**
	 * Shows a preview of the Auto Response message.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionAutoResponsePreview()
	{
		$this->_assertPostOnly();
		
		$autoResponseMessage = $this->getHelper('Editor')->getMessageText('autoResponseMessage', $this->_input);
		
		$autoResponseMessage = XenForo_Helper_String::autoLinkBbCode($autoResponseMessage);
		$autoResponseMessage = XenForo_Helper_String::censorString($autoResponseMessage);
		
		$viewParams = array(
			'autoResponseMessage' => $autoResponseMessage
		);
		
		return $this->responseView('ConvEss_ViewPublic_Account_AutoResponsePreview', 'convess_auto_response_preview', $viewParams);
	}
	
	/**
	 * Displays a list of prefixes.
	 *
	 * @return XenForo_ControllerResponse_Base
	 */
	public function actionConversationPrefixes()
	{
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		if (!$this->_getConversationModel()->canCreatePrefixes($viewingUser))
		{
			return $this->responseNoPermission();
		}
		
		$maxPrefixes = XenForo_Application::get('options')->convess_max_prefixes;
		$conversationModel = $this->_getConversationModel();
		$prefixes = $conversationModel->getPrefixes($viewingUser['user_id']);
		
		$viewParams = array(
			'prefixes' => $prefixes,
			'maxPrefixes' => $maxPrefixes,
		);
		
		return $this->_getWrapper(
			'account', 'conversation-prefixes',
			$this->responseView('XenForo_ViewPublic_Base', 'convess_prefixes', $viewParams)
		);	
	}
	
	protected function _getPrefixAddEdit(array $prefix)
	{
		$phrase = new XenForo_Phrase('convess_color_red');
		$red = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_orange');
		$orange = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_yellow');
		$yellow = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_light_green');
		$lightGreen = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_green');
		$green = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_olive');
		$olive = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_primary');
		$primary = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_secondary');
		$secondary = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_blue');
		$blue = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_royal_blue');
		$royalBlue = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_sky_blue');
		$skyBlue = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_gray');
		$gray = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_silver');
		$silver = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_ocean');
		$ocean = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_rust');
		$rust = $phrase->render();
		
		$phrase = new XenForo_Phrase('convess_color_brown');
		$brown = $phrase->render();
		
		$displayStyles = array(
			$red => 'prefix prefixRed',
			$orange => 'prefix prefixOrange',
			$yellow => 'prefix prefixYellow',
			$lightGreen => 'prefix prefixLightGreen',
			$green => 'prefix prefixGreen',
			$olive => 'prefix prefixOlive',			
			$primary => 'prefix prefixPrimary',
			$secondary => 'prefix prefixSecondary',
			$blue => 'prefix prefixBlue',
			$royalBlue => 'prefix prefixRoyalBlue',
			$skyBlue => 'prefix prefixSkyBlue',
			$gray => 'prefix prefixGray',
			$silver => 'prefix prefixSilver',
			$ocean => 'prefix prefixOcean',
			$rust => 'prefix prefixRust',
			$brown => 'prefix prefixBrown'
		);
		
		$viewParams = array(
			'prefix' => $prefix,
			'displayStyles' => $displayStyles
		);
		
		return $this->_getWrapper(
			'account', 'conversation-prefixes',
			$this->responseView('XenForo_ViewPublic_Base', 'convess_prefix_edit', $viewParams)
		);
	}
	
	public function actionConversationAddPrefix()
	{
		$numPrefixes = $this->_getConversationModel()->getNumPrefixes(XenForo_Visitor::getUserId());
		$maxPrefixes = XenForo_Application::get('options')->convess_max_prefixes;
		if ($numPrefixes >= $maxPrefixes)
		{
			return $this->_getWrapper(
				'account', 'conversation-prefixes',
				$this->responseView('XenForo_ViewPublic_Base', 'convess_prefixes_full', array('maxPrefixes' => $maxPrefixes))
			);
		}
		
		return $this->_getPrefixAddEdit(array('display_order' => 1,	'css_class' => 'prefix prefixRed'));
	}
	
	public function actionConversationEditPrefix()
	{
		$prefixId = $this->_input->filterSingle('prefix_id', XenForo_Input::UINT);
		$conversationModel = $this->_getConversationModel();
		$prefix = $conversationModel->getPrefixById($prefixId);
		
		return $this->_getPrefixAddEdit($prefix);
	}
	
	public function actionConversationAddOrUpdatePrefix()
	{
		$this->_assertPostOnly();
		
		$input = $this->_input->filter(array(
			'prefix_id' => XenForo_Input::UINT,
			'title' => XenForo_Input::STRING,
			'display_order' => XenForo_Input::UINT,
			'css_class' => XenForo_Input::STRING
		));
		
		if (empty($input['title']))
			return $this->responseError(new XenForo_Phrase('convess_prefix_must_have_title'));
		
		$conversationModel = $this->_getConversationModel();
		
		if ($input['prefix_id']) // update
		{
			$conversationModel->updatePrefix($input['prefix_id'], $input['title'], $input['display_order'], $input['css_class']);
		}
		else // insert
		{
			$conversationModel->addPrefix($input['title'], $input['display_order'], $input['css_class']);
		}
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('account/conversation-prefixes')
		);
	}
	
	/**
	 * Deletes the specified prefix
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionConversationDeletePrefix()
	{
		$conversationModel = $this->_getConversationModel();
		$prefixId = $this->_input->filterSingle('prefix_id', XenForo_Input::UINT);
		
		if ($this->isConfirmedPost())
		{
			$conversationModel->deletePrefixById($prefixId);
			
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('account/conversation-prefixes')
			);
		}
		else // show confirmation dialog
		{
			$prefix = $conversationModel->getPrefixById($prefixId);
			
			$viewParams = array(
				'prefix' => $prefix
			);
			return $this->responseView('XenForo_ViewPublic_Base', 'convess_prefix_delete', $viewParams);
		}
	}
	
	/**
	 * Set order of all prefixes
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionConversationSetPrefixOrder()
	{
		$displayOrders = $this->_input->filterSingle('display_order', array(XenForo_Input::UINT, 'array' => true));
		$conversationModel = $this->_getConversationModel();
		
		foreach ($displayOrders AS $prefixId => $displayOrder)
		{
			$conversationModel->setPrefixOrder($prefixId, $displayOrder);
		}
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('account/conversation-prefixes'),
			new XenForo_Phrase('convess_prefix_order_updated')
		);
	}
	
	public function actionPrivacySave()
	{
		$options = XenForo_Application::get('options');
		if ($options->convess_last_read_date && $options->convess_last_read_date_privacy)
		{
			$visitor = XenForo_Visitor::getInstance();
			$lastReadLimit = $options->convess_last_read_date_view_limit;
			
			if ($lastReadLimit == 'no_limit' || ($lastReadLimit == 'admins' && $visitor['is_admin']) || ($lastReadLimit == 'admins_mods' && ($visitor['is_admin'] || $visitor['is_moderator'])))
			{
				$this->_assertPostOnly();
				$privacy = $this->_input->filterSingle('allow_last_read_date', XenForo_Input::UINT);
				$this->_getConversationModel()->setLastReadDatePrivacy($visitor['user_id'], $privacy);
			}
		}
		
		return parent::actionPrivacySave();
	}
			
	/**
	 * @return XenForo_Model_Conversation
	 */
	protected function _getConversationModel()
	{
		return $this->getModelFromCache('XenForo_Model_Conversation');
	}
}