<?php
/**
 * Controller for conversation actions.
 *
 * @package ConvEss
 */
class ConvEss_ControllerPublic_Conversation extends XFCP_ConvEss_ControllerPublic_Conversation
{
	protected function _getConversationListData(array $extraConditions = array())
	{
		$response = parent::_getConversationListData($extraConditions);
		
		if (!empty($response))
		{
			$viewingUser = XenForo_Visitor::getInstance()->toArray();
			$conversationModel = $this->_getConversationModel();
			$options = XenForo_Application::get('options');
			
			if ($options->convess_username_view == 'modified' || $options->convess_last_post_avatar)
			{
				$conversations = $response['conversations'];
				
				if (!empty($conversations))
				{
					$useAvatars = 'no';
					
					if ($options->convess_username_view == 'modified')
					{
						$useAvatars = $this->_input->filterSingle('avatars', XenForo_Input::STRING);
						if (!$useAvatars)
						{
							$convessUserOptions = $conversationModel->getConvEssUserOptionsByUserId($viewingUser['user_id']);
							if (!empty($convessUserOptions)) // preference
							{
								$useAvatars = $convessUserOptions['list_view_avatars'] ? 'yes' : 'no';
							}
							else // default
							{
								$useAvatars = $options->convess_view == 'avatars' ? 'yes' : 'no';
							}
						}
						
						// let links know if to continue to display as usernames or avatars
						$response['avatars'] = $useAvatars;
						$response['pageNavParams']['avatars'] = $useAvatars;
					}
					
					// fetch recipient list keyd by conversation_id
					$recipientArray = $conversationModel->getConversationRecipientsByConversationIds(array_keys($conversations));
					
					// add the participants
					foreach ($conversations AS &$conversation)
					{
						if (!isset($recipientArray[$conversation['conversation_id']])) // should never happen
							continue;
						
						$recipients = $recipientArray[$conversation['conversation_id']];
						if (empty($recipients)) // should never happen
							continue;
						
						// last poster's info
						if (isset($recipients[$conversation['last_message_user_id']]))
							$conversation['lastUser'] = $recipients[$conversation['last_message_user_id']];
						else
							$conversation['lastUser'] = array('user_id' => $conversation['last_message_user_id'], 'username' => $conversation['last_message_username']);
						
						// remove deleted users with a user_id of zero - this was due to a bug in XenForo
						foreach ($recipients AS $key => $recipient)
						{
							if (!isset($recipient['user_id']))
								unset($recipients[$key]);
						}
						
						// the default saved names which does not include conversation starter
						$recipientNames = $conversation['recipientNames'];
						
						// add conversation starter name if user was deleted
						if (!$conversation['user_id'])
							$recipientNames[$conversation['user_id']] = array('user_id' => 0, 'username' => $conversation['username']);
						
						// find deleted users still in the default name array (already deleted from the recipient table) - this was due to a bug in XenForo
						$diff = array_diff_key($recipientNames, $recipients);
						if (!empty($diff))
						{
							foreach ($diff AS $id => $participant)
							{
								$participant['recipient_state'] = 'deleted_ignored';
								$recipients[$id] = $participant;
							}
						}
						
						// remove the visitor as he surely knows he is part of the conversation :-)
						unset($recipients[$viewingUser['user_id']]);
						
						// add recipients to the conversation
						// saved as participants in order not to overwrite the recipients variable (string of names)
						$conversation['participants'] = $recipients;
						
						$conversation['avatars'] = $useAvatars == 'yes' ? 1 : 0;
						
						// pagination page number we are on
						$conversation['page'] = $response['page'];
					}
					
					$response['conversations'] = $conversations;
				}
			}
			
			// prefixes
			if ($conversationModel->canCreatePrefixes($viewingUser))
			{
				$response['prefixes'] = $this->_getAllPrefixes($viewingUser['user_id'], $conversationModel);
				$selectedPrefix = $this->_input->filterSingle('prefix_id', XenForo_Input::UINT);
				if (!$selectedPrefix)
					$selectedPrefix = $this->_input->filterSingle('prefix_id_search', XenForo_Input::UINT);
				$response['selectedPrefix'] = $selectedPrefix;
				$response['pageNavParams']['prefix_id'] = $selectedPrefix;
			}
			
			// inbox size
			$inboxMaxSize = XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'convessInboxSize');
			if ($inboxMaxSize > 0) // size restriction
			{
				if (!empty($extraConditions) || !empty($response['search_type']) || !empty($response['search_user']) || (isset($response['selectedPrefix']) && $response['selectedPrefix'] > 0))
					$inboxTotalUsed = $conversationModel->countConversationsForUser($viewingUser['user_id']);
				else
					$inboxTotalUsed = $response['totalConversations']; // saves a query
								
				$response['inboxTotalUsed'] = $inboxTotalUsed;
				$response['inboxMaxSize'] = $inboxMaxSize;
				$response['inboxPercent'] = ceil(($inboxTotalUsed / $inboxMaxSize) * 100);
			}
			
			// let the inline_mod_control know to add the large 'Leave Conversations...' button
			$response['convess'] = true;
		}
		
		return $response;
	}
	
	/**
	 * Save the list view preference (avatars/text).
	 *
	 * @return XenForo_ControllerResponse_Redirect
	 */
	public function actionToggleView()
	{
		$conversationId = $this->_input->filterSingle('conversation_id', XenForo_Input::UINT);
		$conversation = $this->_getConversationOrError($conversationId);
		$page = $this->_input->filterSingle('page', XenForo_Input::UINT);
		
		$useAvatars = $this->_input->filterSingle('avatars', XenForo_Input::STRING);
		if ($useAvatars)
		{
			$requestedView = ($useAvatars == 'yes' ? 1 : 0);
			$userId = XenForo_Visitor::getUserId();
			$convessUserOptions = $this->_getConversationModel()->getConvEssUserOptionsByUserId($userId);
			if (!empty($convessUserOptions))
			{
				$currentView = $convessUserOptions['list_view_avatars'];
			}
			else
			{
				$currentView = (XenForo_Application::get('options')->convess_view == 'avatars' ? 1 : 0);
			}
				
			// save new preference
			if ($requestedView != $currentView)
			{
				$dw = XenForo_DataWriter::create('ConvEss_DataWriter_UserOptions');
				
				if (!empty($convessUserOptions))
				{
					$dw->setExistingData($userId);
				}
				else
				{
					$dw->set('user_id', $userId);
				}
				
				$dw->set('list_view_avatars', $requestedView);
				$dw->preSave();
		
				if ($dw->hasErrors())
				{
					$errors = $dw->getErrors();
					$errorKey = reset($errors);
					throw new XenForo_Exception($errorKey);
				}
		
				$dw->save();
				
				$useAvatars = ($requestedView == 1 ? 'yes' : 'no');
			}
		}
		
		$prefixId = $this->_input->filterSingle('prefix_id', XenForo_Input::UINT);
		$action = $this->_input->filterSingle('action', XenForo_Input::STRING);
		if ($action == 'Yours' || $action == 'Starred')
			$action = '/' . $action;
		else
			$action = '';
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('conversations' . $action, '', array('avatars' => $useAvatars, 'prefix_id' => $prefixId, 'page' => $page)) . '#conversation-' . $conversationId
		);
	}
	
	protected function _getListConditions()
	{
		$conditions = parent::_getListConditions();
		
		$prefixId = $this->_input->filterSingle('prefix_id', XenForo_Input::UINT);
		if (!$prefixId)
			$prefixId = $this->_input->filterSingle('prefix_id_search', XenForo_Input::UINT);
		
		if ($prefixId)
		{
			$conditions['prefix_id'] = $prefixId;
		}
		
		return $conditions;
	}
	
	/**
	 * Kick a user from a conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */	
	public function actionKickRecipient()
	{
		$conversationModel = $this->_getConversationModel();
		if (!$conversationModel->canKickRecipients())
			return $this->responseNoPermission();
		
		$input = $this->getInput();
		$conversationId = $input->filterSingle('conversation_id', XenForo_Input::UINT);
		$userId = $input->filterSingle('user_id', XenForo_Input::UINT);
		$page = $input->filterSingle('page', XenForo_Input::UINT);
		$conversation = $this->_getConversationOrError($conversationId);
		
		// not conversation author
		if ($conversation['user_id'] != XenForo_Visitor::getUserId())
			return $this->responseNoPermission();
		
		if ($this->isConfirmedPost())
		{
			// register this user as kicked so we may allow for a future invite
			$conversationModel->addKicked($conversationId, $userId);
			
			// kick
			$conversationModel->deleteConversationForUser(
				$conversationId, $userId, 'delete_ignore'
			);
			
			// reset last message read in case they are re-invited again at some point
			$conversationModel->markConversationAsUnreadForKickedUser($conversationId, $userId);
			
			// send alert
			if (XenForo_Application::get('options')->convess_alert_upon_kick)
			{
				// reason
				$reason = $input->filterSingle('user_reason', XenForo_Input::STRING);
				$reason = trim(htmlspecialchars_decode(strip_tags($reason)));
				$reason = XenForo_Helper_String::censorString($reason);
				
				$visitor = XenForo_Visitor::getInstance();
								
				XenForo_Model_Alert::alert(
							$userId,
							$visitor['user_id'],
							$visitor['username'],
							'convess',
							0,
							'kick',
							array('conversation_title' => $conversation['title'], 'user_reason' => $reason)
				);
			}
							
			if ($this->_noRedirect())
			{
				$recipients = $conversationModel->getConversationRecipients($conversationId);
				$conversationModel->processLastReadDate($conversationId, $recipients); // by reference
				
				$viewParams = array(
					'conversation' => $conversation,
					'recipients' => $recipients,
					'canKickRecipients' => $conversationModel->canKickRecipients(),
					'page' => $page
				);
				
				// just reload the participants template
				return $this->responseView(
					'XenForo_ViewPublic_Conversation_InviteInsert',
					'conversation_recipients',
					$viewParams
				);
			}
			
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('conversations', $conversation, array('page' => $page))
			);
		}
		else
		{
			$user = $this->_getUserModel()->getUserById($userId);
			if (empty($user))
				throw $this->getErrorOrNoPermissionResponseException('requested_member_not_found');
			
			// cannot kick admins or moderators
			$visitor = XenForo_Visitor::getInstance();
			if ($user['is_admin'] || ($user['is_moderator'] && !$visitor['is_admin']))
				throw $this->getErrorOrNoPermissionResponseException('convess_cannot_kick_staff');
			
			$user = array('user_id' => $user['user_id'], 'username' => $user['username']);
			
			// if no one is left in the conversation then 'lock' it by telling the form to redirect
			$forceRedirect = false;
			$numRecipients = $conversation['recipient_count'] - 1; // -1 for author
			$numLeftPermanently = $conversationModel->countNumLeftPermanently($conversationId) + 1; // +1 for current user being kicked
			if ($numRecipients == $numLeftPermanently)
				$forceRedirect = true; // no one left
			
			$viewParams = array(
				'conversationId' => $conversationId,
				'user' => $user,
				'page' => $page,
				'forceRedirect' => $forceRedirect
			);
			return $this->responseView('XenForo_ViewPublic_Base','convess_kick_recipient', $viewParams);
		}
	}
	
	/**
	 * Invite a kicked user back to the conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */	
	public function actionInviteRecipient()
	{
		$conversationModel = $this->_getConversationModel();
		if (!$conversationModel->canKickRecipients())
			return $this->responseNoPermission();
		
		$input = $this->getInput();
		$conversationId = $input->filterSingle('conversation_id', XenForo_Input::UINT);
		$userId = $input->filterSingle('user_id', XenForo_Input::UINT);
		$page = $input->filterSingle('page', XenForo_Input::UINT);
		$conversation = $this->_getConversationOrError($conversationId);
		
		// not conversation author
		if ($conversation['user_id'] != XenForo_Visitor::getUserId())
			return $this->responseNoPermission();
		
		if ($this->isConfirmedPost())
		{
			// since the form should be posted to actionInviteInsert(), this should not happen
			return $this->responseNoPermission();
		}
		else
		{
			$user = $this->_getUserModel()->getUserById($userId);
			if (empty($user))
				throw $this->getErrorOrNoPermissionResponseException('requested_member_not_found');
			
			$user = array('user_id' => $user['user_id'], 'username' => $user['username']);
			
			// 'unlock' the conversation by telling the form to redirect when all have previously left
			$forceRedirect = false;
			$numRecipients = $conversation['recipient_count'] - 1; // -1 for author
			$numLeftPermanently = $conversationModel->countNumLeftPermanently($conversationId);
			if ($numRecipients == $numLeftPermanently) // currently all have left
				$forceRedirect = true;
			
			$viewParams = array(
				'conversationId' => $conversationId,
				'user' => $user,
				'page' => $page,
				'forceRedirect' => $forceRedirect
			);
			return $this->responseView('XenForo_ViewPublic_Base','convess_invite_recipient', $viewParams);
		}		
	}
	
	/**
	 * Displays a conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionView()
	{
		$response = parent::actionView();
		
		if (!empty($response->params))
		{
			$conversationId = $response->params['conversation']['conversation_id'];
			$recipients = $response->params['recipients'];
			$conversationModel = $this->_getConversationModel();
			$numLeftPermanently = $conversationModel->processLastReadDate($conversationId, $recipients); // by reference
			$response->params['recipients'] = $recipients;
			
			// all participants (other than viewer) have left permanently so 'lock' it
			if (count($recipients) == ($numLeftPermanently + 1)) // +1 for viewing user
			{
				$response->params['canReplyConversation'] = false; // prevent replying
				$response->params['noParticipantsLeft'] = true; // for conversation status
			}
			
			// can kick recipients?
			$response->params['canKickRecipients'] = $conversationModel->canKickRecipients();
		}
		
		return $response;
	}
	
	/**
	 * Displays a form to invite users to a conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionInvite()
	{
		$response = parent::actionInvite();
		
		if (!empty($response->params))
		{
			$conversationModel = $this->_getConversationModel();
			
			if (XenForo_Application::get('options')->convess_invite_groups)
			{
				$viewingUser = XenForo_Visitor::getInstance()->toArray();
				$canCreateGroups = $conversationModel->canCreateParticipantGroups($viewingUser);
				if ($canCreateGroups)
				{
					$response->params['canCreateParticipantGroups'] = true;
					
					// fetch participant groups
					$participantGroups = $conversationModel->getParticipantGroups($viewingUser['user_id']);
					if (!empty($participantGroups))
					{
						foreach ($participantGroups AS $group)
						{
							$processedGroup = $conversationModel->processParticipantGroupData($group);
							$groups[$group['group_name']] = $processedGroup['group_users'] . ', ';
						}
						
						$response->params['participantGroups'] = $groups;
					}
				}
			}
			
			$conversationId = $response->params['conversation']['conversation_id'];
			if ($conversationModel->canKickRecipients())
				$response->params['kicked'] = $conversationModel->countKickedFromConversation($conversationId);
		}
		
		return $response;
	}
	
	/**
	 * Invites users to a conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionInviteInsert()
	{
		$this->_assertPostOnly();
				
		$conversationModel = $this->_getConversationModel();
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		
		// exempt admins and moderators from recipient inbox restrictions
		if (!$viewingUser['is_admin'] && !$viewingUser['is_moderator'])
		{
			$recipients = $this->_input->filterSingle('recipients', XenForo_Input::STRING);
			$recipients = explode(',', $recipients);
			
			// is receiver's inbox full?
			$this->_checkRecipientsHaveInboxSpace($recipients, true); // true for invite
		}
		
		$canKickRecipients = $conversationModel->canKickRecipients();
		
		// custom errors only if there are kicked users involved
		if ($canKickRecipients)
		{
			// if max available additional recipients is the same as kicked users, then ensure the recipients
			// provided are same as the kicked (re-invite), otherwise deny permission
			$conversationId = $this->_input->filterSingle('conversation_id', XenForo_Input::UINT);
			$conversation = $this->_getConversationOrError($conversationId);
			$recipients = $this->_input->filterSingle('recipients', XenForo_Input::STRING);
			$kicked = $conversationModel->getKickedByConversationId($conversationId);
			
			if ($recipients && !empty($kicked))
			{
				$recipients = $this->_getUserModel()->getUsersByNames(explode(',', $recipients));
				$diff = array_diff_key($recipients, $kicked);
				$kickedCount = count($kicked);
				$insertCount = count($diff); // new inserts, not re-invites
				$remaining = $conversationModel->allowedAdditionalConversationRecipients($conversation);
							
				if ($remaining > -1) // otherwise unlimited
				{
					if ($insertCount > max(0, $remaining - $kickedCount))
					{
						if ($remaining > 0 && !$kickedCount)
						{
							throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('you_may_invite_up_to_x_members', array('number' =>  $remaining)));
						}
						else
						{
							if ($remaining && $kickedCount)
							{
								if ($remaining == $kickedCount)
								{
									throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_you_may_invite_up_to_x_kicked_members_only', array('kicked' =>  $kickedCount)));
								}
								else
								{
									if ($remaining > $kickedCount)
									{
										throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_you_may_invite_up_to_x_new_and_y_kicked_members', array('new' =>  $remaining - $kickedCount, 'kicked' => $kickedCount)));
									}
									else
									{
										throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_you_may_invite_up_to_x_new_and_y_kicked_members', array('new' =>  $kickedCount - $remaining, 'kicked' => $kickedCount)));
									}
								}
							}
						}
					}
				}
			}
		}
		
		if ($this->_noRedirect()) // fetch last read dates for participants block
		{
			$response = parent::actionInviteInsert();
			
			if (!empty($response->params))
			{
				$recipients = $response->params['recipients'];
				$conversationId = $response->params['conversation']['conversation_id'];
				
				// reloading the participant block, so fetch last read dates
				$conversationModel->processLastReadDate($conversationId, $recipients);
				$response->params['recipients'] = $recipients;
				$response->params['canKickRecipients'] = $canKickRecipients;
			}
			
			return $response;
		}
		else
		{
			return parent::actionInviteInsert();
		}
	}
	
	/**
	 * Displays a form to edit a conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionEdit()
	{
		$response = parent::actionEdit();
		
		if (!empty($response->params))
		{
			$viewingUser = XenForo_Visitor::getInstance()->toArray();
			
			// sticky
			if (XenForo_Application::get('options')->convess_sticky)
			{
				$conversation = &$response->params['conversation'];
				$conversationModel = $this->_getConversationModel();
				
				if ($conversationModel->isSticky($conversation['conversation_id'], $viewingUser['user_id']))
					$conversation['sticky'] = 1;
			}
			
			// prefixes
			$conversationModel = $this->_getConversationModel();
			if ($conversationModel->canCreatePrefixes($viewingUser))
			{
				$response->params['prefixes'] = $this->_getAllPrefixes($viewingUser['user_id'], $conversationModel);
				$response->params['selectedPrefix'] = $conversationModel->getPrefixByConversationId($conversation['conversation_id'], $viewingUser['user_id']);
			}
		}
		
		return $response;
	}
	
	/**
	 * Updates a conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionUpdate()
	{
		$this->_assertPostOnly();
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		$conversationId = $this->_input->filterSingle('conversation_id', XenForo_Input::UINT);
		$conversationModel = $this->_getConversationModel();
		$doUpdate = false;
		
		// sticky
		$sticky = 0;
		if (XenForo_Application::get('options')->convess_sticky)
		{
			$sticky = $this->_input->filterSingle('conversation_sticky', XenForo_Input::UINT);
			$doUpdate = true;
		}
		
		// prefix
		$prefixId = 0;
		if ($conversationModel->canCreatePrefixes($viewingUser))
		{
			$prefixId = $this->_input->filterSingle('prefix_id', XenForo_Input::UINT);
			$doUpdate = true;
		}
		
		if ($doUpdate)
			$conversationModel->setConversationStickyAndOrPrefix($conversationId, $viewingUser['user_id'], $sticky, $prefixId);
		
		return parent::actionUpdate();
	}
	
	/**
	 * Displays a form to create a conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionAdd()
	{
		$conversationModel = $this->_getConversationModel();
		if (!$conversationModel->canStartConversations($errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
		
		// is sender's inbox full?
		$this->_checkSenderHasInboxSpace($conversationModel);
		
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		// exempt admins and moderators from recipient inbox restrictions
		if (!$viewingUser['is_admin'] && !$viewingUser['is_moderator'])
		{
			$recipient = $this->_input->filterSingle('to', XenForo_Input::STRING);
			if ($recipient !== '' && strpos($recipient, ',') === false && $recipient != $viewingUser['username'])
			{			
				// is receiver's inbox full?
				$this->_checkRecipientHasInboxSpace($conversationModel, $recipient);
			}
		}
		
		$canCreateGroups = $conversationModel->canCreateParticipantGroups($viewingUser);
		$canCreatePrefixes = $conversationModel->canCreatePrefixes($viewingUser);
		if ($canCreateGroups || $canCreatePrefixes)
		{
			$response = parent::actionAdd();
			if (!empty($response->params))
			{
				if ($canCreatePrefixes)
				{
					$response->params['prefixes'] = $this->_getAllPrefixes($viewingUser['user_id'], $conversationModel);
				}
				
				if ($canCreateGroups)
				{
					$response->params['canCreateParticipantGroups'] = true;
					
					// fetch participant groups
					$participantGroups = $conversationModel->getParticipantGroups($viewingUser['user_id']);
					if (!empty($participantGroups))
					{
						foreach ($participantGroups AS $group)
						{
							$processedGroup = $conversationModel->processParticipantGroupData($group);
							$groups[$group['group_name']] = $processedGroup['group_users'] . ', ';
						}
						
						$response->params['participantGroups'] = $groups;
					}
				}
			}
			
			return $response;
		}
		else
		{
			return parent::actionAdd();
		}
	}
	
	/**
	 * Inserts a new conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionInsert()
	{
		$this->_assertPostOnly();
		$conversationModel = $this->_getConversationModel();
		
		if (!$conversationModel->canStartConversations($errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
		
		// is sender's inbox full?
		$this->_checkSenderHasInboxSpace($conversationModel);
		
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		// exempt admins and moderators from recipient inbox restrictions
		if (!$viewingUser['is_admin'] && !$viewingUser['is_moderator'])
		{
			$recipients = $this->_input->filterSingle('recipients', XenForo_Input::STRING);
			$recipients = explode(',', $recipients);
			
			// is receiver's inbox full?
			$this->_checkRecipientsHaveInboxSpace($recipients);
		}
		
		// prefix
		$prefixId = $this->_input->filterSingle('prefix_id', XenForo_Input::UINT);
		
		// sticky
		$sticky = $this->_input->filterSingle('conversation_sticky', XenForo_Input::UINT);
		
		if ($prefixId || $sticky)
		{
			$response = parent::actionInsert();
			$target = $response->redirectTarget;
			
			// find the conversation_id
			$includeTitleInUrls = XenForo_Application::get('options')->includeTitleInUrls;
			if ($includeTitleInUrls) // conversations/eee.12/
			{
				$delimiter = XenForo_Application::URL_ID_DELIMITER;
				$startPos = strrpos($target, $delimiter);
			}
			else // conversations/12/
			{
				$startPos = strrpos($target, '/', -2); // before last '/'
			}
			
			if ($startPos !== false)
			{
				$conversationId = intval(substr($target, $startPos +1, -1)); // -1 is for the last '/'
				
				if ($conversationId)
				{			
					// sticky
					if (!XenForo_Application::get('options')->convess_sticky)
						$sticky = null;
					
					// prefix
					if (!$conversationModel->canCreatePrefixes($viewingUser))
						$prefixId = null;
					
					if ($prefixId || $sticky)
						$conversationModel->setConversationStickyAndOrPrefix($conversationId, $viewingUser['user_id'], $sticky, $prefixId);
				}
			}
			
			return $response;
		}
		
		return parent::actionInsert();
	}
	
	/**
	 * Inserts a reply into a conversation.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionInsertReply()
	{
		$this->_assertPostOnly();

		if ($this->_input->inRequest('more_options'))
		{
			return $this->responseReroute('XenForo_ControllerPublic_Conversation', 'reply');
		}
		
		$conversationId = $this->_input->filterSingle('conversation_id', XenForo_Input::UINT);
		$conversation = $this->_getConversationOrError($conversationId);
		$actionUser = XenForo_Visitor::getInstance();
		$autoResponseUsers = array(); // array of users that have an active auto response
		
		// this is all done here rather than in the DataWriter as we cannot add a message while inside the writer's _postSave
		if ($conversation['conversation_open'])
		{
			$conversationModel = $this->_getConversationModel();
			$fetchOptions = array('join_user_permissions' => 1, 'ignoringUserId' => $actionUser['user_id']);
			$exclude = XenForo_Application::get('options')->convess_auto_response_exclude; // exclude people the user follows
			if ($exclude)
			{
				$fetchOptions['followingUserId'] = $actionUser['user_id'];
			}
			
			$recipients = $conversationModel->getConversationRecipients($conversationId, $fetchOptions);
			
			if (!empty($recipients))
			{
				$userModel = $this->getModelFromCache('XenForo_Model_User');
										
				foreach ($recipients AS $recipient)
				{
					if ($recipient['recipient_state'] != 'deleted_ignored' && $recipient['user_id'] != $actionUser['user_id'])
					{
						$autoResponse = $conversationModel->getAutoResponse($recipient['user_id'], 1); // 1 for active state
						if (!empty($autoResponse))
						{
							$recipient['permissions'] = XenForo_Permission::unserializePermissions($recipient['global_permission_cache']);
							if (!$conversationModel->canAutoRespond($recipient))
							{
								// has an auto response but no longer has permission
								$conversationModel->autoResponseDelete($recipient['user_id'], true); // true for existance of auto response established
								continue;
							}
							
							$dates = unserialize($autoResponse['dates']);
							
							// start date/time
							$startDate = $dates['start']['ymd'];
							$startHour = $dates['start']['hh'];
							$startMinute = $dates['start']['mm'];
							$datetimeStart = new DateTime("$startDate $startHour:$startMinute", new DateTimeZone($recipient['timezone']));
							
							// end date/time
							$endDate = $dates['end']['ymd'];
							$endHour = $dates['end']['hh'];
							$endMinute = $dates['end']['mm'];
							$datetimeEnd = new DateTime("$endDate $endHour:$endMinute", new DateTimeZone($recipient['timezone']));
							
							$timeNow = XenForo_Application::$time;
							if ($timeNow >= $datetimeStart->format('U'))
							{
								if ($timeNow < $datetimeEnd->format('U')) // active
								{
									if (!isset($recipient['ignoring_' . $actionUser['user_id']]) || ($recipient['ignoring_' . $actionUser['user_id']] < 1)) // recipient not ignoring sender
									{
										if (!$exclude || !$autoResponse['exclude'] || !isset($recipient['following_' . $actionUser['user_id']]) || ($recipient['following_' . $actionUser['user_id']] < 1)) // exclude responding to those user follows
										{
											$autoResponded = $conversationModel->hasAutoResponded($conversationId, $recipient['user_id']);
											if (!$autoResponded) // current auto response has not yet left a message in this conversation
											{
												$recipient['autoResponseMessage'] = $autoResponse['message'];
												$autoResponseUsers[] = $recipient;
											}
										}
									}
								}
								else // expired
								{
									// set state as inactive
									$conversationModel->autoResponseSetActiveState($recipient['user_id'], 0, $errorKey);
								}
							}
						}
					}
				}
			}
		}
		
		// send out the auto responses
		// we collected the users in an array so that we only set '_xfNoRedirect' when we have to
		if (!empty($autoResponseUsers))
		{
			$this->_request->setParam('_xfNoRedirect', false); // ensure we reload the page in order to display the new responses
			$response = parent::actionInsertReply(); // insert the new message prior to responses
			
			foreach ($autoResponseUsers AS $responder)
			{
				$conversationModel->sendAutoResponse($conversationId, $responder, $responder['autoResponseMessage']);
			}
			
			// mark as read since the auto response triggers the unread alert
			$conversationModel->markConversationAsRead($conversationId, $actionUser['user_id'], XenForo_Application::$time, 0, false);
			
			return $response;
		}
		
		return parent::actionInsertReply();
	}
	
	/**
	 * Checks inbox size restrictions and if full throws an error message
	 *
	 * @param XenForo_Model_Conversation $conversationModel
	 *
	 */
	protected function _checkSenderHasInboxSpace(XenForo_Model_Conversation $conversationModel)
	{
		// is sender's inbox full?
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		$inboxMaxSize = XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'convessInboxSize');
		if ($inboxMaxSize > 0) // size restriction
		{
			$total = $conversationModel->countConversationsForUser($viewingUser['user_id']);
			if ($total >= $inboxMaxSize) // reached max conversations
			{
				throw $this->responseException($this->responseError(new XenForo_Phrase('convess_inbox_full_start')));
			}
		}
	}
	
	/**
	 * Checks inbox size restrictions and if full throws an error message
	 *
	 * @param XenForo_Model_Conversation $conversationModel
	 * @param string $recipient - username
	 *
	 */
	protected function _checkRecipientHasInboxSpace(XenForo_Model_Conversation $conversationModel, $recipient)
	{
		$recipient = trim($recipient);
		
		// is receiver's inbox full?
		$toUser = $this->getModelFromCache('XenForo_Model_User')->getUserByName($recipient, array(
			'join' => XenForo_Model_User::FETCH_USER_FULL | XenForo_Model_User::FETCH_USER_PERMISSIONS
		));
		
		if (empty($toUser))
		{
			throw $this->responseException($this->responseError(new XenForo_Phrase('the_following_recipients_could_not_be_found_x', array('names' => $recipient))));
		}
		
		$toUser['permissions'] = XenForo_Permission::unserializePermissions($toUser['global_permission_cache']);
		
		$inboxMaxSize = XenForo_Permission::hasPermission($toUser['permissions'], 'conversation', 'convessInboxSize');
		if ($inboxMaxSize > 0) // size restriction
		{
			$total = $conversationModel->countConversationsForUser($toUser['user_id']);
			if ($total >= $inboxMaxSize) // reached max conversations
			{
				// send alert
				$viewingUser = XenForo_Visitor::getInstance();
				$conversationModel->sendInboxFullAlert($toUser['user_id'], $viewingUser['user_id'], $viewingUser['username']);
				
				throw $this->responseException($this->responseError(new XenForo_Phrase('convess_inbox_full_receive', array('user' => $toUser['username']))));
			}					
		}
	}
	
	/**
	 * Checks inbox size restrictions and if full throws an error message
	 *
	 * @param array $recipients - username array
	 * @param boolean $invite - invite to conversation rather than start one
	 *
	 */
	protected function _checkRecipientsHaveInboxSpace(array $recipients, $invite = false)
	{
		$users = $this->_getUserModel()->getUsersByNames(
			$recipients,
			array('join' => XenForo_Model_User::FETCH_USER_FULL | XenForo_Model_User::FETCH_USER_PERMISSIONS),
			$notFound
		);

		if ($notFound)
		{
			throw $this->responseException($this->responseError(new XenForo_Phrase('the_following_recipients_could_not_be_found_x', array('names' => implode(', ', $notFound)))));
		}
		else
		{
			$conversationModel = $this->_getConversationModel();
			$viewingUser = XenForo_Visitor::getInstance();
			$noStart = array();
			
			foreach ($users AS $key => $user)
			{
				if ($user['user_id'] != $viewingUser['user_id'])
				{
					$user['permissions'] = XenForo_Permission::unserializePermissions($user['global_permission_cache']);
			
					$inboxMaxSize = XenForo_Permission::hasPermission($user['permissions'], 'conversation', 'convessInboxSize');
					if ($inboxMaxSize > 0) // size restriction
					{
						$total = $conversationModel->countConversationsForUser($user['user_id']);
						if ($total >= $inboxMaxSize) // reached max conversations
						{
							// send alert
							$viewingUser = XenForo_Visitor::getInstance();
							$conversationModel->sendInboxFullAlert($user['user_id'], $viewingUser['user_id'], $viewingUser['username']);
					
							$noStart[] = $user['username'];
						}					
					}	
				}
			}
			
			if (!empty($noStart))
			{
				if (count($noStart) > 1)
				{
					if ($invite)
						$phrase = 'convess_inbox_full_invite_x';
					else
						$phrase = 'convess_inbox_full_receive_x';
					
					throw $this->responseException($this->responseError(new XenForo_Phrase($phrase, array('names' => implode(', ', $noStart)))));
				}
				else
				{
					if ($invite)
						$phrase = 'convess_inbox_full_invite';
					else
						$phrase = 'convess_inbox_full_receive';
					
					throw $this->responseException($this->responseError(new XenForo_Phrase($phrase, array('user' => $noStart[0]))));
				}
			}
		}
	}
	
	/**
	 * Gets a preview of the last message.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionTitlePreview()
	{
		$conversationId = $this->_input->filterSingle('conversation_id', XenForo_Input::UINT);
		$conversationModel = $this->_getConversationModel();
		$conversation = $conversationModel->getConversationMasterById($conversationId);
		
		$conversation['lastUser'] = $conversationModel->getConversationRecipient($conversationId, $conversation['last_message_user_id']);
		if (empty($conversation['lastUser']))
		{
			$conversation['lastUser'] = array('user_id' => 0, 'username' => '');
		}
		
		$message = $conversationModel->getConversationMessageById($conversation['last_message_id']);
		$message = XenForo_Helper_String::bbCodeStrip($message['message'], true); // stripQuote
		
		$formatter = XenForo_BbCode_Formatter_Base::create('XenForo_BbCode_Formatter_Text');
		$parser = new XenForo_BbCode_Parser($formatter);
		
		$message = $parser->render($message);
		$conversation['lastMessage'] = XenForo_Helper_String::censorString($message);
		
		$viewParams = array('conversation' => $conversation);
		return $this->responseView('XenForo_ViewPublic_Base', 'convess_title_preview', $viewParams);
	}
	
	/**
	 * Displays all participants
	 *
	 * @return XenForo_ControllerResponse_Base
	 */
	public function actionViewAll()
	{
		$conversationId = $this->_input->filterSingle('conversation_id', XenForo_Input::UINT);
		$conversationModel = $this->_getConversationModel();
		
		// grab conversation title
		$conversation = $conversationModel->getConversationMasterById($conversationId);
		$conversation = array('conversation_id' => $conversation['conversation_id'], 'title' => $conversation['title']);
		
		// fetch recipients
		$recipients = $conversationModel->getConversationRecipients($conversationId);
		
		// remove the visitor from the list
		unset($recipients[XenForo_Visitor::getUserId()]);
		
		$viewParams = array('conversation' => $conversation, 'recipients' => $recipients);
		return $this->responseView('XenForo_ViewPublic_Base', 'convess_conversation_participants', $viewParams);
	}
	
	/**
	 * Add a participant group
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */	
	function actionAddOrUpdateGroup()
	{
		$this->_assertPostOnly();
		$conversationModel = $this->_getConversationModel();
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		
		if (!$conversationModel->canCreateParticipantGroups())
		{
			return $this->responseNoPermission();
		}		
		
		$groupId = $this->_input->filterSingle('group_id', XenForo_Input::UINT);
		$action = $this->_input->filterSingle('action', XenForo_Input::STRING);
		
		if ($action == 'delete') // delete
		{
			$nextGroupId = 0;
			
			// name of first group on page load
			$firstGroupName = $this->_input->filterSingle('first_group_name', XenForo_Input::STRING);
			$deleteName = $this->_input->filterSingle('group_name_'.$groupId, XenForo_Input::STRING);
			// find out if we are deleting the first group - whether from page load or since some groups were deleted
			if ($deleteName == $firstGroupName || $deleteName == $conversationModel->getFirstGroupName($viewingUser['user_id']))
			{
				// get the id of the next group so we can remove its fieldest line once this group is deleted
				$nextGroupId = $conversationModel->getNextGroupIdByNameOrder($viewingUser['user_id'], $deleteName);
			}
			
			$conversationModel->deleteGroupById($groupId);
			$phrase = new XenForo_Phrase('convess_your_group_has_been_deleted');
		}
		else
		{
			if ($action == 'insert') // insert
			{
				$groupName = $this->_input->filterSingle('group_name', XenForo_Input::STRING);
				if (!$groupName)
					throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_must_provide_group_name'));
				
				if ($conversationModel->doesGroupNameExist($viewingUser['user_id'], $groupName))
					throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_group_name_x_must_be_unique', array('name' => $groupName)));
				
				$usernames = explode(',', $this->_input->filterSingle('recipients', XenForo_Input::STRING));
			}
			else // update
			{
				$groupName = $this->_input->filterSingle('group_name_'.$groupId, XenForo_Input::STRING);
				if (!$groupName)
					throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_must_provide_group_name'));
				
				$currentNmae = $conversationModel->getGroupName($groupId);
				if (strtolower($currentNmae) != strtolower($groupName))
				{
					if ($conversationModel->doesGroupNameExist($viewingUser['user_id'], $groupName))
						throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_group_name_x_must_be_unique', array('name' => $groupName)));		
				}
				
				$usernames = explode(',', $this->_input->filterSingle('recipients_'.$groupId, XenForo_Input::STRING));
			}
			
			// weed out empty strings
			foreach ($usernames AS $key => &$username)
			{
				$username = trim($username);
				if (!$username)
					unset($usernames[$key]);
			}
			
			if (empty($usernames))
				throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_must_provide_recipients'));
			
			if ($this->_canAddTheseRecipients($usernames))
			{
				if ($groupId) // update
				{
					$conversationModel->updateParticipantGroup($groupId, $groupName, $usernames);
					$phrase = new XenForo_Phrase('convess_your_group_has_been_updated');
				}
				else // insert
				{
					$groupId = $conversationModel->addParticipantGroup($groupName, $usernames);
				}
			}
		}
		
		if ($action == 'insert')
		{
			$maxRecipients = XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'maxRecipients');
			$group = $conversationModel->getParticipantGroupById($groupId);
			$group = $conversationModel->processParticipantGroupData($group, $maxRecipients);
			$numGroups = $conversationModel->countParticipantGroups($viewingUser['user_id']);
			$maxGroups = XenForo_Application::get('options')->convess_max_groups;
			
			$viewParams = array(
				'group' => $group,
				'action' => $action,
				'id' => $groupId,
				'inlineInsert' => true,
				'numGroups' => $numGroups,
				'maxGroups' => $maxGroups
			);
			
			return $this->responseView('ConvEss_ViewPublic_ConvEss', 'convess_group_item', $viewParams);
		}
		else // update or delete
		{
			$params = array('action' => $action, 'id' => $groupId);
			
			// see if we are deleting last group
			if ($action == 'delete')
			{
				$params['numGroups'] =  $conversationModel->countParticipantGroups($viewingUser['user_id']);
				$params['maxGroups'] = XenForo_Application::get('options')->convess_max_groups;
				$params['nextGroupId'] = $nextGroupId;
			}
			else
			{
				$params['groupName'] = $groupName;
				$params['recipients'] = implode(', ', $usernames);
			}
			
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('account/conversation-groups'),
				$phrase,
				$params
			);
		}
	}
	
	/**
	 * Checks whether recipients can be added. This checks that the visitor/invite user can send to the recipients.
	 *
	 * @param array $usernames
	 *
	 */
	protected function _canAddTheseRecipients(array &$usernames)
	{		
		$visitor = XenForo_Visitor::getInstance()->toArray();
		
		$users = $this->_getUserModel()->getUsersByNames(
			$usernames,
			array(
				'join' => XenForo_Model_User::FETCH_USER_PRIVACY + XenForo_Model_User::FETCH_USER_OPTION,
				'followingUserId' => $visitor['user_id']
			),
			$notFound
		);
		
		if ($notFound)
		{
			throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('the_following_recipients_could_not_be_found_x', array('names' => implode(', ', $notFound))), 'recipients');
		}
		else
		{
			$conversationModel = $this->_getConversationModel();
			$noStart = array();
			$canStart = array();
			foreach ($users AS $key => $user)
			{
				if ($visitor['user_id'] == $user['user_id'])
				{
					// skip trying to add self
					continue;
				}
				
				if (!$conversationModel->canStartConversationWithUser($user, $null, $visitor))
				{
					$noStart[] = $user['username'];
				}
				else
				{
					$canStart[$user['user_id']] = $user['username'];
				}
			}
			
			$usernames = $canStart;
			
			if ($noStart)
			{
				throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('you_may_not_start_a_conversation_with_the_following_recipients_x', array('names' => implode(', ', $noStart))), 'recipients');
			}
			else
			{
				if (empty($usernames))
					throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_must_provide_recipients_other_than_self'));
				
				$maxRecipients = XenForo_Permission::hasPermission($visitor['permissions'], 'conversation', 'maxRecipients');
				if ($maxRecipients > -1 && count($users) > $maxRecipients)
					throw $this->getErrorOrNoPermissionResponseException(new XenForo_Phrase('convess_you_may_only_have_x_recipients', array('max' => $maxRecipients)));
				
				return true;
			}
		}
	}
	
	/**
	 * Get all conversation prefixes for a certain user
	 *
	 * @param array $userId
	 * @param XenForo_Model_Conversation $conversationModel
	 *
	 * return array
	 *
	 */	
	protected function _getAllPrefixes($userId, XenForo_Model_Conversation $conversationModel = null)
	{
		if (!isset($conversationModel))
			$conversationModel = $this->_getConversationModel();
		
		return $conversationModel->getPrefixes($userId);
	}
	
	/**
	 * Displays a form to like a message or likes a message (via, uhh, POST).
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionLike()
	{
		// conversation exists and viewable
		$conversationId = $this->_input->filterSingle('conversation_id', XenForo_Input::UINT);
		$conversation = $this->_getConversationOrError($conversationId);
		
		// message exists
		if ($messageId = $this->_input->filterSingle('m', XenForo_Input::UINT)) // 'm' is a shortcut for 'message_id'
		{
			$conversationModel = $this->_getConversationModel();
			
			if ($message = $conversationModel->getConversationMessageByIdForLike($messageId))
			{
				if (!$conversationModel->canLikeMessage($message['user_id']))
				{
					return $this->responseError(new XenForo_Phrase('convess_cannot_like_own_message'));
				}
				
				if ($message['conversation_id'] != $conversationId)
				{
					return $this->responseError(new XenForo_Phrase('convess_not_possible_to_like_message_not_same_conversation'));
				}
				
				$likeModel = $this->_getLikeModel();
				$existingLike = $likeModel->getContentLikeByLikeUser('convess_message', $messageId, XenForo_Visitor::getUserId());
				
				if ($this->_request->isPost())
				{
					if ($existingLike)
					{
						$latestUsers = $likeModel->unlikeConversationMessage($existingLike);
					}
					else
					{
						$latestUsers = $likeModel->likeConversationMessage('convess_message', $messageId, $message['user_id']);
					}
					
					$liked = ($existingLike ? false : true);
					
					if ($this->_noRedirect() && $latestUsers !== false)
					{
						$message['likeUsers'] = $latestUsers;
						$message['likes'] += ($liked ? 1 : -1);
						$message['like_date'] = ($liked ? XenForo_Application::$time : 0);
		
						$viewParams = array(
							'message' => $message,
							'liked' => $liked
						);
		
						return $this->responseView('ConvEss_ViewPublic_Conversation_LikeConfirmed', '', $viewParams);
					}
					else
					{
						return $this->responseRedirect(
							XenForo_ControllerResponse_Redirect::SUCCESS,
							XenForo_Link::buildPublicLink('conversations/message', $conversation, array('message_id' => $message['message_id'])) . '#message-' . $message['message_id']
						);
					}
				}
				else
				{
					$viewParams = array(
						'conversation' => $conversation,
						'message' => $message,
						'like' => $existingLike
					);
								
					return $this->responseView('XenForo_ViewPublic_Base', 'convess_message_like', $viewParams);
				}
			}
			else
			{
				return $this->responseError(new XenForo_Phrase('requested_message_not_found'));
			}
		}
		
		$errorPhraseKey = '';
		throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
	}

	/**
	 * List of everyone that liked this post.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionLikes()
	{
		// conversation exists and viewable
		$conversationId = $this->_input->filterSingle('conversation_id', XenForo_Input::UINT);
		$conversation = $this->_getConversationOrError($conversationId);
		
		// message exists
		if ($messageId = $this->_input->filterSingle('m', XenForo_Input::UINT)) // 'm' is a shortcut for 'message_id'
		{	
			$message = $this->_getConversationModel()->getConversationMessageById($messageId);
			
			if (empty($message))
			{
				return $this->responseError(new XenForo_Phrase('requested_message_not_found'));
			}
			
			if ($message['conversation_id'] != $conversationId)
			{
				return $this->responseError(new XenForo_Phrase('convess_not_possible_to_like_message_not_same_conversation'));
			}
			
			$likes = $this->_getLikeModel()->getContentLikes('convess_message', $messageId);
			if (!$likes)
			{
				return $this->responseError(new XenForo_Phrase('no_one_has_liked_this_post_yet'));
			}
			
			$viewParams = array(
				'conversation' => $conversation,
				'message' => $message,
				'likes' => $likes
			);
			
			return $this->responseView('XenForo_ViewPublic_Base', 'convess_message_likes', $viewParams);
		}
		
		$errorPhraseKey = '';
		throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
	}
	
	/**
	 * @return XenForo_Model_User
	 */
	protected function _getUserModel()
	{
		return $this->getModelFromCache('XenForo_Model_User');
	}
	
	/**
	 * @return XenForo_Model_Like
	 */
	protected function _getLikeModel()
	{
		return $this->getModelFromCache('XenForo_Model_Like');
	}
}