<?php

/**
 * Model for conversations.
 *
 * @package ConvEss
 */
class ConvEss_Model_Conversation extends XFCP_ConvEss_Model_Conversation
{
	/**
	 * Count multiple auto responses - used to ensure the last auto response is also the last indicated message
	 *
	 * @var integer
	 */
	protected $_autoResponseCount = 1;
	
	/**
	 * Prepare a message for display or further processing.
	 *
	 * @param array $message
	 * @param array $conversation
	 *
	 * @return array Prepared message
	 */
	public function prepareMessage(array $message, array $conversation)
	{
		$message = parent::prepareMessage($message, $conversation);
		
		if (!array_key_exists('canLike', $message)) // was not already processed in function prepareMessages()
		{
			$message['canLike'] = $this->canLikeMessage($message['user_id']);
			$messageLikes = $this->getMessageLikesByMessageId($message['message_id'], XenForo_Visitor::getUserId());
			
			if (!empty($messageLikes))
			{
				$message['likes'] = $messageLikes['likes'];
				$message['likeUsers'] = unserialize($messageLikes['like_users']);
				$message['like_date'] = $messageLikes['like_date'];
			}
		}
		
		return $message;
	}
	
	/**
	 * Prepare a collection of messages (in the same conversation) for display or
	 * further processing.
	 *
	 * @param array $messages
	 * @param array $conversation
	 *
	 * @return array Prepared messages
	 */

	public function prepareMessages(array $messages, array $conversation)
	{
		$messageIds = array_keys($messages);
		$messageLikes = $this->getMessageLikesByMessageIds($messageIds, XenForo_Visitor::getUserId());
		$userCanLikeArray = array();
		
		// batch process them instead of individually in prepareMessage()
		foreach ($messages AS $messageId => &$message)
		{
			$userId = $message['user_id'];
			if (!array_key_exists($userId, $userCanLikeArray))
				$userCanLikeArray[$userId] = $this->canLikeMessage($userId);
			
			$message['canLike'] = $userCanLikeArray[$userId];
			
			if (!empty($messageLikes) && isset($messageLikes[$messageId]))
			{
				$message['likes'] = $messageLikes[$messageId]['likes'];
				$message['likeUsers'] = unserialize($messageLikes[$messageId]['like_users']);
				$message['like_date'] = $messageLikes[$messageId]['like_date'];
			}
		}
		
		return parent::prepareMessages($messages, $conversation);
	}
	
	/**
	 * Get the recipient state from message id.
	 *
	 * @param integer $messageId
	 * @param integer $userId
	 *
	 * @return string recipient_state
	 */

	public function getRecipientStateFromMessageId($messageId, $userId)
	{
		return $this->_getDb()->fetchOne('
			SELECT conversation_recipient.recipient_state
			FROM xf_conversation_recipient AS conversation_recipient
			INNER JOIN xf_conversation_message AS conversation_message ON (conversation_message.message_id = ? AND conversation_message.conversation_id = conversation_recipient.conversation_id)
			WHERE conversation_recipient.user_id = ?
		', array($messageId, $userId));
	}
	
	/**
	 * Gets the specified conversation message record.
	 *
	 * @param integer $conversationId
	 * @param integer $messageId
	 *
	 * @return array|false
	 */
	public function getConversationMessageByIdForLike($messageId)
	{
		return $this->_getDb()->fetchRow('
			SELECT message.*, likes.likes,
				user.*, IF(user.username IS NULL, message.username, user.username) AS username,
				user_profile.*
			FROM xf_conversation_message AS message
			LEFT JOIN convess_message_likes AS likes ON
				(likes.message_id = message.message_id)
			LEFT JOIN xf_user AS user ON
				(user.user_id = message.user_id)
			LEFT JOIN xf_user_profile AS user_profile ON
				(user_profile.user_id = message.user_id)
			WHERE message.message_id = ?
		', $messageId);
	}
	
	/**
	 * Get message likes by id
	 *
	 * @param integer $messageId
	 * @param integer $userId
	 *
	 * @return array convess_message_likes
	 */	
	public function getMessageLikesByMessageId($messageId, $userId)
	{
		return $this->_getDb()->fetchRow('
			SELECT likes.*, liked.like_date
			FROM convess_message_likes AS likes
			LEFT JOIN xf_liked_content AS liked ON (liked.content_type = \'convess_message\' AND liked.content_id = likes.message_id AND liked.like_user_id = ?)
			WHERE likes.message_id = ?
		', array($userId, $messageId));
	}
	
	/**
	 * Get all message likes by ids
	 *
	 * @param array $messageIds
	 * @param integer $userId
	 *
	 * @return array message_id
	 */	
	public function getMessageLikesByMessageIds(array $messageIds, $userId)
	{
		if (!$messageIds)
		{
			return array();
		}
		
		return $this->fetchAllKeyed('
			SELECT likes.*, liked.like_date
			FROM convess_message_likes AS likes
			LEFT JOIN xf_liked_content AS liked ON (liked.content_type = \'convess_message\' AND liked.content_id = likes.message_id AND liked.like_user_id = ?)
			WHERE likes.message_id IN (' . $this->_getDb()->quote($messageIds) . ')
		', 'message_id', $userId);
	}
	
	/**
	 * Get all message like users by conversation id
	 *
	 * @param integer $messageId
	 *
	 * @return array message_id
	 */	
	public function getMessageLikes($messageId)
	{
		$messageId = intval($messageId);
		
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM convess_message_likes
			WHERE message_id = ?
			LIMIT 1
		', $messageId);
	}
	
	/**
	 * Get the conversation id and title associated to specific messages.
	 *
	 * @param array $messageIds
	 *
	 * @return array message_id => (conversation_id, message_id, title)
	 */
	public function getMessagesForLikesByIds(array $messageIds)
	{
		if (!$messageIds)
		{
			return array();
		}
		
		return $this->fetchAllKeyed('
			SELECT conversation_message.*, conversation_master.title
			FROM xf_conversation_message AS conversation_message
			INNER JOIN xf_conversation_master AS conversation_master ON	(conversation_master.conversation_id = conversation_message.conversation_id)
			INNER JOIN xf_conversation_recipient AS conversation_recipient ON (conversation_recipient.user_id = conversation_message.user_id AND conversation_recipient.conversation_id = conversation_master.conversation_id AND conversation_recipient.recipient_state = \'active\')
			WHERE conversation_message.message_id IN (' . $this->_getDb()->quote($messageIds) . ')
		', 'message_id');
	}
	
	/**
	 * Get messages ids for a given conversation.
	 *
	 * @param integer $conversationId
	 *
	 * @return array Format [message id] => message_id
	 */
	public function getConversationMessageIds($conversationId)
	{
		return $this->fetchAllKeyed('
				SELECT message_id, user_id
				FROM xf_conversation_message
				WHERE conversation_id = ?
		', 'message_id', $conversationId);
	}
	
	/**
	 * Get all prefixes belonging to a specific user
	 *
	 * @param integer $userId 
	 *
	 * @return array convess_prefix
	 */	
	public function getPrefixes($userId)
	{
		$userId = intval($userId);
		
		return $this->fetchAllKeyed('
			SELECT *
			FROM convess_prefix
			WHERE user_id = ?
			ORDER BY display_order ASC
		', 'prefix_id', $userId);
	}
	
	/**
	 * Get a prefix
	 *
	 * @param integer $prefixId - the auto increment value of the item
	 *
	 * @return array convess_prefix
	 */	
	public function getPrefixById($prefixId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM convess_prefix
			WHERE prefix_id = ?
		', $prefixId);		
	}
	
	/**
	 * Get a convess by conversation and User Id
	 *
	 * @param integer $conversationId
	 * @param integer $userId 
	 *
	 * @return array convess
	 */	
	public function getConvessByConversationUserId($conversationId, $userId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM convess
			WHERE conversation_id = ? AND user_id = ?
		', array($conversationId, $userId));		
	}
	
	/**
	 * Get all convess belonging to a certain user
	 *
	 * @param integer $userId 
	 *
	 * @return array convess
	 */	
	function getConvessByUserId($userId)
	{
		$userId = intval($userId);
		
		return $this->fetchAllKeyed('
			SELECT *
			FROM convess
			WHERE user_id = ?
		', 'conversation_id', $userId);
	}
	
	/**
	 * Get all convess belonging to a certain conversation
	 *
	 * @param integer $conversationId 
	 *
	 * @return array convess
	 */	
	function getConvessByConversationId($conversationId)
	{
		$conversationId = intval($conversationId);
		
		return $this->fetchAllKeyed('
			SELECT *
			FROM convess
			WHERE conversation_id = ?
		', 'user_id', $conversationId);
	}
	
	/**
	 * Finds out if a user was kicked from a conversation
	 *
	 * @param integer $conversationId
	 * @param integer $userId 
	 *
	 * @return boolean
	 */	
	public function isUserKickedFromConversation($conversationId, $userId)
	{
		return $this->_getDb()->fetchOne('
			SELECT kicked
			FROM convess
			WHERE conversation_id = ? AND user_id = ?
			LIMIT 1', array($conversationId, $userId));
	}
	
	/**
	 * Get all kicked users by conversation id
	 *
	 * @param integer $conversationId 
	 *
	 * @return array user_id
	 */	
	public function getKickedByConversationId($conversationId)
	{
		$conversationId = intval($conversationId);
		
		return $this->fetchAllKeyed('
			SELECT user_id
			FROM convess
			WHERE conversation_id = ? AND kicked = 1
		', 'user_id', $conversationId);
	}
	
	/**
	 * Count number of kicked users from a conversation
	 *
	 * @param integer $conversationId
	 *
	 * @return integer
	 */	
	public function countKickedFromConversation($conversationId)
	{
		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM convess
			WHERE conversation_id = ? AND kicked = 1
			LIMIT 1', $conversationId);
	}
	
	/**
	 * Count number of users who have left the conversation permanently
	 *
	 * @param integer $conversationId
	 *
	 * @return integer
	 */	
	public function countNumLeftPermanently($conversationId)
	{
		$db = $this->_getDb();
		
		return $db->fetchOne('
			SELECT COUNT(*)
			FROM xf_conversation_recipient
			WHERE conversation_id = ? AND recipient_state = ' . $db->quote('deleted_ignored') . '
			LIMIT 1', $conversationId);
	}
	
	/**
	 * Get a prefix by conversation Id
	 *
	 * @param integer $conversationId
	 * @param integer $userId 
	 *
	 * @return array convess_prefix
	 */	
	public function getPrefixByConversationId($conversationId, $userId)
	{
		return $this->_getDb()->fetchOne('
			SELECT prefix_id
			FROM convess
			WHERE conversation_id = ? AND user_id = ?
		', array($conversationId, $userId));		
	}
	
	/**
	 * Delete a prefix
	 *
	 * @param integer $prefixId
	 */		
	function deletePrefixById($prefixId)
	{
		$this->_getDb()->query('
			DELETE FROM convess_prefix
			WHERE prefix_id = ?
		', $prefixId);		
	}
	
	/**
	 * Get the current count of prefixes for a given user
	 *
	 * @param string $userId
	 *
	 * @return integer
	 */
	public function getNumPrefixes($userId)
	{
		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM convess_prefix
			WHERE user_id = ?
		', $userId);		
	}
	
	/**
	 * Get all participant groups belonging to a specific user
	 *
	 * @param integer $userId 
	 *
	 * @return array convess_group
	 */	
	function getParticipantGroups($userId)
	{
		$userId = intval($userId);
		
		return $this->fetchAllKeyed('
			SELECT *
			FROM convess_group
			WHERE user_id = ?
			ORDER BY group_name ASC
		', 'group_id', $userId);
	}
	
	/**
	 * Get a participant group
	 *
	 * @param integer $groupId - the auto increment value of the item
	 *
	 * @return array convess_group
	 */	
	public function getParticipantGroupById($groupId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM convess_group
			WHERE group_id = ?
		', $groupId);		
	}
	
	/**
	 * Count all participant groups belonging to a specific user
	 *
	 * @param integer $userId
	 *
	 * @return integer
	 */	
	public function countParticipantGroups($userId)
	{
		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM convess_group
			WHERE user_id = ?
		', $userId);
	}
	
	/**
	 * Check to see if there is already a group with a particular name
	 *
	 * @param integer $userId
	 * @param string $groupName
	 *
	 * @return boolean
	 */	
	public function doesGroupNameExist($userId, $groupName)
	{
		return $this->_getDb()->fetchOne('
			SELECT group_id
			FROM convess_group
			WHERE user_id = ? AND group_name = ?
		', array($userId, $groupName));
	}
	
	/**
	 * Get a participant group user
	 *
	 * @param integer $groupId
	 * @param integer $userId 
	 *
	 * @return array convess_group_user
	 */	
	public function getParticipantGroupUserById($groupId, $userId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM convess_group_user
			WHERE group_id = ? AND user_id = ?
		', array($groupId, $userId));		
	}
	
	/**
	 * Get all participant users by group id
	 *
	 * @param integer $groupId
	 * @param string $key - fetch by this key name
	 *
	 * @return array user_id, username
	 */	
	function getParticipantGroupUsers($groupId, $key = 'username')
	{
		$key = trim($key);
		if ($key != 'username' && $key != 'user_id')
			$key = 'username';
		
		$groupId = intval($groupId);
		
		return $this->fetchAllKeyed('
			SELECT user_id, username
			FROM convess_group_user
			WHERE group_id = ?
		', $key, $groupId);
	}
	
	/**
	 * Check to see if this conversation is a sticky
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 *
	 * @return boolean
	 */	
	public function isSticky($conversationId, $userId)
	{
		return $this->_getDb()->fetchOne('
			SELECT sticky
			FROM convess
			WHERE conversation_id = ? AND user_id = ?
		', array($conversationId, $userId));
	}
	
	/**
	 * Check to see if a user has already auto responded to a conversation
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 *
	 * @return boolean
	 */	
	public function hasAutoResponded($conversationId, $userId)
	{
		return $this->_getDb()->fetchOne('
			SELECT auto_responded
			FROM convess
			WHERE conversation_id = ? AND user_id = ?
		', array($conversationId, $userId));
	}
	
	/**
	 * Reset the auto response flag on all conversations
	 *
	 * @param integer $userId 
	 *
	 */	
	protected function _resetConvEssAutoResponseFlag($userId)
	{
		$userId = intval($userId);
				
		$this->_getDb()->query('
			UPDATE convess
			SET auto_responded = 0
			WHERE user_id = ?
			', $userId);
	}
	
	/**
	 * Get the group's name
	 *
	 * @param integer $groupId
	 *
	 * @return string
	 */	
	public function getGroupName($groupId)
	{
		return $this->_getDb()->fetchOne('
			SELECT group_name
			FROM convess_group
			WHERE group_id = ?
			LIMIT 1
		', $groupId);
	}
	
	/**
	 * Get the first group's name
	 *
	 * @param integer $userId
	 *
	 * @return string
	 */	
	public function getFirstGroupName($userId)
	{
		return $this->_getDb()->fetchOne('
			SELECT group_name
			FROM convess_group
			WHERE user_id = ?
			ORDER BY group_name ASC
			LIMIT 1
		', $userId);
	}
	
	/**
	 * Get the next group's id by order of group_name
	 *
	 * @param integer $userId
	 * @param string $groupName
	 *
	 * @return integer
	 */	
	public function getNextGroupIdByNameOrder($userId, $groupName)
	{
		$db = $this->_getDb();
		
		return $db->fetchOne('
			SELECT group_id
			FROM convess_group
			WHERE user_id = ? AND group_name > ?
			ORDER BY group_name ASC
			LIMIT 1
		', array($userId, $groupName));
	}
	
	/**
	 * Return a list of conversations, with no replies, to which the specified user is the recipient,
	 *
	 * @param integer $userId
	 * @param integer $cutOff - limit conversations that are only X days old
	 *
	 * @return array
	 */
	public function findConversationsWithNoRepliesByRecipientId($userId, $cutOff)
	{
		return $this->fetchAllKeyed('
				SELECT conversation_master.*
				FROM xf_conversation_master AS conversation_master
				INNER JOIN xf_conversation_recipient AS conversation_recipient ON
					(conversation_recipient.conversation_id = conversation_master.conversation_id
					AND conversation_recipient.user_id = ?)
				WHERE conversation_master.reply_count = 0 AND conversation_master.last_message_date > ?
		', 'user_id', array($userId, $cutOff));
	}
	
	/**
	 * Gets basic information about all recipients of a conversation.
	 *
	 * @param integer $conversationId
	 * @param string $keyed by
	 *
	 * @return array Format: [user id] => info
	 */
	public function getConversationRecipientsSimple($conversationId, $key = 'username')
	{
		return $this->fetchAllKeyed('
			SELECT conversation_recipient.*, user.username
			FROM xf_conversation_recipient AS conversation_recipient
			LEFT JOIN xf_user AS user ON
				(user.user_id = conversation_recipient.user_id)
			WHERE conversation_recipient.conversation_id = ?
			ORDER BY user.username
		', $key, $conversationId);
	}
	
	/**
	 * Get last read date from convess
	 *
	 * @param integer $conversationId
	 * @param integer $userId 
	 *
	 * @return array convess
	 */	
	public function getRecipientReadDate($conversationId, $userId)
	{
		return $this->_getDb()->fetchRow('
			SELECT last_read_message_id, last_read_message_date
			FROM convess
			WHERE conversation_id = ? AND user_id = ?
		', array($conversationId, $userId));		
	}
	
	/**
	 * Get a user's convess user options settings
	 *
	 * @param integer $userId 
	 *
	 * @return array convess_user_options
	 */	
	public function getConvEssUserOptionsByUserId($userId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM convess_user_options
			WHERE user_id = ?
			LIMIT 1
		', $userId);		
	}
	
	/**
	 * Get a user's convess user options
	 *
	 * @param array $userIds
	 *
	 * @return array convess_user_options
	 */	
	public function getConvEssUserOptionsyByUserIds(array $userIds)
	{
		if (!$userIds)
		{
			return array();
		}
		
		return $this->fetchAllKeyed('
			SELECT *
			FROM convess_user_options
			WHERE user_id IN (' . $this->_getDb()->quote($userIds) . ')
		', 'user_id');		
	}
	
	/**
	 * Gets a message ID by post date.
	 *
	 * @param integer $conversationId
	 * @param integer $messageDate Finds first message posted after this
	 *
	 * @return array|false
	 */
	public function getMessageIdByDate($conversationId, $messageDate)
	{
		// searching for smaller or equals to date because when marking a conversation
		// as unread, then read, it assigns current timestamp and not last message date

		return $this->_getDb()->fetchOne('
			SELECT message_id
			FROM xf_conversation_message
			WHERE conversation_id = ?
				AND message_date <= ?
			ORDER BY message_id DESC
			LIMIT 1
		', array($conversationId, $messageDate));
	}
	
	/**
	 * Update the read date record for a conversation
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 * @param integer $newReadDate
	 * @param Zend_Db_Adapter_Abstract $db
	 *
	 * @return integer $newReadDate
	 */
	protected function _updateConversationReadDate($conversationId, $userId, $newReadDate, Zend_Db_Adapter_Abstract $db = null)
	{
		if (XenForo_Application::get('options')->convess_last_read_date)
		{
			$newReadDate = intval($newReadDate);
			
			if ($newReadDate > 0)
			{
				$exists = $this->getConvessByConversationUserId($conversationId, $userId);
				$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
				if (!empty($exists))
				{
					$setVars = array(
						'conversation_id' => $conversationId,
						'user_id' => $userId
					);
					$dw->setExistingData($setVars);
				}
				else
				{
					$vars['conversation_id'] = intval($conversationId);
					$vars['user_id'] = intval($userId);
				}
				
				// $newReadDate can contain either current time (unread to read toggle) or last message time (actually viewing conversation)
				// use current time instead so that after say 2 days of it saying the user never read the conversation,
				// it does not all of a sudden say they have read it when last message was created.
				$time = XenForo_Application::$time;
				$vars['last_read_message_date'] = $time; //$newReadDate;
				$vars['last_read_message_id'] = $this->getMessageIdByDate($conversationId, $time); // point to last message they read
				
				$dw->bulkSet($vars);
				$dw->save();			
			}
		}

		return parent::_updateConversationReadDate($conversationId, $userId, $newReadDate, $db);
	}
		
	/**
	 * Get the auto response info
	 *
	 * @param integer $userId
	 * @param integer $active
	 *
	 * @return array / false
	 */		
	public function getAutoResponse($userId, $active = null)
	{
		$userId = intval($userId);
		$whereClause = '';
		if (isset($active))
		{
			$active = intval($active);
			$whereClause = ' AND active = ' . $active;
		}
		
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM convess_auto_response
			WHERE user_id = ?' . $whereClause . '
		', $userId);
	}
	
	/**
	 * fetch and process the general auto response conversation message
	 *
	 * @param integer $conversationId
	 * @param string $toUsername
	 * @param string $fromUsername
	 *
	 * @return string $message
	 */		
	public function getGeneralAutoResponseMessage($conversationId, $toUsername, $fromUsername)
	{
		$message = XenForo_Application::get('options')->convess_mod_away_response;
		$message = str_replace('{to}' , $toUsername, $message);
		$message = str_replace('{from}' , $fromUsername, $message);
		
		// insert a url for each of the moderator names
		// [MOD]moderator_name[/MOD]
		$end = 0;
		while (($start = stripos($message, '[MOD]', $end)) !== false)
		{
			$end = stripos($message, '[/MOD]', $start);
			$name = substr($message, $start + 5, ($end - $start) - 5);
			$modName = trim($name);
			
			if (!empty($modName))
			{
				$user = $this->_getUserModel()->getUserByName($modName);
			}
			
			// valid user
			if (!empty($modName) && !empty($user))// && ($user['is_admin'] || $user['is_moderator']))
			{			
				if ($modName == $fromUsername) // the AWOL mod
				{
					$message = str_ireplace('[MOD]'.$name.'[/MOD]' , $modName.' - [I]' . new XenForo_Phrase('convess_currently_unavailable').'[/I]', $message);
				}
				else
				{
					$url = XenForo_Link::buildPublicLink('canonical:conversations/add?to='.$modName);
					$recipients = $this->getConversationRecipientsSimple($conversationId);
					if (array_key_exists($modName, $recipients)) // already a participant
					{
						$state = $recipients[$modName]['recipient_state'];
						if ($state == 'deleted')
							$phrase = new XenForo_Phrase('convess_user_left');
						else if ($state == 'deleted_ignored')
							$phrase = new XenForo_Phrase('convess_user_left_permanently');
						else
							$phrase = new XenForo_Phrase('convess_already_participant');
							
						$message = str_ireplace('[MOD]'.$name.'[/MOD]' , '[URL='.$url.']'.$modName.'[/URL] - [I]' . $phrase .'[/I]', $message);
					}
					else // an avialable mod
					{
						$message = str_ireplace('[MOD]'.$name.'[/MOD]' , '[URL='.$url.']'.$modName.'[/URL]', $message);
					}				
				}
			}
			else // not a valid user
			{
				if (empty($modName))
					$modName = new XenForo_Phrase('n_a');
				
				$message = str_ireplace('[MOD]'.$name.'[/MOD]' , $modName, $message);
			}
		}
									
		$message = XenForo_Helper_String::autoLinkBbCode($message);
		$message = XenForo_Helper_String::censorString($message);
		return $message;
	}
	
	/**
	 * Gets information about all recipients of a conversation.
	 *
	 * @param array $conversationIds
	 * @param array $fetchOptions Options for extra data to fetch
	 *
	 * @return array Format: [conversation_id]['user_id'] => info
	 */
	public function getConversationRecipientsByConversationIds(array $conversationIds, array $fetchOptions = array())
	{
		$db = $this->_getDb();
		$fetchOptions['skip_convess'] = 1;
		$sqlClauses = $this->prepareConversationFetchOptions($fetchOptions);
		
		$recipients = $db->fetchAll('
			SELECT conversation_recipient.*,
				user.*, user_option.*
				' . $sqlClauses['selectFields'] . '
			FROM xf_conversation_recipient AS conversation_recipient
			LEFT JOIN xf_user AS user ON
				(user.user_id = conversation_recipient.user_id)
			LEFT JOIN xf_user_option AS user_option ON
				(user_option.user_id = user.user_id)
			' . $sqlClauses['joinTables'] . '
			WHERE conversation_recipient.conversation_id IN (' . $db->quote($conversationIds) . ')
			ORDER BY user.username
		');
		
		$recipientsArray = array();
		
		// group by conversation_id
		foreach ($recipients AS $recipient)
		{
			$recipientsArray[$recipient['conversation_id']][$recipient['user_id']] = $recipient;
		}
		
		return $recipientsArray;
	}
	
	/**
	 * send an auto response conversation message
	 *
	 * @param integer $conversationId
	 * @param array $user - the sender
	 * @param string $message
	 *
	 * @return boolean True on success
	 */		
	public function sendAutoResponse($conversationId, array $user = array(), $message)
	{
		$conversationId = intval($conversationId);
		if (!$conversationId || empty($user) || empty($message))
			return false;
		
		$message = XenForo_Helper_String::censorString($message);
		
		$messageDw = XenForo_DataWriter::create('XenForo_DataWriter_ConversationMessage');
		$messageDw->setExtraData(XenForo_DataWriter_ConversationMessage::DATA_MESSAGE_SENDER, $user);
		$messageDw->setOption(XenForo_DataWriter_ConversationMessage::OPTION_SET_IP_ADDRESS, false);
		
		$messageDw->set('conversation_id', $conversationId);
		$messageDw->set('message_date', XenForo_Application::$time + $this->_autoResponseCount); // force the last indicated message to be this one
		$messageDw->set('user_id', $user['user_id']);
		$messageDw->set('username', $user['username']);
		$messageDw->set('message', $message);
		$messageDw->preSave();
		$messageDw->save();
		
		$this->_autoResponseCount += 1; // increment the count so that the last of multiple auto responses will be the last indicated message
		$this->_saveAutoResponseAction($conversationId, $user['user_id']);
	}
	
	/**
	 * ensure response will not be sent again next time this conversation gets a message
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 *
	 */		
	protected function _saveAutoResponseAction($conversationId, $userId)
	{
		$exists = $this->getConvessByConversationUserId($conversationId, $userId);
		$convEssDw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
		$setVars = array(
				'conversation_id' => $conversationId,
				'user_id' => $userId
		);
		
		if (!empty($exists))
		{
			$convEssDw->setExistingData($setVars);
			$convEssDw->set('auto_responded', 1);
		}
		else
		{
			$setVars['auto_responded'] = 1;				
			$convEssDw->bulkSet($setVars);
		}
						
		$convEssDw->save();
	}
	
	/**
	 * Sets the active state of the auto response message
	 *
	 * @param string $userId
	 * @param integer $state  1 = active
	 * @param string $errorKey - passed by reference
	 *
	 * @return boolean
	 */
	public function autoResponseSetActiveState($userId, $state, &$errorKey)
	{
		$userId = intval($userId);
		$state = intval($state);
		
		$exists = $this->getAutoResponse($userId);
		if (empty($exists))
		{
			$errorKey = 'autoResponseSetActiveState(): Auto Response item not found';
			return false;
		}
		
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_AutoResponse');
		$dw->setExistingData(array('user_id' => $userId));
		$dw->set('active', $state);
		$dw->preSave();

		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			return false;
		}

		$dw->save();
		return true;
	}
	
	/**
	 * Saves or updates an auto response
	 *
	 * @param string $userId
	 * @param string $dates - serialized array of dates info
	 * @param string $message
	 * @param boolean $exclude - whether to exclude members the user follows
	 * @param string $errorKey - passed by reference
	 *
	 * @return boolean
	 */
	public function autoResponseSave($userId, $dates, $message, $exclude, &$errorKey)
	{
		$userId = intval($userId);
		if (empty($message)) // form was reset
			$active = 0;
		else
			$active = 1;
			
		$exists = $this->getAutoResponse($userId);
		
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_AutoResponse');
		
		if (!empty($exists)) // update
		{
			$dw->setExistingData(array('user_id' => $userId));
		}
		
		$vars = array(
			'user_id' => $userId,
			'active' => $active,
			'dates' => $dates,
			'message' => $message,
			'exclude' => $exclude
		);	
		
		$dw->bulkSet($vars);
		$dw->preSave();
		
		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			return false;
		}
		
		$dw->save();
		
		// remove any past auto response indication from conversations
		if ($active)
			$this->_resetConvEssAutoResponseFlag($userId);
		
		return true;
	}
					
	/**
	 * Lock conversation.
	 *
	 * @param integer $conversationId
	 * @param integer $state: 1 = open, 0 = closed
	 */
	public function setConversationLockState($conversationId, $state)
	{
		$state = intval($state);
		if ($state > 1)
			$state = 1;
		
		$conversationDw = XenForo_DataWriter::create('XenForo_DataWriter_ConversationMaster', XenForo_DataWriter::ERROR_SILENT);
		$conversationDw->setExistingData($conversationId);
		$conversationDw->set('conversation_open', $state);
		$conversationDw->save();
	}
	
	/**
	 * Determines if the viewing user can create participant groups.
	 *
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */	
	public function canCreateParticipantGroups(array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$this->canStartConversations($errorPhraseKey, $viewingUser))
			return false;

		if (!$viewingUser['user_id'])
		{
			return false;
		}

		if ($viewingUser['user_state'] != 'valid')
		{
			return false;
		}
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'convessParticipantGroups');
	}
	
	/**
	 * Determines if the viewing user can auto respond to conversations.
	 *
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */	
	public function canAutoRespond(array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (!$viewingUser['user_id'])
		{
			return false;
		}

		if ($viewingUser['user_state'] != 'valid')
		{
			return false;
		}
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'convessAutoResponse');
	}
	
	/**
	 * Determines if the viewing user can create conversation prefixes.
	 *
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */	
	public function canCreatePrefixes(array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'])
		{
			return false;
		}

		if ($viewingUser['user_state'] != 'valid')
		{
			return false;
		}
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'convessPrefixes');
	}
	
	/**
	 * Determines if the viewing user can kick recipients from conversations.
	 *
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */	
	public function canKickRecipients(array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (!$viewingUser['user_id'])
		{
			return false;
		}

		if ($viewingUser['user_state'] != 'valid')
		{
			return false;
		}
		
		return XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'convessKickRecipients');
	}
	
	/**
	 * Processes the participant group data.
	 * If any users no longer exists, update the db convess_group table accordingly
	 *
	 * @param array $group
	 * @param integer $maxRecipients, -1 = unlimited
	 *
	 * @return array
	 */	
	public function processParticipantGroupData(array $group, $maxRecipients = -1)
	{
		if (empty($group))
			return $group;

		$groupUsers = $this->getParticipantGroupUsers($group['group_id'], 'user_id');
		$userIds = array_keys($groupUsers);
		
		$userModel = $this->_getUserModel();
		$users = $userModel->getUsersByIds($userIds);
		$foundUsers = array();
			
		foreach ($users AS $user)
		{
			$foundUsers[$user['user_id']] = $user['username'];
			
			// has the user name changed?
			if (!array_key_exists($user['username'], $groupUsers))
			{
				// update the group with the new user name
				$this->updateGroupUserNameById($group['group_id'], $user['user_id'], $user['username']);
			}
		}
		
		$removeUsers = array();
		$removeUsers = array_diff_key($groupUsers, $foundUsers);
		
		$recipientCount = count($foundUsers);
		
		// update group with correct list that excludes deleted users
		if (!empty($removeUsers))
			$this->removeParticipantsFromGroup($group['group_id'], array_keys($removeUsers));
		
		$group['group_users'] = implode(', ', $foundUsers);
			

		if ($maxRecipients == -1)
		{
			$remaining = -1;
		}
		else
		{	
			if ($recipientCount)
			{
				$remaining = $maxRecipients - $recipientCount;
				$remaining = max(0, $remaining);
			}
			else
			{
				$remaining = $maxRecipients;
			}
		}
		
		$group['remaining'] = $remaining;
		return $group;
	}
	
	/**
	 * Add a participant group for this user
	 *
	 * @param string $groupName
	 * @param array $usernames
	 * @param array|null $viewingUser
	 *
	 * @return group_id
	 */		
	public function addParticipantGroup($groupName, array $usernames, array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$groupName)
			throw new XenForo_Exception('Function addParticipantGroup() - no group name');
			
		if (empty($usernames))
			throw new XenForo_Exception('Function addParticipantGroup() - no usernames');
			
		$viewingUserId = $viewingUser['user_id'];
		
		// new group
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroup');
		$vars = array(
			'user_id' => $viewingUserId,
			'group_name' => $groupName

		);	
		
		$dw->bulkSet($vars);
		$dw->preSave();

		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			throw new XenForo_Exception($errorKey);
		}

		$dw->save();
		
		$newGroup = $dw->getMergedData();
		$groupId = $newGroup['group_id'];

		$db = $this->_getDb();
		XenForo_Db::beginTransaction($db);
				
		// insert new names		
		foreach ($usernames AS $userId => $username)
		{
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroupUser');
			$vars = array(
				'group_id' => $groupId,
				'user_id' => $userId,
				'username' => $username
			);
			$dw->bulkSet($vars);
			$dw->preSave();
	
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
	
			$dw->save();
		}
		
		XenForo_Db::commit($db);
		return $groupId;
	}
	
	/**
	 * Update a group by removing deleted users
	 *
	 * @param integer $groupId
	 * @param array $removeUserIds
	 */	
	protected function removeParticipantsFromGroup($groupId, array $removeUserIds)
	{
		if (!$groupId)
			throw new XenForo_Exception('Function removeParticipantsFromGroup() - no group id');
			
		if (empty($removeUserIds))
			throw new XenForo_Exception('Function removeParticipantsFromGroup() - no usernames');

		$db = $this->_getDb();
		XenForo_Db::beginTransaction($db);
		
		foreach ($removeUserIds AS $userId)
		{
			$exists = $this->getParticipantGroupUserById($groupId, $userId);
			if (!empty($exists))
			{
				$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroupUser');
				$dw->setExistingData(array('group_id' => $groupId, 'user_id' => $userId));
				$dw->preDelete();
		
				if ($dw->hasErrors())
				{
					$errors = $dw->getErrors();
					$errorKey = reset($errors);
					throw new XenForo_Exception($errorKey);
				}
		
				$dw->delete();
			}
		}
		
		XenForo_Db::commit($db);
	}
	
	/**
	 * Update a group with new users
	 *
	 * @param integer $groupId
	 * @param string $groupName
	 * @param array $usernames
	 */		
	public function updateParticipantGroup($groupId, $groupName, array $usernames)
	{
		if (!$groupId)
			throw new XenForo_Exception('Function updateParticipantGroup() - no group id');
			
		if (!$groupName)
			throw new XenForo_Exception('Function updateParticipantGroup() - no group name');
			
		if (empty($usernames))
			throw new XenForo_Exception('Function updateParticipantGroup() - no usernames');
			
		// find which users are to be removed
		$groupUsers = $this->getParticipantGroupUsers($groupId);
		$currentUsers = array_keys($groupUsers);
		$removeUsers = array_diff($currentUsers, $usernames);
		
		$db = $this->_getDb();
		XenForo_Db::beginTransaction($db);

		// remove names
		if (!empty($removeUsers))
		{
			foreach ($removeUsers AS $username)
			{
				$removeUserId = $groupUsers[$username]['user_id'];
				
				$exists = $this->getParticipantGroupUserById($groupId, $removeUserId);
				if (!empty($exists))
				{
					$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroupUser');
					$dw->setExistingData(array('group_id' => $groupId, 'user_id' => $removeUserId));
					$dw->preDelete();
			
					if ($dw->hasErrors())
					{
						$errors = $dw->getErrors();
						$errorKey = reset($errors);
						throw new XenForo_Exception($errorKey);
					}
			
					$dw->delete();
				}
			}
		}		

		// find which users are new
		$newUsers = array_diff($usernames, $removeUsers);
		$newUsers = array_diff($newUsers, $currentUsers);
								
		// insert new names	
		if (!empty($newUsers))
		{
			foreach ($newUsers AS $userId => $username)
			{
				$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroupUser');
				$vars = array(
					'group_id' => $groupId,
					'user_id' => $userId,
					'username' => $username
				);
				$dw->bulkSet($vars);
				$dw->preSave();

				if ($dw->hasErrors())
				{
					$errors = $dw->getErrors();
					$errorKey = reset($errors);
					throw new XenForo_Exception($errorKey);
				}
		
				$dw->save();
			}
		}

		// group name
		$exists = $this->getParticipantGroupById($groupId);
		if (empty($exists))
		{
			throw new XenForo_Exception('updateParticipantGroup(): Group item not found');
		}
		
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroup');
		$dw->setExistingData(array('group_id' => $groupId));
		$dw->set('group_name', $groupName);
		$dw->preSave();

		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			throw new XenForo_Exception($errorKey);
		}

		$dw->save();
		
		XenForo_Db::commit($db);
	}
	
	/**
	 * Delete a participant group
	 *
	 * @param integer $groupId
	 */		
	function deleteGroupById($groupId)
	{
		$exists = $this->getParticipantGroupById($groupId);
		if (empty($exists))
		{
			throw new XenForo_Exception('deleteGroupById(): Group item not found');
		}
		
		// group
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroup');
		$dw->setExistingData(array('group_id' => $groupId));
		$dw->preDelete();
		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			throw new XenForo_Exception($errorKey);
		}

		$dw->delete();
	}
	
	/**
	 * Update the name of a participant group user (username has changed)
	 *
	 * @param integer $groupId
	 * @param integer $userId
	 * @param string $newName - new username
	 */		
	function updateGroupUserNameById($groupId, $userId, $newName)
	{
		$exists = $this->getParticipantGroupUserById($groupId, $userId);
		if (!empty($exists))
		{
			$newName = trim($newName);
			
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroupUser');
			$dw->setExistingData(array('group_id' => $groupId, 'user_id' => $userId));
			$dw->set('username', $newName);
			$dw->preSave();
	
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
	
			$dw->save();
		}
	}
	
	/**
	 * Prepares a set of conditions against which to select conversations.
	 *
	 * @param array $conditions List of conditions.
	 * --popupMode (boolean) constrains results to unread, or sent within timeframe specified by options->conversationPopupExpiryHours
	 * @param array $fetchOptions The fetch options that have been provided. May be edited if criteria requires.
	 *
	 * @return string Criteria as SQL for where clause
	 */
	public function prepareConversationConditions(array $conditions, array &$fetchOptions)
	{
		if (empty($fetchOptions['skip_convess']))
		{
			if (!empty($conditions['prefix_id']) && $this->canCreatePrefixes(XenForo_Visitor::getInstance()->toArray()))
			{
				$sqlConditions = parent::prepareConversationConditions($conditions, $fetchOptions);
				
				$sqlConditions .= ' AND convess_prefix.prefix_id = ' . $conditions['prefix_id'];
				
				return $sqlConditions;
			}
		}
		
		return parent::prepareConversationConditions($conditions, $fetchOptions);
	}
	
	public function prepareConversationFetchOptions(array $fetchOptions)
	{
		$skipConvess = empty($fetchOptions['skip_convess']) ? false : true;
		$fetchUserPermissions = isset($fetchOptions['join_user_permissions']) ? true : false;
		$fetchIgnoreUserId = isset($fetchOptions['ignoringUserId']) ? intval($fetchOptions['ignoringUserId']) : 0;
		$fetchFollowUserId = isset($fetchOptions['followingUserId']) ? intval($fetchOptions['followingUserId']) : 0;
		
		if (!$skipConvess || $fetchUserPermissions || $fetchIgnoreUserId || $fetchFollowUserId)
		{
			$fetchOptions = parent::prepareConversationFetchOptions($fetchOptions);
			
			if (!$skipConvess)
			{
				$fetchPrefixes = $this->canCreatePrefixes(XenForo_Visitor::getInstance()->toArray());
				$fetchStickies = XenForo_Application::get('options')->convess_sticky;
				
				if ($fetchPrefixes || $fetchStickies)
				{
					$fetchOptions['joinTables'] .= '
								LEFT JOIN convess AS convess ON
									(convess.conversation_id = conversation_master.conversation_id AND convess.user_id = conversation_user.owner_user_id )';
					
					// fetch the prefix and sticky state
					if ($fetchStickies)
					{
						$fetchOptions['selectFields'] .= ',
									IF(convess.sticky IS NULL, 0, convess.sticky) AS sticky';
					}
					
					if ($fetchPrefixes)
					{
						$fetchOptions['selectFields'] .= ',
									IF(convess_prefix.prefix_id IS NULL, 0, convess_prefix.prefix_id) AS prefix_id,
									IF(convess_prefix.title IS NULL, "", convess_prefix.title) AS prefix_title,
									IF(convess_prefix.css_class IS NULL, "", convess_prefix.css_class) AS prefix_css';
						$fetchOptions['joinTables'] .= '
									LEFT JOIN convess_prefix AS convess_prefix ON
										(convess_prefix.prefix_id = convess.prefix_id)';
					}
				}
			}
			
			if ($fetchUserPermissions)
			{
				$fetchOptions['selectFields'] .= ',
					permission_combination.cache_value AS global_permission_cache';
				$fetchOptions['joinTables'] .= '
					LEFT JOIN xf_permission_combination AS permission_combination ON
						(permission_combination.permission_combination_id = user.permission_combination_id)';
			}
			
			if ($fetchIgnoreUserId)
			{
				$fetchOptions['selectFields'] .= ',
					IF(user_ignored.user_id IS NOT NULL, 1, 0) AS ignoring_' . $fetchIgnoreUserId;
				$fetchOptions['joinTables'] .= '
					LEFT JOIN xf_user_ignored AS user_ignored ON
						(user_ignored.user_id = user.user_id AND user_ignored.ignored_user_id = ' . $fetchIgnoreUserId . ')';
			}
			
			if ($fetchFollowUserId)
			{
				$fetchOptions['selectFields'] .= ',
					IF(user_follow.user_id IS NOT NULL, 1, 0) AS following_' . $fetchFollowUserId;
				$fetchOptions['joinTables'] .= '
					LEFT JOIN xf_user_follow AS user_follow ON
						(user_follow.user_id = user.user_id AND user_follow.follow_user_id = ' . $fetchFollowUserId . ')';
			}
			
			return $fetchOptions;
		}
		
		return parent::prepareConversationFetchOptions($fetchOptions);
	}
	
	/**
	 * Get conversations that a user can see, ordered by the latest message first.
	 *
	 * @param integer $userId
	 * @param array $conditions Conditions for the WHERE clause
	 * @param array $fetchOptions Options for extra data to fetch
	 *
	 * @return array Format: [conversation id] => info
	 */
	public function getConversationsForUser($userId, array $conditions = array(), array $fetchOptions = array())
	{
		// MUST rewrite parent in order to have stickies in ORDER BY clause
		// A fix was requested: http://xenforo.com/community/threads/conversation-php.28931/
		if (XenForo_Application::get('options')->convess_sticky)
		{
			$whereClause = $this->prepareConversationConditions($conditions, $fetchOptions);
			$sqlClauses = $this->prepareConversationFetchOptions($fetchOptions);
			$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
			
			$sql = $this->limitQueryResults(
				'
					SELECT conversation_master.*,
						conversation_user.*,
						conversation_starter.*,
						conversation_master.username AS username,
						conversation_recipient.recipient_state, conversation_recipient.last_read_date
						' . $sqlClauses['selectFields'] . '
					FROM xf_conversation_user AS conversation_user
					INNER JOIN xf_conversation_master AS conversation_master ON
						(conversation_user.conversation_id = conversation_master.conversation_id)
					INNER JOIN xf_conversation_recipient AS conversation_recipient ON
						(conversation_user.conversation_id = conversation_recipient.conversation_id
						AND conversation_user.owner_user_id = conversation_recipient.user_id)
					LEFT JOIN xf_user AS conversation_starter ON
						(conversation_starter.user_id = conversation_master.user_id)
						' . $sqlClauses['joinTables'] . '
					WHERE conversation_user.owner_user_id = ?
						AND ' . $whereClause . '
					ORDER BY sticky DESC, conversation_user.last_message_date DESC
				', $limitOptions['limit'], $limitOptions['offset']
			);
			
			return $this->fetchAllKeyed($sql, 'conversation_id', $userId);
		}
		else
		{
			return parent::getConversationsForUser($userId, $conditions, $fetchOptions);
		}
	}
	
	/**
	 * Gets information about all recipients of a conversation.
	 *
	 * @param integer $conversationId
	 * @param array $fetchOptions Options for extra data to fetch
	 *
	 * @return array Format: [user id] => info
	 */
	public function getConversationRecipients($conversationId, array $fetchOptions = array())
	{
		$fetchOptions['skip_convess'] = 1;
		return parent::getConversationRecipients($conversationId, $fetchOptions);
	}
	
	/**
	 * Gets info about a single recipient of a conversation.
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 * @param array $fetchOptions Options for extra data to fetch
	 *
	 * @return array|false
	 */
	public function getConversationRecipient($conversationId, $userId, array $fetchOptions = array())
	{
		$fetchOptions['skip_convess'] = 1;
		return parent::getConversationRecipient($conversationId, $userId, $fetchOptions);
	}
	
	/**
	 * Set/update a conversation's prefix an/or sticky
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 * @param integer $prefixId
	 */
	public function setConversationStickyAndOrPrefix($conversationId, $userId, $stickyState = null, $prefixId = null)
	{
		if (isset($stickyState))
			$stickyState = intval($stickyState);
		else
			$stickyState = -1;
		
		if (isset($prefixId))
			$prefixId = intval($prefixId);
		else
			$prefixId = -1;
		
		$exists = $this->getConvessByConversationUserId($conversationId, $userId);
		if ((empty($exists) && ($stickyState || $prefixId)) || (!empty($exists) && ($exists['prefix_id'] != $prefixId || $exists['sticky'] != $stickyState)))
		{
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
			if (!empty($exists))
			{
				$setVars = array(
					'conversation_id' => $conversationId,
					'user_id' => $userId
				);
				$dw->setExistingData($setVars);
			}
			else
			{
				$vars['conversation_id'] = intval($conversationId);
				$vars['user_id'] = intval($userId);
			}
			
			if ($prefixId > -1)
				$vars['prefix_id'] = $prefixId;
			if ($stickyState > -1)
				$vars['sticky'] = $stickyState;
			
			$dw->bulkSet($vars);
			$dw->preSave();
			
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->save();
		}
	}
	
	/**
	 * Delets a conversation record for a specific user. If all users have deleted the conversation,
	 * it will be completely removed.
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 * @param string $deleteType Type of deletion (either delete, or delete_ignore)
	 */
	public function deleteConversationForUser($conversationId, $userId, $deleteType)
	{
		// fetch the messages ids in case they get deleted if the whole conversation is deleted
		$messageIds = $this->getConversationMessageIds($conversationId);
		
		parent::deleteConversationForUser($conversationId, $userId, $deleteType);

		$exists = $this->getConvessByConversationUserId($conversationId, $userId);
		if (!empty($exists))
		{
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
			$dw->setExistingData(array('conversation_id' => $conversationId, 'user_id' => $userId));
			
			// need to keep last date or kicked state
			if (XenForo_Application::get('options')->convess_last_read_date || $dw->get('kicked') > 0)
			{
				// reset values but keep the last_read_message_date, therefor we do not delete it
				$vars['sticky'] = 0;
				$vars['prefix_id'] = 0;
					
				$dw->bulkSet($vars);
				$dw->preSave();
		
				if ($dw->hasErrors())
				{
					$errors = $dw->getErrors();
					$errorKey = reset($errors);
					throw new XenForo_Exception($errorKey);
				}
		
				$dw->save();
			}
			else // delete
			{
				$dw->preDelete();
		
				if ($dw->hasErrors())
				{
					$errors = $dw->getErrors();
					$errorKey = reset($errors);
					throw new XenForo_Exception($errorKey);
				}
		
				$dw->delete();				
			}
		}
		
		$conversation = $this->getConversationMasterById($conversationId);
		$db = $this->_getDb();
		
		if (empty($conversation))
		{
			while (count($messageIds))
			{
				$deleteIds = array_slice($messageIds, 0, 500); // process 500 at a time
				
				// all alerts
				$db->delete('xf_user_alert', 'content_type = \'convess_message\' AND content_id IN (' . $db->quote($deleteIds) . ')');
				
				// all likes
				$db->delete('xf_liked_content', 'content_type = \'convess_message\' AND content_id IN (' . $db->quote($deleteIds) . ')');
				
				// all like totals
				$db->delete('convess_message_likes', 'message_id IN (' . $db->quote($deleteIds) . ')');
				
				// remove 500 from the $messageIds array
				$messageIds = array_splice($messageIds, 500);
			}

		}
		else
		{
			// only need the ones this user created
			foreach ($messageIds AS $id => $message)
			{
				if ($message['user_id'] != $userId)
					unset($messageIds[$id]);
			}
			
			while (count($messageIds))
			{
				$deleteIds = array_slice($messageIds, 0, 500); // process 500 at a time
				
				// alerts for this user only
				$db->delete('xf_user_alert',
					'content_type = \'convess_message\' AND content_id IN (' . $db->quote($deleteIds) . ') AND alerted_user_id = ' . $db->quote($userId)
				);
				
				// remove 500 from the $messageIds array
				$messageIds = array_splice($messageIds, 500);
			}
		}
	}
	
	/**
	 * Delete a user's auto response
	 *
	 * @param integer $userId
	 * @param boolean $exists - whether the existance of the column has already been established
	 */
	public function autoResponseDelete($userId, $exists = false)
	{
		$userId = intval($userId);
		
		if (!$exists)
			$exists = $this->getAutoResponse($userId);
		
		if (!empty($exists))
		{
			// convess_auto_response
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_AutoResponse');
			$dw->setExistingData(array('user_id' => $userId));
			$dw->preDelete();
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->delete();
		}	
	}
	
	/**
	 * Delete all ConvEss tables that belong to a specific user
	 *
	 * @param integer $userId
	 */
	public function deleteAllConvEssByUserId($userId)
	{
		$userId = intval($userId);
		
		// convess_auto_response
		$exists = $this->getAutoResponse($userId);
		if (!empty($exists))
		{
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_AutoResponse');
			$dw->setExistingData(array('user_id' => $userId));
			$dw->preDelete();
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->delete();
		}

		// convess
		$conversations = $this->getConvessByUserId($userId);
		foreach ($conversations AS $conversationId => $conversation)
		{
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
			$dw->setExistingData(array('conversation_id' => $conversationId, 'user_id' => $userId));
			$dw->preDelete();
			
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->delete();
		}						
		
		// convess_prefix
		$prefixes = $this->getPrefixes($userId);
		foreach ($prefixes AS $prefix)
		{
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_Prefix');
			$dw->setExistingData(array('prefix_id' => $prefix['prefix_id']));
			$dw->preDelete();
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->delete();
		}
		
		// convess_group_user
		$db = $this->_getDb();
		$groups = $this->getParticipantGroups($userId);
		foreach ($groups AS $group)
		{
			$quotedGroupId = $db->quote($group['group_id']);
			$db->delete('convess_group_user', 'group_id = ' . $quotedGroupId);
		}
		
		// convess_group
		$groups = $this->getParticipantGroups($userId);
		foreach ($groups AS $group)
		{
			// group
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ParticipantGroup');
			$dw->setExistingData(array('group_id' => $group['group_id']));
			$dw->preDelete();
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->delete();
		}
		
		// convess_user_options
		$privacy = $this->getConvEssUserOptionsByUserId($userId);
		if (!empty($privacy))
		{
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_UserOptions');
			$dw->setExistingData(array('user_id' => $userId));
			$dw->preDelete();
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->delete();
		}
		
		// xf_user_alert_optout
		$db->delete('xf_user_alert_optout', 'user_id = ' . $userId . ' AND alert = ' . $db->quote('convess_message_like'));
	}
	
	/**
	 * Add a prefix
	 *
	 * @param string $title
	 * @param integer $display_order
	 * @param string $css_class
	 * @param array|null $viewingUser
	 *
	 * @return group_id
	 */		
	public function addPrefix($title, $display_order, $css_class, array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$title)
			throw new XenForo_Exception('Function addPrefix() - no title');
			
		if (empty($css_class))
			throw new XenForo_Exception('Function addPrefix() - no css_class');

		$vars = array(
			'user_id' => $viewingUser['user_id'],
			'title' => $title,
			'display_order' => $display_order,
			'css_class' => $css_class

		);			
		
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_Prefix');
		$dw->bulkSet($vars);
		$dw->preSave();

		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			throw new XenForo_Exception($errorKey);
		}

		$dw->save();
		$newPrefix = $dw->getMergedData();
		
		return $newPrefix['prefix_id'];
	}
	
	/**
	 * Update a prefix
	 *
	 * @param integer $prefixId
	 * @param string $title
	 * @param integer $display_order
	 * @param string $css_class
	 *
	 */		
	public function updatePrefix($prefixId, $title, $display_order, $css_class)
	{
		if (!$prefixId)
			throw new XenForo_Exception('Function updatePrefix() - no prefixId');
			
		if (!$title)
			throw new XenForo_Exception('Function updatePrefix() - no title');
			
		if (empty($css_class))
			throw new XenForo_Exception('Function updatePrefix() - no css_class');

		$exists = $this->getPrefixById($prefixId);
		if (empty($exists))
		{
			throw new XenForo_Exception('updatePrefix(): prefix item not found');
		}
		
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_Prefix');
		$dw->setExistingData(array('prefix_id' => $prefixId));

		$vars = array(
			'title' => $title,
			'display_order' => $display_order,
			'css_class' => $css_class

		);
		
		$dw->bulkSet($vars);
		$dw->preSave();

		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			throw new XenForo_Exception($errorKey);
		}

		$dw->save();
	}
	
	/**
	 * Update a prefix's order
	 *
	 * @param integer $prefixId
	 * @param integer $display_order
	 *
	 */		
	public function setPrefixOrder($prefixId, $display_order)
	{
		if (!$prefixId)
			throw new XenForo_Exception('Function updatePrefix() - no prefixId');

		$exists = $this->getPrefixById($prefixId);
		if (empty($exists))
		{
			throw new XenForo_Exception('updatePrefix(): prefix item not found');
		}
		
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_Prefix');
		$dw->setExistingData(array('prefix_id' => $prefixId));
		
		$dw->set('display_order', $display_order);
		$dw->preSave();

		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			throw new XenForo_Exception($errorKey);
		}

		$dw->save();
	}
	
	/**
	 * Add a kicked user
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 *
	 * return boolean
	 */		
	public function addKicked($conversationId, $userId)
	{
		$exists = $this->isUserKickedFromConversation($conversationId, $userId);
		
		if ($exists === false) // insert new kick
		{
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
			$dw->set('conversation_id', $conversationId);
			$dw->set('user_id', $userId);
			$dw->set('kicked', 1);
			$dw->preSave();
			
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->save();
			return true;
		}
		else 
		{
			if ($exists) // already kicked
				return false;
			
			// update state
			$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
			$dw->setExistingData(array('conversation_id' => $conversationId, 'user_id' => $userId));
			$dw->set('kicked', 1);
			$dw->preSave();
			
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
			
			$dw->save();
			return true;
		}
	}
	
	/**
	 * Insert a new conversation recipient record.
	 *
	 * @param array $conversation Conversation info
	 * @param integer $user User to insert for
	 * @param array $existingRecipient Information about the existing recipient record (if there is one)
	 * @param string $insertState State to insert the conversation for with this user
	 *
	 * @return boolean True if an insert was required (may be false if user is already an active recipient or is ignoring)
	 */
	public function insertConversationRecipient(array $conversation, $userId, array $existingRecipient = null, $insertState = 'active')
	{
		$conversationId = $conversation['conversation_id'];
		if ($existingRecipient === null)
		{
			$existingRecipient = $this->getConversationRecipient($conversationId, $userId);
		}
		
		// check to see if a deleted_ignored user was kicked and if so allow to be invited back
		if (!empty($existingRecipient) && $existingRecipient['recipient_state'] == 'deleted_ignored')
		{

			$kicked = $this->getKickedByConversationId($conversationId);
			if (!empty($kicked))
			{
				// re-insert a kicked user
				$kickedUsers = array_keys($kicked);
				if (in_array($existingRecipient['user_id'], $kickedUsers))
				{
					$db = $this->_getDb();

					XenForo_Db::beginTransaction($db);
					
					$db->query('
						UPDATE xf_conversation_recipient
						SET	recipient_state = \'active\'
						WHERE conversation_id = ? AND user_id = ?
					', array($conversationId, $userId));
					
					$db->query('
						INSERT IGNORE INTO xf_conversation_user
							(conversation_id, owner_user_id, is_unread, reply_count,
							last_message_date, last_message_id, last_message_user_id, last_message_username)
						VALUES
							(?, ?, 1, ?,
							?, ?, ?, ?)
					', array(
						$conversationId, $userId, $conversation['reply_count'],
						$conversation['last_message_date'], $conversation['last_message_id'],
						$conversation['last_message_user_id'], $conversation['last_message_username']
					));
					
					$this->rebuildUnreadConversationCountForUser($userId);
					
					// remove kicked status
					$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
					$dw->setExistingData(array('conversation_id' => $conversationId, 'user_id' => $userId));
					$dw->set('kicked', 0);
					$dw->preSave();
			
					if ($dw->hasErrors())
					{
						$errors = $dw->getErrors();
						$errorKey = reset($errors);
						return false;
					}
			
					$dw->save();
					
					XenForo_Db::commit($db);
					return true;
				}
			}
		}
		
		if (empty($existingRecipient))
			$existingRecipient = array();
		
		return parent::insertConversationRecipient($conversation, $userId, $existingRecipient, $insertState);
	}
	
	/**
	 * Calculates the allowed number of additional conversation receiptions the
	 * viewing user can add to the given conversation.
	 * @param array $conversation Conversation; if empty array, assumes new conversation
	 * @param array|null $viewingUser
	 *
	 * @return integer -1 means unlimited; 0 is no more invites; other is remaining count
	 */
	public function allowedAdditionalConversationRecipients(array $conversation, array $viewingUser = null)
	{
		$remaining = parent::allowedAdditionalConversationRecipients($conversation, $viewingUser);
		
		if ($this->canKickRecipients())
		{
			// add our kicked users to the count so that they may be re-invited. But only if not unlimited invites and not new conversation
			// we check for re-invitations rather than invitations once names are selected in actionInviteInsert()
			if ($remaining > -1 && !empty($conversation))
			{
				$kicked = $this->countKickedFromConversation($conversation['conversation_id']);
				$remaining += $kicked;
			}
		}
			
		return $remaining;
	}
	
	/**
	 * Update a user's last read date privacy
	 *
	 * @param integer $conversationId
	 * @param array $recipients by reference
	 *
	 * return integer $numLeftPermanently - number of users who have left the conversation permanently
	 */		
	public function processLastReadDate($conversationId, array &$recipients)
	{
		$conversationId = intval($conversationId);
		$numLeftPermanently = 0;
		
		// can't use the $fetchOptions array to include the convess table as function actionView()
		// does not pass it through to getConversationRecipients()
		$participants = $this->getConvessByConversationId($conversationId);
		$options = XenForo_Application::get('options');
		$visitor = XenForo_Visitor::getInstance();
		$canBypassUserPrivacy = false;
		$bypassPrivacy = $options->convess_last_read_date_bypass_privacy;
		
		if ($options->convess_last_read_date_privacy)
		{
			$convessUserOptions = $this->getConvEssUserOptionsyByUserIds(array_keys($recipients));
			
			if ($bypassPrivacy != 'no')
			{
				$canBypassUserPrivacy = $this->_getUserModel()->canBypassUserPrivacy();
			}
		}
		
		foreach ($recipients AS $recipientId => &$recipient)
		{
			// left permanently or a deleted user
			if ($recipient['recipient_state'] == 'deleted_ignored' || !$recipient['user_id'])
				$numLeftPermanently++;
			
			// kicked status
			if (isset($participants[$recipientId]))
				$recipient['kicked'] = $participants[$recipientId]['kicked'];
			else
				$recipient['kicked'] = 0; // does not yet have a column in the convess table
			
			if ($options->convess_last_read_date)
			{
				$lastReadDateViewLimit = $options->convess_last_read_date_view_limit;
				$canViewLastReadDate = false;
				if ($lastReadDateViewLimit == 'no_limit' || ($lastReadDateViewLimit == 'admins' && $visitor['is_admin']) || ($lastReadDateViewLimit == 'admins_mods' && ($visitor['is_moderator'] || $visitor['is_admin'])))
				{
					$canViewLastReadDate = true;
				}
				
				$recipient['last_read_message_date'] = -1;
				
				if ($canViewLastReadDate && isset($recipient['user_id']))
				{
					$privacy = false;
					if ($options->convess_last_read_date_privacy)
					{
						if ($lastReadDateViewLimit == 'no_limit' || ($lastReadDateViewLimit == 'admins' && $recipient['is_admin']) || ($lastReadDateViewLimit == 'admins_mods' && ($recipient['is_moderator'] || $recipient['is_admin'])))
						{
							if (isset($convessUserOptions[$recipientId]))
							{
								if (!$convessUserOptions[$recipientId]['allow_last_read_date'])
								{
									$privacy = true;
								
									if (($recipient['user_id'] != $visitor['user_id']) && $bypassPrivacy != 'no' && $canBypassUserPrivacy)
									{
										if ($bypassPrivacy == 'yes' || ($bypassPrivacy == 'yes_skip_admins' && !$recipient['is_admin']) || ($bypassPrivacy == 'yes_skip_admins_mods' && !$recipient['is_moderator'] && !$recipient['is_admin']))
										{
											$privacy = false;
										}
									}
								}
							}
						}
					}
					
					if (!$privacy)
					{
						$userId = $recipient['user_id'];
					
						// last read date
						if (isset($participants[$userId]))
							$participant = $participants[$userId];
						else
							$participant = array(); // does not yet have a column in the convess table
						
						if (!empty($participant) && $participant['last_read_message_date'] > 0)
						{
							$recipient['last_read_message_id'] = $participant['last_read_message_id'];
							$recipient['last_read_message_date'] = $participant['last_read_message_date']; 
						}
						else
						{
							$time = $recipient['last_read_date'];
							
							// conversation was read at some point
							if ($time > 0)
							{
								$dw = XenForo_DataWriter::create('ConvEss_DataWriter_ConvEss');
								$vars = array();
								
								// pre version 1.0.8 conversation which does not have last_read_message_date for this user
								// so insert or create it for next time
								if (!empty($participant))
								{
									$setVars = array(
										'conversation_id' => $conversationId,
										'user_id' => $userId
									);
									$dw->setExistingData($setVars);
								}
								else
								{
									$vars['conversation_id'] = $conversationId;
									$vars['user_id'] = $userId;
								}
								
								// last read date
								$vars['last_read_message_date'] = $time;
								
								// point to last message they read
								$messageId = $this->getMessageIdByDate($conversationId, $time);
								$vars['last_read_message_id'] = $messageId;
								
								$dw->bulkSet($vars);
								$dw->save();
								
								$recipient['last_read_message_id'] = $messageId;
								$recipient['last_read_message_date'] = $time;
							}
							else
							{
								$recipient['last_read_message_date'] = 0;
							}
						}
					}
				}
			}
		}
		
		return $numLeftPermanently;
	}
	
	/**
	 * Update a user's last read date privacy
	 *
	 * @param integer $userId
	 * @param integer $privacy : 0 = private, 1 = not private
	 *
	 */		
	public function setLastReadDatePrivacy($userId, $privacy)
	{
		$userId = intval($userId);
		if (!$userId)
			throw new XenForo_Exception('Function setLastReadDatePrivacy() - no userId');
		
		$dw = XenForo_DataWriter::create('ConvEss_DataWriter_UserOptions');
		$exists = $this->getConvEssUserOptionsByUserId($userId);
		
		if (!empty($exists))
		{
			$dw->setExistingData($userId);
		}
		else
		{
			$dw->set('user_id', $userId);
		}
		
		$dw->set('allow_last_read_date', $privacy);
		$dw->preSave();

		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			throw new XenForo_Exception($errorKey);
		}

		$dw->save();
	}
	
	/**
	 * Marks the conversation as (completely) unread for a user when kicked.
	 *
	 * @param integer $conversationId
	 * @param integer $userId
	 */
	public function markConversationAsUnreadForKickedUser($conversationId, $userId)
	{
		$db = $this->_getDb();
		$this->_updateConversationReadDate($conversationId, $userId, 0, $db);
	}
	
	/**
	 * Determines if the specified user can reply to the conversation.
	 * Does not check conversation viewing permissions.
	 *
	 * @param array $conversation
	 * @param string $errorPhraseKey Returned phrase key for a specific error
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function canReplyToConversation(array $conversation, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$canReply = parent::canReplyToConversation($conversation, $errorPhraseKey, $viewingUser);

		if (!$canReply && !$conversation['conversation_open'] && XenForo_Application::get('options')->convess_mod_reply_to_locked)
		{
			$this->standardizeViewingUserReference($viewingUser);
			if ($viewingUser['is_admin'] || $viewingUser['is_moderator'])
				$canReply = true;
		}

		return $canReply;
	}
	
	/**
	 * Determines if the specified user can like conversation messages.
	 * Does not check conversation viewing permissions.
	 *
	 * @param integer $contentUserId
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function canLikeMessage($contentUserId, array $viewingUser = null)	
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($viewingUser['user_id'] != $contentUserId)
		{
			if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'conversation', 'convessLikeMessages'))
			{
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Alert a user that someone tried to contact them but couldn't due to a full inbox
	 *
	 * @param integer $toUserId - user ID of person receiving alert
	 * @param integer $fromUserId -  user ID of person sending alert
	 * @param integer $fromUsername -  user name of person sending alert
	 */
	public function sendInboxFullAlert($toUserId, $fromUserId, $fromUsername)
	{
		if ($toUserId && $fromUserId && $fromUsername)
		{
			// has this user already been alerted by this sender in the last 24 hours
			$alertExists = $this->_getDb()->fetchOne('
				SELECT alert_id
				FROM xf_user_alert
				WHERE alerted_user_id = ? AND event_date > ? AND content_type = ? AND action = ? AND user_id = ?
				ORDER BY event_date DESC
				LIMIT 1
			', array($toUserId, (XenForo_Application::$time - 86400), 'convess', 'inbox_full', $fromUserId));
			
			// do not send multiple alerts from same person
			if (!$alertExists)
			{
				XenForo_Model_Alert::alert(
											$toUserId,
											$fromUserId,
											$fromUsername,
											'convess',
											0,
											'inbox_full'
										);
			}
		}
	}
	
	/**
	 * @return XenForo_Model_Like
	 */
	protected function _getLikeModel()
	{
		return $this->getModelFromCache('XenForo_Model_Like');
	}
}