<?php

/**
 * Model for liking content.
 *
 * @package ConvEss
 */
class ConvEss_Model_Like extends XFCP_ConvEss_Model_Like
{	
	/**
	 * Inserts a new like for a piece of content.
	 *
	 * @param string $contentType
	 * @param integer $contentId
	 * @param integer $contentUserId User that owns/created the content
	 * @param integer|null $likeUserId User liking content; defaults to visitor
	 * @param integer|null $likeDate Timestamp of liking; defaults to now.
	 *
	 * @return array|false List of latest like users or false
	 */
	public function likeConversationMessage($contentType, $contentId, $contentUserId, $likeUserId = null, $likeDate = null)
	{
		$visitor = XenForo_Visitor::getInstance();
		
		if ($likeUserId === null)
		{
			$likeUserId = $visitor['user_id'];
		}
		
		if (!$likeUserId || $likeUserId == $contentUserId)
		{
			return false;
		}
		
		if ($likeUserId != $visitor['user_id'])
		{
			$user = $this->getModelFromCache('XenForo_Model_User')->getUserById($likeUserId);
			if (!$user)
			{
				return false;
			}
			$likeUsername = $user['username'];
		}
		else
		{
			$likeUsername = $visitor['username'];
		}
		
		if ($likeDate === null)
		{
			$likeDate = XenForo_Application::$time;
		}
		
		$likeHandler = $this->getLikeHandler($contentType);
		if (!$likeHandler)
		{
			return false;
		}
		
		$db = $this->_getDb();
		XenForo_Db::beginTransaction($db);
		
		$result = $db->query('
			INSERT IGNORE INTO xf_liked_content
				(content_type, content_id, content_user_id, like_user_id, like_date)
			VALUES
				(?, ?, ?, ?, ?)
		', array($contentType, $contentId, $contentUserId, $likeUserId, $likeDate));
		
		if (!$result->rowCount())
		{
			XenForo_Db::commit($db);
			return false;
		}
		
		if ($contentUserId)
		{
			$userModel = $this->getModelFromCache('XenForo_Model_User');
			$contentUser = $this->getModelFromCache('XenForo_Model_User')->getUserById($contentUserId, array(
				'join' => XenForo_Model_User::FETCH_USER_OPTION | XenForo_Model_User::FETCH_USER_PROFILE
			));
			
			if ($contentUser)
			{
				$recipientState = $this->getModelFromCache('XenForo_Model_Conversation')->getRecipientStateFromMessageId($contentId, $contentUserId);
				
				if ($recipientState == 'active') // only send alert if user can view conversation
				{
					if (XenForo_Application::get('options')->convess_like_alert_preference &&
						!$userModel->isUserIgnored($contentUser, $likeUserId)
						&& XenForo_Model_Alert::userReceivesAlert($contentUser, $contentType, 'like')
					)
					{
						XenForo_Model_Alert::alert(
							$contentUserId,
							$likeUserId,
							$likeUsername,
							$contentType,
							$contentId,
							'like'
						);
					}
				}
			}
		}
		
		$latestLikeUsers = $this->getLatestContentLikeUsers($contentType, $contentId);
		$likeHandler->incrementLikeCounter($contentId, $latestLikeUsers);
		
		XenForo_Db::commit($db);
		
		return $latestLikeUsers;
	}
	
	/**
	 * Unlikes the specified like record.
	 *
	 * @param array $like
	 *
	 * @return array|false List of latest like users or false
	 */
	public function unlikeConversationMessage(array $like)
	{
		$likeHandler = $this->getLikeHandler($like['content_type']);
		if (!$likeHandler)
		{
			return false;
		}
		
		$db = $this->_getDb();
		XenForo_Db::beginTransaction($db);
		
		$result = $db->query('
			DELETE FROM xf_liked_content
			WHERE like_id = ?
		', $like['like_id']);
		
		if (!$result->rowCount())
		{
			XenForo_Db::commit($db);
			return false;
		}
		
		if ($like['content_user_id'])
		{
			$this->_getAlertModel()->deleteAlerts(
				$like['content_type'], $like['content_id'], $like['like_user_id'], 'like'
			);
		}
		
		$latestLikeUsers = $this->getLatestContentLikeUsers($like['content_type'], $like['content_id']);
		$likeHandler->incrementLikeCounter($like['content_id'], $latestLikeUsers, -1);

		XenForo_Db::commit($db);

		return $latestLikeUsers;
	}
}