<?php

class Insy_MultipleAccounts_Detector
{
	/** @var int Current IP Address */
	protected $_ipAddress;
	
	/** @var array Addon Configuration */
	protected $_options = array(
		'detection_methods'		=> array(),
		'additional_checks'		=> array(),
		'actions'				=> array(),
		'conversation_sender'	=> 1,
		'conversation_text'		=> '',
		'cookie_lifetime'		=> 600,
	);
	
	/** @var Insy_MultipleAccounts_Model_User */
	protected $_userModel;
	
	/**
	 * Constructor
	 * @param string $ipAddress Current IP Address
	 */
	public function __construct($ipAddress = null)
	{
		$this->_init($ipAddress);
		$this->_loadOptions();
	}
	
	/**
	 * Initialize the addon
	 * @param string $ipAddress
	 */
	protected function _init($ipAddress)
	{
		if ( $ipAddress == null || strlen($ipAddress) == 0 )
		{
			$this->_ipAddress = isset($_SERVER['REMOTE_ADDR']) ?
				$_SERVER['REMOTE_ADDR'] : null;
		}
		else
		{
			$this->_ipAddress = $ipAddress;
		}
	}
	
	/**
	 * Load options
	 */
	protected function _loadOptions()
	{
		/* @var XenForo_Options $options */
		$options = XenForo_Application::getOptions();
		
		$this->_options['enabled'] 				= $options->get('isd_enabled');
		$this->_options['additional_checks'] 	= $options->get('isd_additional_checks');
		$this->_options['actions']			 	= $options->get('isd_actions');
		$this->_options['conversation_sender']	= $options->get('isd_conversation_sender');
		$this->_options['conversation_text']	= $options->get('isd_conversation_text');
		$this->_options['cookie_lifetime']		= $options->get('isd_cookie_lifetime');
		$this->_options['cookie_name']			= $options->get('isd_cookie_name');
		$this->_options['days_ago']				= $options->get('isd_days_ago');
	}

	/**
	 * Determine if it's a multiple account
	 * @return boolean
	 */
	public function isMultipleAccount()
	{
		// No IP Address = No way to check
		if ( $this->_ipAddress == null )
			return false;
		
		// Is the addon enabled?
		if ( ! $this->_options['enabled'] )
			return false;
		
		// Is it a guest?
		if ( XenForo_Visitor::getUserId() == 0 )
			return false;
		
		// Is it a (super) administrator?
		if ( XenForo_Visitor::getInstance()->isSuperAdmin()
			|| XenForo_Visitor::getInstance()->get('is_admin') )
			return false;
		
		// Is it a developer machine?
		if ( $this->_ipAddress == '::1' || $this->_ipAddress == '127.0.0.1' )
			return false;
		
		// Is it excluded from verifications
		if ( XenForo_Visitor::getInstance()->hasPermission('general', 'isp_disable_checks') )
			return false;
		
		// Is it an AJAX request?
		if ( isset($this->_options['additional_checks']['ajax'])
			&& $this->_options['additional_checks']['ajax']
			&& isset($_SERVER['HTTP_X_REQUESTED_WITH']) )
			return false;
		
		return (bool) $this->_checkIp();
	}
	
	/**
	 * @return string
	 */
	public function getUserAgent()
	{
		return isset($_SERVER['HTTP_USER_AGENT']) ?
			$_SERVER['HTTP_USER_AGENT'] : null;
	}
	
	/**
	 * Check with the IP
	 */
	protected function _checkIp()
	{
		$users = $this->_getUserModel()->getUsersByIpAndLogDate(
				$this->_ipAddress,
				array(
					Insy_MultipleAccounts_Model_User::LOG_DATE	=>
						$this->_options['days_ago']['permanent'] ? 0 : intval($this->_options['days_ago']['custom']),
					Insy_MultipleAccounts_Model_User::ALREADY_BANNED =>
						$this->_options['additional_checks']['banned'],
					Insy_MultipleAccounts_Model_User::USER_AGENT =>
						$this->_options['additional_checks']['browser'] ?
							$this->getUserAgent() : false,
				)
		);
		return ($users && count($users) >= 2);
	}
	
	/**
	 * Return a custom model
	 * @return Insy_MultipleAccounts_Model_User
	 */
	protected function _getUserModel()
	{
		if ( ! $this->_userModel instanceof Insy_MultipleAccounts_Model_User )
			$this->_userModel = XenForo_Model::create('Insy_MultipleAccounts_Model_User');
		
		return $this->_userModel;
	}
	
}