<?php

/**
 * Model for Moderator Essentials functions.
 *
 * @package ModEss
 */
class ModEss_Model_ModEss extends XenForo_Model
{
	/**
	 * Determines if a thread deletion placeholder can be viewed with the given permissions.
	 *
	 * @param integer $nodeId
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function viewDeletedThreadPlaceholders($nodeId, array $nodePermissions = null, array $viewingUser = null)
	{
		if (defined('IN_MOBIQUO'))
			return false;
		
		$this->standardizeViewingUserReferenceForNode($nodeId, $viewingUser, $nodePermissions);
		return XenForo_Permission::hasContentPermission($nodePermissions, 'viewThreadPlaceholders');
	}
	
	/**
	 * Determines if a post deletion placeholder can be viewed with the given permissions.
	 *
	 * @param integer $nodeId
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function viewDeletedPostPlaceholders($nodeId, array $nodePermissions = null, array $viewingUser = null)
	{
		if (defined('IN_MOBIQUO'))
			return false;
		
		$this->standardizeViewingUserReferenceForNode($nodeId, $viewingUser, $nodePermissions);
		return XenForo_Permission::hasContentPermission($nodePermissions, 'viewPostPlaceholders');
	}
	
	/**
	 * Determines if a deleted thread/post can be viewed with the given permissions.
	 *
	 * @param integer $nodeId
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function viewDeleted($nodeId, array $nodePermissions = null, array $viewingUser = null)
	{
		if (defined('IN_MOBIQUO'))
			return false;
		
		$this->standardizeViewingUserReferenceForNode($nodeId, $viewingUser, $nodePermissions);
		return XenForo_Permission::hasContentPermission($nodePermissions, 'viewDeleted');
	}
	
	/**
	 * Get a user's modess preferences
	 *
	 * @param integer $userId 
	 *
	 * @return array modess_modlog_preferences
	 */	
	public function getModEssPreferencesByUserId($userId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM modess_modlog_preferences
			WHERE user_id = ?
			LIMIT 1
		', $userId);		
	}
	
	/**
	 * Delete all ModEss tables that belong to a specific user
	 *
	 * @param integer $userId
	 */
	public function deleteAllModEssByUserId($userId)
	{
		$userId = intval($userId);
		
		$this->deleteModEssPreferencesByUserId($userId);
	}
	
	/**
	 * Delete ModEss preferences column of a user
	 *
	 * @param integer $userId
	 */
	public function deleteModEssPreferencesByUserId($userId)
	{
		$userId = intval($userId);
		
		// modess_modlog_preferences
		$exists = $this->getModEssPreferencesByUserId($userId);
		if (!empty($exists))
		{
			$dw = XenForo_DataWriter::create('ModEss_DataWriter_Preferences');
			$dw->setExistingData(array('user_id' => $userId));
			$dw->preDelete();
			if ($dw->hasErrors())
			{
				$errors = $dw->getErrors();
				$errorKey = reset($errors);
				throw new XenForo_Exception($errorKey);
			}
	
			$dw->delete();
		}
	}
	
	/**
	 * Fetch the mod log counts from registry or create them
	 *
	 * @param integer $userId
	 *
	 * @return array/false
	 */
	public function getModLogCounts($userId = null)
	{
		if (XenForo_Application::isRegistered('modLogCounts'))
		{
			$logCounts = XenForo_Application::get('modLogCounts');
			if (isset($userId))
			{
				if (!isset($logCounts[$userId]))
					$logCounts = $this->rebuildSessionModLogCountsCache($userId); // add this mod
			}
		}
		else
		{
			$logCounts = $this->rebuildSessionModLogCountsCache($userId);
		}
		
		if (isset($userId))
		{
			if (isset($logCounts[$userId]))
				return $logCounts[$userId];
			else
				return false;
		}
		
		return $logCounts;
	}
	
	/**
	 * Fetch the active state of the mod log counts for a user
	 *
	 * @param integer $userId
	 *
	 * @return boolean
	 */
	public function isModLogCountsActive($userId)
	{
		$userId = intval($userId);
		
		if ($logCounts = $this->getModLogCounts($userId))
		{
			return $logCounts['isActive'];
		}
		
		return false;
	}
	
	/**
	 * Find out whether the canViewThreadLog permission if available in any forum
	 *
	 * @param array $modId
	 *
	 * @return boolean
	 */
	protected function _canViewThreadLogs($user)
	{
		if (empty($user) || !isset($user['permission_combination_id']))
			return false;
		
		$nodeIds = $this->_getDb()->fetchCol('
				SELECT node_id
				FROM xf_node
		');
		
		$permissionCacheModel = $this->_getPermissionCacheModel();
		
		foreach ($nodeIds AS $nodeId)
		{
			$nodePermissions = $permissionCacheModel->getContentPermissionsForItem($user['permission_combination_id'], 'node', $nodeId);
			$errorPhraseKey= '';
			if ($this->_getThreadModel()->canViewThreadModeratorLog(array('node_id' => $nodeId), array(), $errorPhraseKey, $nodePermissions, $user))
			{
				return true; // found one so no need to keep on checking
			}
		}
		
		return false;
	}
	
	/**
	 * Get the saved session mod log desired action preferences
	 *
	 * @param integer $userId
	 * @param $doIncludeThreadBans - whether or not to include Thread Ban actions
	 * @param $doIncludeResources - whether or not to include Resource actions
	 * @param $doIncludeWarnings - whether or not to include user warnings
	 *
	 * @return array of $actions[key] = array(type, checked, name)
	 */
	public function getModLogActionPreferencesByUserId($userId, $doIncludeThreadBans, $doIncludeResources, $doIncludeWarnings)
	{
		$specificActions = $this->getModLogActionsForTemplate($doIncludeThreadBans, $doIncludeResources, $doIncludeWarnings);
		$preferences = $this->getModEssPreferencesByUserId($userId);
		
		if (!empty($preferences))
		{
			$desiredType = $preferences['action_type'];
			
			if (($desiredType == 'threadban' && !$doIncludeThreadBans) || ($desiredType == 'resource' && !$doIncludeResources) || ($desiredType == 'modess' && !$doIncludeWarnings))
			{
				// type no longer relevant so reset preferences
				$this->deleteModEssPreferencesByUserId($userId);
				
				// no user defined preference so check all actions (user has never saved his own desired actions)
				foreach ($specificActions AS $key => &$value)
				{
					$value['checked'] = 1;
				}
				
				$desiredType = 'all';
			}
			else
			{
				$desiredActions = json_decode($preferences['action_names']);
				
				foreach ($specificActions AS $key => &$value)
				{
					if ($desiredType != 'none')
					{
						$doCheck = false;
						
						switch ($desiredType)
						{
							case 'all':
							{
								$doCheck = true;
								break;
							}
							case 'post':
							{
								if ($value['type'] == 'all' || $value['type'] == 'post' || $value['type'] == 'post_thread')
									$doCheck = true;
								break;
							}
							case 'thread':
							{
								if ($value['type'] == 'all' || $value['type'] == 'thread' || $value['type'] == 'post_thread')
									$doCheck = true;
								break;
							}
							case 'profile_post':
							{
								if ($value['type'] == 'all' || $value['type'] == 'profile_post')
									$doCheck = true;
								break;
							}
							case 'threadban':
							{
								if ($value['type'] == 'threadban')
									$doCheck = true;
								break;
							}
							case 'resource':
							{
								if ($value['type'] == 'all' || $value['type'] == 'resource')
									$doCheck = true;
								break;
							}
							case 'modess':
							{
								if ($value['type'] == 'modess')
									$doCheck = true;
								break;
							}
						}
						
						if ($doCheck)
						{
							if (in_array($key, $desiredActions))
							{
								$value['checked'] = 1;
							}
							
							continue;
						}
					}
					
					$value['disabled'] = 1;
				}
			}
		}
		else
		{
			// no user defined preference so check all actions (user has never saved his own desired actions)
			foreach ($specificActions AS $key => &$value)
			{
				$value['checked'] = 1;
			}
			
			$desiredType = 'all';
		}
		
		$actionPreferences = array(
			'type' => $desiredType,
			'actions' => $specificActions
		);
		
		return $actionPreferences;		
	}
	
	/**
	 * Inserts the specified Session Mod Log search.
	 *
	 * @param array $resultIds List of results, in format [] => id
	 * @param string $searchType The type of the search (usually content type or blank, but could be general string)
	 * @param integer|null $userId User doing search
	 * @param integer|null $searchDate Time of search or null for now
	 * @param array $params search parameters
	 *
	 * @return array Search info, including search_id
	 */
	public function insertModLogSearch(array $resultIds, array $params, $userId = null, $searchDate = null)
	{
		if ($userId === null)
			$userId = XenForo_Visitor::getUserId();
		
		if ($searchDate === null)
			$searchDate = XenForo_Application::$time;
		
		$search = array(
			'search_results' => json_encode($resultIds),
			'result_count' => count($resultIds),
			'user_id' => $userId,
			'search_date' => $searchDate,
			'search_params' => json_encode($params)
		);
		
		$this->_getDb()->insert('modess_modlog_search', $search);
		
		$search['search_id'] = $this->_getDb()->lastInsertId();
		return $search;
	}
	
	/**
	 * Gets the specified Session Mod Log search.
	 *
	 * @param integer $searchId
	 *
	 * @return array|false
	 */
	public function getModLogSearchById($searchId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM modess_modlog_search
			WHERE search_id = ?
		', $searchId);
	}
	
	/**
	 * Update a user's mod log preferences
	 *
	 * @param integer $userId
	 * @param integer $preferences: bitwise actions
	 *
	 */		
	public function setModLogPreferences($userId, array $preferences)
	{
		$userId = intval($userId);
		if (!$userId)
			throw new XenForo_Exception('Function setModLogPreferences() - no userId');
		
		$dw = XenForo_DataWriter::create('ModEss_DataWriter_Preferences');
		$exists = $this->getModEssPreferencesByUserId($userId);
		
		if (!empty($exists))
		{
			$dw->setExistingData($userId);
		}
		else
		{
			$dw->set('user_id', $userId);
		}
		
		// ensure the variables we are getting are valid
		$fields = $dw->getFieldNames();
		unset($fields['user_id']);
		foreach ($preferences AS $Key => $value)
		{
			if (!in_array($Key, $fields))
				unset($preferences[$Key]);
		}
		
		$dw->bulkSet($preferences);
		
		$dw->preSave();

		if ($dw->hasErrors())
		{
			$errors = $dw->getErrors();
			$errorKey = reset($errors);
			throw new XenForo_Exception($errorKey);
		}

		$dw->save();
		
		// adjust the log count to match the new preferences
		$this->rebuildSessionModLogCountsCache($userId);
	}
	
	public function getViewableModLogEntries(array $user, $cutOff)
	{
		if (empty($user))
			return 0;
		
		$cutOff = intval($cutOff);
		
		// if Thread Ban is installed then add those actions
		$addOns = XenForo_Application::get('addOns'); // $this->_getDataRegistryModel()->get('addOns');
		if (isset($addOns['ThreadBan']))
			$doIncludeThreadBans = 1;
		else
			$doIncludeThreadBans = 0;
		
		// if Resource Manager is installed then add those actions
		if (isset($addOns['XenResource']) && XenForo_Permission::hasPermission($user['permissions'], 'resource', 'view'))
			$doIncludeResources = 1;
		else
			$doIncludeResources = 0;
		
		// if user warnings are logged then add those actions
		$doIncludeWarnings = XenForo_Application::get('options')->modess_log_warnings['modlog'] && XenForo_Permission::hasPermission($user['permissions'], 'general', 'viewWarning') ? 1 : 0;
		
		$sessionLogDisabled = false;
		$fetchOptions = array();
		$preferences = $this->getModEssPreferencesByUserId($user['user_id']);
		
		if (!empty($preferences))
		{
			$desiredType = $preferences['action_type'];
			
			if ($desiredType == 'none')
			{
				$sessionLogDisabled = true; // user does not want to be notified of new entries
			}
			else if (($desiredType == 'threadban' && !$doIncludeThreadBans) || ($desiredType == 'resource' && !$doIncludeResources) || ($desiredType == 'modess' && !$doIncludeWarnings))
			{
				// type no longer relevant so reset preferences
				$this->deleteModEssPreferencesByUserId($user['user_id']);
			}
			else
			{
				$fetchActions = array();
				$desiredActions = json_decode($preferences['action_names']);
				
				if (!empty($desiredActions)) // if all were un-checked then skip and fetch all
				{
					$specificActions = $this->getModLogActionsForTemplate($doIncludeThreadBans, $doIncludeResources, $doIncludeWarnings, $desiredType);
					
					if (count($specificActions) != count($desiredActions)) // if all were checked then skip specific actions and fetch all
					{
						foreach ($specificActions AS $key => $value)
						{
							if (in_array($key, $desiredActions))
							{
								if ($value['type'] == 'all')
								{
									if (empty($desiredType) || $desiredType == 'all')
									{
										$fetchActions['post'][] = $key;
										$fetchActions['profile_post'][] = $key;
										$fetchActions['thread'][] = $key;
										$fetchActions['resource'][] = $key;
										$fetchActions['resource_update'][] = $key;
										$fetchActions['modess'][] = $key;
									}
									else if ($desiredType == 'resource')
									{
										$fetchActions['resource'][] = $key;
										$fetchActions['resource_update'][] = $key;
									}
									else
									{
										$fetchActions[$desiredType][] = $key;
									}
								}
								else if ($value['type'] == 'post_thread')
								{
									if (empty($desiredType) || $desiredType == 'all')
									{
										$fetchActions['post'][] = $key;
										$fetchActions['thread'][] = $key;
									}
									else
									{
										$fetchActions[$desiredType][] = $key;
									}
								}
								else if ($value['type'] == 'threadban')
								{
									if ($doIncludeThreadBans)
										$fetchActions['thread'][] = $key;
									else
										continue;
								}
								else if ($value['type'] == 'resource')
								{
									if ($doIncludeResources)
									{
										if ($key == 'resource_edit' || $key == 'resource_update_edit')
										{
											$realKey = 'edit';
										}
										else if ($key == 'resource_reassign')
										{
											$realKey = 'reassign';
										}
										else if ($key == 'resource_feature')
										{
											$realKey = 'feature';
										}
										else if ($key == 'resource_unfeature')
										{
											$realKey = 'unfeature';
										}
										else
										{
											$realKey = $key;
										}
										
										$fetchActions['resource'][] = $realKey;
									}
									else
										continue;
								}
								else if ($value['type'] == 'modess')
								{
									if ($key == 'warn' && $doIncludeWarnings)
									{
										// we count all actions under 'warn'
										$fetchActions['modess'][] = 'warn';
										$fetchActions['modess'][] = 'warn_update';
										$fetchActions['modess'][] = 'warn_delete';
									}
									else
										continue;
								}
								else
								{
									$fetchActions[$value['type']][] = $key;
								}
							}
						}
					}
				}
				
				$skipActions = array();
				
				if ($desiredType != 'all')
				{
					if ($desiredType == 'threadban')
						$fetchOptions['content_type'] = 'thread';
					else
						$fetchOptions['content_type'] = $desiredType;
					
					if ($doIncludeThreadBans)
					{
						if (!isset($fetchActions['thread']))
						{
							if ($desiredType == 'threadban')
							{
								// if no specific action selected, add all threadban actions
								// since threadban uses the thread type we must include those explicitly
								$fetchActions['thread'][] = 'threadban';
								$fetchActions['thread'][] = 'threadban_edit';
								$fetchActions['thread'][] = 'threadbanlift_cron';
								$fetchActions['thread'][] = 'threadbanlift';
							}
							else if ($desiredType == 'thread')
							{
								// since threadban uses the thread type we must exclude those from a thread type search
								$skipActions['include']['thread'][] = 'threadban';
								$skipActions['include']['thread'][] = 'threadban_edit';
								$skipActions['include']['thread'][] = 'threadbanlift_cron';
								$skipActions['include']['thread'][] = 'threadbanlift';
							}
						}
					}
				}
				
				if ($desiredType == 'thread' || $desiredType == 'all')
				{
					// if AVForums_ModCheckpoint is installed then do not fetch the 'posts_checked' entries as they are not needed
					if (isset($addOns['AVForums_ModCheckpoint']))
					{
						$skipActions['include']['thread'][] = 'posts_checked';
						$skipActions['include']['thread'][] = 'bulk_posts_checked';
					}
				}
				
				if (!empty($fetchActions))
					$fetchOptions['actions'] = $fetchActions;
				
				if (!empty($skipActions))
					$fetchOptions['skip_actions'] = $skipActions;
			}
		}
		
		if (!$sessionLogDisabled)
		{
			// skip logs by self
			$fetchOptions['exclude_user_ids'] = array($user['user_id']);
			$maxEntries = XenForo_Application::get('options')->modess_mod_log_max_entries; // limit to maximum items
			return $this->_getLogModel()->getViewableModLogEntries($cutOff, $fetchOptions, $maxEntries);
		}
		else
			return array();
	}
	
	/**
	 * Update the mod log session count.
	 *
	 */
	public function updateSessionModLogCounts($logId, $action, $contentType, array $content, array $logUser = null)
	{
		if (!$logId)
		{
			return false;
		}
		
		$this->standardizeViewingUserReference($logUser);
		
		$mods = $this->getSessionModLogModerators();
		$permissionCacheModel = $this->_getPermissionCacheModel();
		$threadModel = $this->_getThreadModel();
		$userModel = $this->_getUserModel();
		$userProfileModel = $this->_getUserProfileModel();
		
		$logCounts = $this->getModLogCounts();
		
		// check for and add any promoted moderators
		$promotedMods = array_diff_key($mods, $logCounts);
		if (!empty($promotedMods))
		{
			foreach ($promotedMods AS $promoteId => $promote)
			{
				$logCounts = $this->rebuildSessionModLogCountsCache($promoteId); // add this mod, will also demote mods if necessary
			}
		}
		else // above promotions would have also demoted as necessary via rebuildSessionModLogCountsCache()
		{
			// check for and remove any demoted moderators
			$demotedMods = array_diff_key($logCounts, $mods);
			if (!empty($demotedMods))
			{
				foreach ($demotedMods AS $demoteId => $demote)
				{
					unset($logCounts[$demoteId]);
					$this->deleteModEssPreferencesByUserId($demoteId); // remove action preferences
				}
			}
		}
		
		$cache = $logCounts;
		
		foreach ($mods AS $userId => $user)
		{
			// not counting logs by self
			if ($userId != $logUser['user_id'])
			{
				// skip mods that do not have the view thread log in any forum
				if (!empty($logCounts) && isset($logCounts[$userId]) && $logCounts[$userId]['isActive'])
				{
					$canViewNodeThreadLog = false;
					$doCount = false;
					
					// if Thread Ban is installed then add those actions
					$addOns = XenForo_Application::get('addOns'); // $this->_getDataRegistryModel()->get('addOns');
					if (isset($addOns['ThreadBan']))
						$doIncludeThreadBans = 1;
					else
						$doIncludeThreadBans = 0;
					
					// if Resource Manager is installed then add those actions
					if (isset($addOns['XenResource']) && XenForo_Permission::hasPermission($user['permissions'], 'resource', 'view'))
						$doIncludeResources = 1;
					else
						$doIncludeResources = 0;
					
					// if user warnings are logged then add those actions
					$doIncludeWarnings = XenForo_Application::get('options')->modess_log_warnings['modlog'] && XenForo_Permission::hasPermission($user['permissions'], 'general', 'viewWarning') ? 1 : 0;
					
					$preferences = $this->getModEssPreferencesByUserId($userId);
					if (!empty($preferences))
					{
						$actionType = $preferences['action_type']; // specific type (all, post, thread, profile_post, threadban, resource, modess)
						
						if (($actionType == 'threadban' && !$doIncludeThreadBans) || ($actionType == 'resource' && !$doIncludeResources) || ($actionType == 'modess' && !$doIncludeWarnings))
						{
							// type no longer relevant so reset preferences
							$this->deleteModEssPreferencesByUserId($userId);
							$doCount = true;
						}
						else if ($actionType != 'none' && ($actionType == 'all' || $actionType == $contentType || ($actionType == 'threadban' && $contentType == 'thread')))
						{
							$desiredActions = json_decode($preferences['action_names']);
							
							if ($contentType == 'resource' || $contentType == 'resource_update')
							{
								if ($action == 'edit' || $action == 'reassign' || $action == 'feature' || $action == 'unfeature')
									$action = 'resource_' . $action;
							}
							
							if ($contentType == 'modess')
							{
								if ($action == 'warn_update' || $action == 'warn_delete')
									$action = 'warn'; // since we are only using one preference checkbox for all warning actions ('warn', 'warn_update', 'warn_delete')
							}
							
							if (in_array($action, $desiredActions))
								$doCount = true; // action is one of the selected in preferences
						}
					}
					else
					{
						$doCount = true; // preferences have not yet been saved, so use default which is to notify on all actions
					}
					
					// mod wishes to be notified of this action
					if ($doCount)
					{
						if ($contentType == 'post' || $contentType == 'thread')
						{
							$threadId = $content['thread_id'];
							
							if ($contentType == 'post' || !isset($content['node_id']))
								$thread = $threadModel->getThreadById($threadId);
							else
								$thread = $content;
							
							if (!empty($thread))
							{
								$nodePermissions = $permissionCacheModel->getContentPermissionsForItem(
									$user['permission_combination_id'], 'node', $thread['node_id']
								);
								
								$errorPhraseKey= '';
								$canViewNodeThreadLog = $threadModel->canViewThreadModeratorLog(array('node_id' => $thread['node_id']), array(), $errorPhraseKey, $nodePermissions, $user);
							}
						}
						else if ($contentType == 'profile_post')
						{
							// can view user profile
							$errorPhraseKey = '';
							$fetchOptions = array('join' => XenForo_Model_User::FETCH_USER_FULL | XenForo_Model_User::FETCH_USER_PERMISSIONS);
							$profileUser = $userModel->getUserById($content['profile_user_id'], $fetchOptions);
							if ($userProfileModel->canViewFullUserProfile($profileUser, $errorPhraseKey, $user))
								$canViewNodeThreadLog = true;
						}
						else if ($contentType == 'resource' || $contentType == 'resource_update')
						{
							if ($doIncludeResources)
								$canViewNodeThreadLog = true;
						}
						else if ($contentType == 'modess')
						{
							if ($doIncludeWarnings && ($action == 'warn' || $action == 'warn_update' || $action == 'warn_delete'))
							{
								// can view user profile
								$errorPhraseKey = '';
								$fetchOptions = array('join' => XenForo_Model_User::FETCH_USER_FULL | XenForo_Model_User::FETCH_USER_PERMISSIONS);
								$profileUser = $userModel->getUserById($content['user_id'], $fetchOptions);
								if ($userProfileModel->canViewFullUserProfile($profileUser, $errorPhraseKey, $user))
								{
									$canViewNodeThreadLog = true;
								}
							}
						}
					}
					
					$registryCounts = $logCounts[$userId];
					
					if ($canViewNodeThreadLog)
					{
						$registryCounts['total'] = $registryCounts['total'] + 1;
						if (!empty($registryCounts['logIds']))
						{
							$registryCounts['logIds'] = $registryCounts['logIds'] . ',' . $logId;
							
							// only keep the latest ids upto maximum items
							$maxEntries = XenForo_Application::get('options')->modess_mod_log_max_entries;
							if ($registryCounts['total'] > $maxEntries)
							{
								$ids = explode(',', $registryCounts['logIds']);
								$ids = array_slice($ids, ($maxEntries * -1), $maxEntries, true);
								$registryCounts['logIds'] = implode(',', $ids);
							}
						}
						else
						{
							$registryCounts['logIds'] = $logId;
						}
					}
				}
				else
				{
					$registryCounts = array(
						'total' => 0,
						'lastViewDate' => XenForo_Application::$time,
						'logIds' => '',
						'isActive' => true
					);
				}
				
				$cache[$userId] = $registryCounts;
			}
		}
		
		if (!empty($cache))
		{
			$this->_getDataRegistryModel()->set('modLogCounts', $cache);
			XenForo_Application::set('modLogCounts', $cache); // must set it in case there are multiple entries from one function. Usually it is set on page load via Cog.
		}
		else
		{
			$this->_getDataRegistryModel()->delete('modLogCounts');
		}
	}
	
	/**
	 * Rebuilds the mod log session counts cache.
	 *
	 * @param array $modId
	 * @param boolean $useRegistry - fetch using the registry directly
	 *
	 * @return array
	 */
	public function rebuildSessionModLogCountsCache($modId = null, $useRegistry = false)
	{
		$mods = $this->getSessionModLogModerators();
		
		if ($useRegistry)
			$logCounts = $this->_getDataRegistryModel()->get('modLogCounts');
		else if (XenForo_Application::isRegistered('modLogCounts'))
			$logCounts = XenForo_Application::get('modLogCounts');
		else
			$logCounts = array();
		
		// check for and remove any demoted moderators
		if (!empty($logCounts))
		{
			$demotedMods = array_diff_key($logCounts, $mods);
			if (!empty($demotedMods))
			{
				foreach ($demotedMods AS $demoteId => $demote)
				{
					unset($logCounts[$demoteId]);
					$this->deleteModEssPreferencesByUserId($demoteId); // remove action preferences
				}
			}
		}
		
		$cache = $logCounts;
		
		if (isset($modId))
		{
			if (!array_key_exists($modId, $mods)) // no such mod log moderator
				return $cache;
		}
		
		foreach ($mods AS $userId => $user)
		{
			// only update the one moderator
			if (isset($modId) && $modId != $userId)
				continue;
			
			// does mod have the view thread log permission in any forum
			$isActive = $this->_canViewThreadLogs($user);
			
			if ($isActive && !empty($logCounts) && isset($logCounts[$userId]))
			{
				$sessionLogCounts = $logCounts[$userId];
				
				// get count since last session
				$entries = $this->getViewableModLogEntries($user, $sessionLogCounts['lastViewDate']);
				$sessionLogCounts['total'] = count($entries);
				$keys = array_keys($entries);
				$sessionLogCounts['logIds'] = implode(',', $keys);
				$sessionLogCounts['isActive'] = true;
			}
			else
			{		
				$sessionLogCounts = array(
					'total' => 0,
					'lastViewDate' => XenForo_Application::$time,
					'logIds' => '',
					'isActive' => $isActive
				);
			}
			
			$cache[$userId] = $sessionLogCounts;
		}
		
		if (!empty($cache))
			$this->_getDataRegistryModel()->set('modLogCounts', $cache);
		else
			$this->_getDataRegistryModel()->delete('modLogCounts');
		
		return $cache;
	}
	
	/**
	 * Gets all available Moderator Log actions
	 *
	 * @param boolean $includeThreadBans - whether or not to include Thread Ban actions
	 * @param boolean $includeResources - whether or not to include Resource actions
	 *
	 * @return array of $actions[key] = array(type, checked, name)
	 */
	public function getModLogActionsForTemplate($includeThreadBans = false, $includeResources = false, $includeWarnings = false, $type = 'all')
	{
		$actions = array();
		
		if ($type != 'threadban')
		{
			$all = array(
				'delete_soft' => array('type' => 'all', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_delete_soft')),
				'delete_hard' => array('type' => 'all', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_delete_hard')),
				'undelete' => array('type' => 'all', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_undelete')),
				'approve' => array('type' => 'all', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_approve')),
				'unapprove' => array('type' => 'all', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_unapprove'))
			);
			
			$actions = array_merge($actions, $all);
		}
		
		if ($type == 'all' || $type == 'post')
		{
			$post = array(
				'merge_target' => array('type' => 'post_thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_merge_target')),
				'edit' => array('type' => 'post', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_edit'))
			);
			
			$actions = array_merge($actions, $post);
		}
		
		if ($type == 'all' || $type == 'thread')
		{
			$thread = array(
				'merge_target' => array('type' => 'post_thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_merge_target')),
				'title' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_title')),
				'stick' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_stick')),
				'unstick' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_unstick')),
				'lock' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_lock')),
				'unlock' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_unlock')),
				'prefix' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_prefix')),
				'move' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_move')),
				'post_move_source' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_post_move_source')),
				'post_move_target' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_post_move_target')),
				'post_move_target_existing' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_post_move_target_existing')),
				'post_copy_source' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_post_copy_source')),
				'post_copy_target' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_post_copy_target')),
				'post_copy_target_existing' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_post_copy_target_existing')),
				'poll_edit' => array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_poll_edit'))
			);
			
			if (XenForo_Application::$versionId > 1040030) // XF 1.4 or newer
			{
				$thread['poll_add'] = array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_poll_add'));
				$thread['reply_ban'] = array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_reply_ban'));
				$thread['reply_ban_delete'] = array('type' => 'thread', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_reply_ban_delete'));
			}
			
			$actions = array_merge($actions, $thread);
		}
		
		if ($includeThreadBans && ($type == 'all' || $type == 'threadban'))
		{
			$actions['threadban'] = array('type' => 'threadban', 'checked' => 0, 'name' => new XenForo_Phrase('modess_threadban'));
			$actions['threadban_edit'] = array('type' => 'threadban', 'checked' => 0, 'name' => new XenForo_Phrase('modess_threadban_edit'));
			$actions['threadbanlift_cron'] = array('type' => 'threadban', 'checked' => 0, 'name' => new XenForo_Phrase('modess_threadbanlift_cron'));
			$actions['threadbanlift'] = array('type' => 'threadban', 'checked' => 0, 'name' => new XenForo_Phrase('modess_threadbanlift'));
		}
		
		if ($includeResources && ($type == 'all' || $type == 'resource'))
		{
			$actions['resource_feature'] = array('type' => 'resource', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_resource_feature'));
			$actions['resource_unfeature'] = array('type' => 'resource', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_resource_unfeature'));
			$actions['resource_edit'] = array('type' => 'resource', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_resource_edit'));
			$actions['resource_reassign'] = array('type' => 'resource', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_resource_reassign'));
		}
		
		if ($includeWarnings && ($type == 'all' || $type == 'modess'))
		{
			$actions['warn'] = array('type' => 'modess', 'checked' => 0, 'name' => new XenForo_Phrase('modess_modlog_user_warnings'));
		}
		
		return $actions;	
	}
	
	/**
	 * Gets all Moderators who can view the mod log counts bar
	 *
	 * @return array Format: [user id] => user info
	 */
	public function getSessionModLogModerators()
	{
		$moderators = $this->fetchAllKeyed('
			SELECT user.*, permission_combination.cache_value AS global_permission_cache
			FROM xf_moderator AS moderator
			INNER JOIN xf_user AS user ON (user.user_id = moderator.user_id)
			LEFT JOIN xf_permission_combination AS permission_combination ON (permission_combination.permission_combination_id = user.permission_combination_id)
		', 'user_id');
		
		foreach ($moderators AS $userId => &$user)
		{
			$user['permissions'] = XenForo_Permission::unserializePermissions($user['global_permission_cache']);
			if (!XenForo_Permission::hasPermission($user['permissions'], 'general', 'canViewModLogCounts'))
				unset($moderators[$userId]);
		}
		
		return $moderators;
	}
	
	/**
	 * @return XenForo_Model_Log
	 */
	protected function _getLogModel()
	{
		return $this->getModelFromCache('XenForo_Model_Log');
	}
	
	/**
	 * @return XenForo_Model_User
	 */
	protected function _getUserModel()
	{
		return $this->getModelFromCache('XenForo_Model_User');
	}
	
	/**
	 * @return XenForo_Model_UserProfile
	 */
	protected function _getUserProfileModel()
	{
		return $this->getModelFromCache('XenForo_Model_UserProfile');
	}
	
	/**
	 * @return XenForo_Model_Thread
	 */
	protected function _getThreadModel()
	{
		return $this->getModelFromCache('XenForo_Model_Thread');
	}
	
	/**
	 * @return XenForo_Model_PermissionCache
	 */
	protected function _getPermissionCacheModel()
	{
		return $this->getModelFromCache('XenForo_Model_PermissionCache');
	}
	
	/**
	 * @return XenForo_Model_AddOn
	 */
	protected function _getAddOnModel()
	{
		return $this->getModelFromCache('XenForo_Model_AddOn');
	}
}