<?php

/**
 * ControllerPublic class for user accounts
 *
 * @package UserEss
 */
class UserEss_ControllerPublic_Account extends XFCP_UserEss_ControllerPublic_Account
{
	/**
	 * Displays a form to set a password for FaceBook users.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionSetPassword()
	{
		$visitor = XenForo_Visitor::getInstance()->toArray();

		if (!$visitor['user_id'])
			return $this->responseNoPermission();
		
		if ($this->isConfirmedPost())
		{
			if (!$this->_input->filterSingle('understand', XenForo_Input::UINT))
			{
				$viewParams = array(
					'errors' => array(new XenForo_Phrase('useress_unc_must_tick_box'))
				);
						
				return $this->responseView('XenForo_ViewPublic_Base', 'useress_unc_errors', $viewParams);				
			}
			
			$confirmationModel = $this->getModelFromCache('XenForo_Model_UserConfirmation');
			$options = XenForo_Application::get('options');
			
			if ($options->lostPasswordTimeLimit)
			{
				if ($confirmation = $confirmationModel->getUserConfirmationRecord($visitor['user_id'], 'password'))
				{
					$timeDiff = XenForo_Application::$time - $confirmation['confirmation_date'];
					
					if ($options->lostPasswordTimeLimit > $timeDiff)
					{
						$wait = $options->lostPasswordTimeLimit - $timeDiff;
						
						return $this->responseError(new XenForo_Phrase('must_wait_x_seconds_before_performing_this_action', array('count' => $wait)));
					}
				}
			}
			
			$confirmationModel->sendPasswordResetRequest($visitor);
			
			// log the user out
			$this->getModelFromCache('XenForo_Model_Session')->processLastActivityUpdateForLogOut($visitor['user_id']);
			
			XenForo_Application::get('session')->delete();
			XenForo_Helper_Cookie::deleteAllCookies(
				array('session'),
				array('user' => array('httpOnly' => false))
			);
			
			XenForo_Visitor::setup(0);
			
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('index'),
				new XenForo_Phrase('password_reset_request_has_been_emailed_to_you')
			);			
		}
		else
		{
			return $this->responseView('XenForo_ViewPublic_Base', 'useres_unc_set_password', array());
		}
	}
		
	/**
	 * Form to change user name
	 *
	 * @return XenForo_ViewPublic_Base
	 */
	public function actionUserNameChange()
	{
		$viewingUser = XenForo_Visitor::getInstance()->toArray();
		$uncModel = $this->_getUNCModel();
		if (!$uncModel->canUserNameChange($viewingUser))
		{
			return $this->responseNoPermission();
		}
		
		$options = XenForo_Application::get('options');
		
		if ($options->uncModerate && !($viewingUser['is_moderator'] || $viewingUser['is_admin']))
			$doModerate = true;
		else
			$doModerate = false;
		
		$maxNameChanges = $options->uncMaxChanges;
		$changesLeft = $uncModel->numUserNameChangesLeft($viewingUser);
		
		$conditions = array();
		// do we count name changes made via the ACP?
		if (!XenForo_Application::get('options')->uncDisplayChangesViaACP)
			$conditions['skip_acp'] = 1;
		$previousChanges = $uncModel->getUNCByUserId($viewingUser['user_id'], $conditions);
		
		$auth = $this->_getUserModel()->getUserAuthenticationObjectByUserId($viewingUser['user_id']);
		if (!$auth)
		{
			return $this->responseNoPermission();
		}
		
		// check if there is a post limit after which changes are not allowed
		// this allows new members to only change their name soon after registering
		$maxPostCount = $options->uncMaxPostCount;
		$currnetPostCount = $viewingUser['message_count'];
		if ($maxPostCount && $currnetPostCount > $maxPostCount)
		{
			$maxPostCountReached = true;
			$changesLeft = 0;
		}
		else
		{
			$maxPostCountReached = false;
		}
		
		$nextCyclePeriod = array();
		$resetDays = $options->uncResetDays;
		$lastChange = array();
		$nameChangePending = false;
		if (!empty($previousChanges))
		{
			$lastChange = reset($previousChanges);
			if ($lastChange['status'] == 'moderated')
			{
				$nameChangePending = true;
			}
			else if ($resetDays) // not pending so find out when they can change name again
			{
				if (!$changesLeft && $changesLeft > -1) // not unlimited
				{
					// find the first entry in this cycle
					$firstChange = array();
					$count = 0;
					foreach ($previousChanges AS $previousChange)
					{
						if ($previousChange['status'] == 'denied')
							continue;
						
						$count++;
						if ($count == $maxNameChanges)
						{
							$firstChange = $previousChange;
							break;
						}
					}
					
					if (!empty($firstChange))
					{
						$cutOff = XenForo_Application::$time - (86400 * $resetDays);
						$nextCyclePeriod['days'] = ($firstChange['date'] - $cutOff) / 86400;
						
						if ($nextCyclePeriod['days'] <= 1)
						{
							$nextCyclePeriod['hours'] = ($firstChange['date'] - $cutOff) / 3600;
						}
					}
				}
			}
		}
		
		$viewParams = array(
			'maxNameChanges' => $maxNameChanges,
			'maxPostCount' => $maxPostCount,
			'maxPostCountReached' => $maxPostCountReached,
			'changesLeft' => $changesLeft,
			'previousChanges' => $previousChanges,
			'nameChangePending' => $nameChangePending,
			'noPassword' => !$auth->hasPassword(),
			'doModerate' => $doModerate,
			'resetDays' => $resetDays,
			'nextCyclePeriod' => $nextCyclePeriod
		);
		
		return $this->_getWrapper(
			'account', 'unc',
			$this->responseView('XenForo_ViewPublic_Base', 'useress_unc', $viewParams)
		);
	}
	
	/**
	 * Change user name.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionChangeName()
	{
		$this->_assertPostOnly();
		
		$options = XenForo_Application::get('options');
		$viewingUser = XenForo_Visitor::getInstance();
		$userId = $viewingUser['user_id'];
		$password = $this->_input->filterSingle('password', XenForo_Input::STRING);
		$username = $this->_input->filterSingle('username', XenForo_Input::STRING);
		$errors = array();
		
		if ($options->uncModerate && !($viewingUser['is_moderator'] || $viewingUser['is_admin']))
			$doModerate = true;
		else
			$doModerate = false;
		
		$auth = $this->_getUserModel()->getUserAuthenticationObjectByUserId($userId);
		
		if (!$auth || !$auth->authenticate($userId, $password))
		{
			$errors['password'] = new XenForo_Phrase('your_existing_password_is_not_correct');
		}
				
		$writer = XenForo_DataWriter::create('XenForo_DataWriter_User');
		$writer->set('username', $username);
		
		// check whether this username was previously used by any other member
		if (!$options->uncAllowPreviousNames)
		{
			$uncModel = $this->_getUNCModel();
			if ($uncModel->doesUNCNameExists($username, $viewingUser['user_id']))
			{
				$writer->error(new XenForo_Phrase('useress_unc_name_previously_used'), 'username', false);
			}
		}
		
		$errors += $writer->getErrors();
		
		if (!empty($errors))
		{
			$viewParams = array(
				'errors' => $errors
			);
					
			return $this->responseView('XenForo_ViewPublic_Base', 'useress_unc_errors', $viewParams);
		}
		
		if ($doModerate) // moderated
		{
			$status = 'moderated';
		}
		else // instant
		{
			$status = 'approved';
			
			$writer->setExistingData($userId);
			$writer->save();
			$username = $writer->get('username'); // get name with white spaces removed
		}
		
		// add to unc table
		$dw = XenForo_DataWriter::create('UserEss_DataWriter_UNC');
		
		$vars = array(
			'user_id' => $userId,
			'from_username' => $viewingUser['username'],
			'to_username' => $username,
			'date' => XenForo_Application::$time,
			'user_note' => $this->_input->filterSingle('note', XenForo_Input::STRING),
			'status' => $status
		);
				
		$dw->bulkSet($vars);
		$dw->save();
		$unc = $dw->getMergedData();
		
		if ($doModerate) // moderated
		{
			// insert into moderation queue
			$this->getModelFromCache('XenForo_Model_ModerationQueue')->insertIntoModerationQueue(
				'unc', $unc['unc_id'], XenForo_Application::$time
			);
			
			// user log
			UserEss_Model_UserLog::logUserAction(
				'user', array('user_id' => $viewingUser['user_id'], 'username' => $viewingUser['username']), 'username_request', array('new' => $username), null, $viewingUser->toArray()
			);
		}
		else // instant
		{
			// log action
			UserEss_Model_UserLog::logUserAction(
				'user', array('user_id' => $viewingUser['user_id'], 'username' => $username), 'username', array('old' => $unc['from_username']), null, $viewingUser->toArray()
			);
			
			if (XenForo_Application::get('options')->uncModerateAlert)
			{
				$alertModel = $this->_getAlertModel();
				$moderatorModel = $this->_getModeratorModel();
				$mods = $moderatorModel->getAllGeneralModerators();
				unset($mods[$viewingUser['user_id']]); // remove in case it is a mod
			
				foreach ($mods AS $mod)
				{
							   //alert($alertUserId, $userId, $username, $contentType, $contentId, $action, array $extraData = null)
					$alertModel->alert($mod['user_id'], $mod['user_id'], $mod['username'], 'unc', $unc['unc_id'], 'mod_approved');			
				}
			}
		}
			
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('account/UserNameChange'),
			$doModerate ? new XenForo_Phrase('useress_name_change_request_sent') : new XenForo_Phrase('useress_name_change_successful')
		);
	}
	
	/**
	 * Validate username field
	 *
	 * @return XenForo_ControllerResponse_Redirect
	 */
	public function actionValidateField()
	{
		$response = parent::actionValidateField();
		
		if (!empty($response) && $response instanceof XenForo_ControllerResponse_Redirect)
		{
			// check whether this username was previously used by any other member
			if (!XenForo_Application::get('options')->uncAllowPreviousNames)
			{
				$data = $this->_getFieldValidationInputParams();
			
				if (!empty($data) && isset($data['name']) && $data['name'] == 'username')
				{
					if ($this->_getUNCModel()->doesUNCNameExists($data['value'], XenForo_Visitor::getUserId()))
					{
						$writer = XenForo_DataWriter::create('XenForo_DataWriter_User');
						$writer->error(new XenForo_Phrase('useress_unc_name_previously_used'), 'username', false);
				
						if ($errors = $writer->getErrors())
						{
							return $this->responseError($errors);
						}
					}
				}
			}
		}
		
		return $response;
	}
	
	/**
	 * @return UserEss_Model_UNC
	 */
	protected function _getUNCModel()
	{
		return $this->getModelFromCache('UserEss_Model_UNC');
	}
	
	/**
	 * @return XenForo_Model_Alert
	 */
	protected function _getAlertModel()
	{
		return $this->getModelFromCache('XenForo_Model_Alert');
	}

	/**
	 * @return XenForo_Model_Moderator
	 */
	protected function _getModeratorModel()
	{
		return $this->getModelFromCache('XenForo_Model_Moderator');
	}
}