<?php

/**
 * Model for threads.
 *
 * @package UserEss
 */
class UserEss_Model_Thread extends XFCP_UserEss_Model_Thread
{
	/**
	 * Returns a thread record that includes thread lock info
	 *
	 * @param integer $threadId
	 * @param array $fetchOptions Collection of options related to fetching
	 *
	 * @return array|false
	 */
	public function getThreadById($threadId, array $fetchOptions = array())
	{
		if (XenForo_Visitor::getUserId())
		{
			$fetchOptions['useress_thread_lock'] = 1;
		}
		
		return parent::getThreadById($threadId, $fetchOptions);
	}
	
	/**
	 * Finds out if a mod has locked a thread last
	 *
	 * @param integer $threadId
	 *
	 * @return boolean
	 */
	public function wasLockedByMod($threadId)
	{
		return $this->_getDb()->fetchOne('
			SELECT IF(user_id IS NULL, 0, 1)
			FROM useress_thread_lock
			WHERE thread_id = ?
			LIMIT 1', $threadId
		);
	}
	
	/**
	 * Determines if a new reply can be posted in the specified thread,
	 * with the given permissions. This does not check viewing permissions.
	 *
	 * @param array $thread
	 * @param array $forum
	 * @param string $errorPhraseKey Returned phrase key for a specific error
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function canReplyToThread(array $thread, array $forum, &$errorPhraseKey = '', array $nodePermissions = null, array $viewingUser = null)
	{
		$canReply = parent::canReplyToThread($thread, $forum, $errorPhraseKey, $nodePermissions, $viewingUser);
		
		if (!$canReply && $errorPhraseKey == 'you_may_not_perform_this_action_because_discussion_is_closed')
		{
			if ($this->canLockUnlockThread($thread, $forum, $errorPhraseKey, $nodePermissions, $viewingUser))
				$canReply = true;
		}
		
		return $canReply;
	}
	
	/**
	 * Determines if the thread can be locked/unlocked with the given permissions.
	 * This does not check viewing permissions.
	 *
	 * @param array $thread
	 * @param array $forum
	 * @param string $errorPhraseKey
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function canLockUnlockThread(array $thread, array $forum, &$errorPhraseKey = '', array $nodePermissions = null, array $viewingUser = null)
	{
		$this->standardizeViewingUserReferenceForNode($thread['node_id'], $viewingUser, $nodePermissions);
		if ($viewingUser['user_id'] && isset($thread['isThreadBanned']))
		{
			return false;
		}
		
		$response = parent::canLockUnlockThread($thread, $forum, $errorPhraseKey, $nodePermissions, $viewingUser);
		
		// only need to check if thread author and was not last locked by an admin or moderator
		if (!$response && $thread['user_id'] == $viewingUser['user_id'] && (isset($thread['lockedByMod']) && !$thread['lockedByMod']))
		{
			// thread author, so check if can lock/unlock own thread
			$response = XenForo_Permission::hasContentPermission($nodePermissions, 'lockUnlockOwnThread');
		}
		
		return $response;
	}
	
	/**
	 * Process thread lock joins
	 * Checks the 'join' key of the incoming array for the presence of the FETCH_x bitfields in this class
	 * and returns SQL snippets to join the specified tables if required
	 *
	 * @param array $fetchOptions containing a 'join' integer key build from this class's FETCH_x bitfields
	 *
	 * @return array Containing selectFields, joinTables, orderClause keys.
	 * 		Example: selectFields = ', user.*, foo.title'; joinTables = ' INNER JOIN foo ON (foo.id = other.id) '; orderClause = ORDER BY x.y
	 */
	public function prepareThreadFetchOptions(array $fetchOptions)
	{
		$visitorId = XenForo_Visitor::getUserId();
		if ($visitorId > 0)
		{
			if (!empty($fetchOptions))
			{
				if (isset($fetchOptions['useress_thread_lock']))
				{
					$joinOptions = parent::prepareThreadFetchOptions($fetchOptions);
					
					$selectFields = ',
						IF(thread_lock.user_id IS NULL, 0, thread_lock.user_id) AS lockedByMod';
					$joinTables = '
						LEFT JOIN useress_thread_lock AS thread_lock ON(thread_lock.thread_id = thread.thread_id)';
					
					$joinOptions['selectFields'] .= $selectFields;
					$joinOptions['joinTables'] .= $joinTables;
					
					return $joinOptions;
				}
			}
		}
		
		return parent::prepareThreadFetchOptions($fetchOptions);
	}
	
	/**
	 * Determines if the thread title be edited with the given permissions.
	 * This does not check thread viewing permissions.
	 *
	 * @param array $thread Info about the thread
	 * @param array $forum Info about the forum the thread is in
	 * @param string $errorPhraseKey Returned phrase key for a specific error
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function canEditThreadTitle(array $thread, array $forum, &$errorPhraseKey = '', array $nodePermissions = null, array $viewingUser = null)
	{
		$canEdit = parent::canEditThreadTitle($thread, $forum, $errorPhraseKey, $nodePermissions, $viewingUser);
		
		// check for time limit on editing own thread title
		if ($canEdit || (is_array($errorPhraseKey) && $errorPhraseKey[0] == 'message_edit_time_limit_expired' && XenForo_Application::get('options')->useressSeparateTitleEdit))
		{
			$this->standardizeViewingUserReferenceForNode($thread['node_id'], $viewingUser, $nodePermissions);
			
			// is thread creator and not moderator with permission
			if ($thread['user_id'] == $viewingUser['user_id'] && !XenForo_Permission::hasContentPermission($nodePermissions, 'manageAnyThread') && XenForo_Permission::hasContentPermission($nodePermissions, 'editOwnThreadTitle'))
			{
				$editLimit = XenForo_Permission::hasContentPermission($nodePermissions, 'editOwnThreadTitleLimit');
				
				if ($editLimit != -1 && $thread['post_date'] < (XenForo_Application::$time - 60 * $editLimit))
				{
					$errorPhraseKey = array('useress_title_edit_time_limit_expired', 'minutes' => $editLimit);
					$canEdit = false;
				}
				else
				{
					$canEdit = true;
				}
			}	
		}
		
		return $canEdit;
	}
	
	/**
	 * Determines if a poll can be added to this thread.
	 *
	 * @param array $thread
	 * @param array $forum
	 * @param string $errorPhraseKey
	 * @param array|null $nodePermissions
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */
	public function canAddPoll(array $thread, array $forum, &$errorPhraseKey = '', array $nodePermissions = null, array $viewingUser = null)
	{
		$canAddPoll = parent::canAddPoll($thread, $forum, $errorPhraseKey, $nodePermissions, $viewingUser);
		
		// now check for user group permission
		if ($canAddPoll)
		{
			$this->standardizeViewingUserReferenceForNode($thread['node_id'], $viewingUser, $nodePermissions);
			
			if (!XenForo_Permission::hasContentPermission($nodePermissions, 'createPoll'))
			{
				$canAddPoll = false;
			}
		}
		
		return $canAddPoll;
	}
}