<?php

/**
 * Model for users.
 *
 * @package UserEss
 */
class UserEss_Model_UNC extends XenForo_Model
{
	/**
	 * Determines if the viewing user can change their own user name.
	 *
	 * @param array|null $viewingUser
	 *
	 * @return boolean
	 */	
	public function canUserNameChange(array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'] || $viewingUser['user_state'] != 'valid')
		{
			return false;
		}
		
		$hasPermission = XenForo_Permission::hasPermission($viewingUser['permissions'], 'general', 'canUserNameChange');
		if ($hasPermission)
		{	
			// check if there is a post limit after which changes are not allowed
			// this allows new members to only change their name soon after registering
			$maxPostCount = XenForo_Application::get('options')->uncMaxPostCount;
			$currnetPostCount = $viewingUser['message_count'];
			if ($maxPostCount && $currnetPostCount > $maxPostCount)
				return false;
			else
				return true;
		}
		else
			return false;
	}
	
	/**
	 * Determines number of times user can still change their name.
	 *
	 * @param array|null $viewingUser
	 *
	 * @return integer - number of changes left
	 */	
	public function numUserNameChangesLeft(array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'] || $viewingUser['user_state'] != 'valid')
		{
			return 0;
		}
		
		$options = XenForo_Application::get('options');
		
		// unlimited changes allowed
		$maxNameChanges = $options->uncMaxChanges;
		if ($maxNameChanges < 1)
		{
			return -1;
		}
		else // find out how many changes user has already made
		{	
			$conditions = array();
			$conditions['skip_acp'] = 1;
			$conditions['status'] = array('approved', 'moderated');
			
			if ($options->uncResetDays)
			{
				$conditions['cut_off'] =  XenForo_Application::$time - (86400 * $options->uncResetDays);
			}
			
			$usedNameChanges = $this->countUNCByUserId($viewingUser['user_id'], $conditions);
			$changesLeft = $maxNameChanges - $usedNameChanges; // number of times user can still change their name
			if ($changesLeft < 0)
				$changesLeft = 0;
				
			return $changesLeft;		
		}
	}
	
	/**
	 * Get a UNC item by ID
	 *
	 * @param integer $uncId - the auto increment value of the item
	 *
	 * @return array unc
	 */	
	public function getUNCById($uncId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM useress_unc
			WHERE unc_id = ?
		', $uncId);		
	}
	
	/**
	 * Get a UNC items by ID's
	 *
	 * @param array $uncIds - the auto increment value of the items
	 *
	 * @return array unc
	 */	
	public function getUNCByIds(array $uncIds)
	{
		$db = $this->_getDb();
		
		return $this->fetchAllKeyed('
			SELECT *
			FROM useress_unc
			WHERE unc_id IN (' . $db->quote($uncIds) . ')
		', 'unc_id');		
	}

	/**
	 * Find matching name
	 *
	 * @param string $username
	 * @param integer $userId
	 *
	 * @return integer
	 */	
	public function doesUNCNameExists($username, $userId = 0)
	{
		return $this->_getDb()->fetchOne("
			SELECT unc_id
			FROM useress_unc
			WHERE from_username = ? AND user_id != ? AND (status = 'approved' OR status = 'moderated')
		", array($username, $userId));		
	}
		
	/**
	 * Get the current count of user name changes
	 *
	 * @param integer $userId
	 * @param array $fetchConditions
	 *
	 * @return integer
	 */
	public function countUNCByUserId($userId, array $fetchConditions = array())
	{
		$db = $this->_getDb();
		$whereConditions = $this->prepareFetchConditions($fetchConditions);
		
		return $db->fetchOne('
			SELECT COUNT(*)
			FROM useress_unc
			WHERE user_id = ? AND ' . $whereConditions . '
		', $userId);
	}
	
	/**
	 * Get all user name changes inserts
	 *
	 * @param integer $userId
	 * @param array $fetchConditions
	 * @param array $fetchOptions
	 *
	 * @return array unc
	 */	
	public function getUNCByUserId($userId, array $fetchConditions = array(), array $fetchOptions = array())
	{
		$userId = intval($userId);
		$whereConditions = $this->prepareFetchConditions($fetchConditions);
		$sqlClauses = $this->prepareFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		
		return $this->fetchAllKeyed($this->limitQueryResults('
			SELECT *
			FROM useress_unc
			WHERE user_id = ? AND ' . $whereConditions . '
			' . $sqlClauses['orderClause'] . '
			', $limitOptions['limit'], $limitOptions['offset']
		), 'unc_id', $userId);
	}
	
	public function prepareFetchConditions(array $conditions)
	{
		$sqlConditions = array();
		$db = $this->_getDb();
		
		if (!empty($conditions['skip_acp']))
		{
			$sqlConditions[] = 'via_acp = 0';
		}
		
		if (!empty($conditions['status']))
		{
			$status = $conditions['status'];
			
			if (is_array($status))
			{
				foreach ($status AS &$stat)
				{
					$stat = $db->quote($stat);
				}
				
				if (count($status) > 1)
					$sqlConditions[] = 'status IN (' . implode(', ', $status) . ')';
				else
					$sqlConditions[] = 'status = ' . implode(', ', $status);			
			}
			else
			{
				$sqlConditions[] = 'status = ' . $db->quote($status);
			}
		}
		
		if (!empty($conditions['cut_off']))
		{
			$sqlConditions[] = 'date > ' . $conditions['cut_off'];
		}
		
		return $this->getConditionsForClause($sqlConditions);
	}
	
	public function prepareFetchOptions(array $fetchOptions)
	{
		$selectFields = '';
		$orderBy = '';
		
		if (!empty($fetchOptions['order']))
		{
			$orderBy = $fetchOptions['order'];
		}
		else
		{
			$orderBy = 'date';
		}
		
		if (!isset($fetchOptions['orderDirection']) || $fetchOptions['orderDirection'] == 'desc')
		{
			$orderBy .= ' DESC';
		}
		else
		{
			$orderBy .= ' ASC';
		}
		
		return array(
			'selectFields' => $selectFields,
			'orderClause'  => ($orderBy ? "ORDER BY $orderBy" : '')
		);	
	}
	
	public function uncRevertCustomTitles()
	{
		$db = $this->_getDb();
		$customTitleDays = XenForo_Application::get('options')->uncCustomTitleDays;
		$doRevert = false;
		
		if ($customTitleDays)
		{
			$doRevert = true;
		}
		else
		{
			// check to see if any are left over from before the option was turned off
			$count = $db->fetchOne('SELECT COUNT(*) FROM useress_unc_cron');
			if ($count)
				$doRevert = true;
		}
		
		if ($doRevert)
		{
			$pruneDate = XenForo_Application::$time - (86400 * $customTitleDays);
			
			$uncs = $this->fetchAllKeyed('
				SELECT *
				FROM useress_unc_cron
				WHERE date < ?
			', 'user_id', $pruneDate);
			
			if (!empty($uncs))
			{
				$userModel = $this->_getUserModel();
				
				foreach ($uncs AS $id => $unc)
				{
					$user = $userModel->getUserById($id);
				
					if (!empty($user))
					{
						// only revert if its the same one we originally created
						if ($user['custom_title'] == $unc['new_custom_title'])
						{
							$userModel->update($user, 'custom_title', $unc['old_custom_title']);
						}
					}
				}
			
				// delete from table
				$userIds = array_keys($uncs);
				$db->delete('useress_unc_cron', 'user_id IN (' . $db->quote($userIds) . ')');
			}
		}	
	}
	
	/**
	 * @return XenForo_Model_User
	 */
	protected function _getUserModel()
	{
		return $this->getModelFromCache('XenForo_Model_User');
	}
}