<?php

class UserEss_ModerationQueueHandler_UNC extends XenForo_ModerationQueueHandler_Abstract
{
	/**
	 * @var XenForo_Model_User
	 */
	protected $_userModel = null;
	
	/**
	 * @var UserEss_Model_UNC
	 */
	protected $_uncModel = null;
	
	/**
	 * @var XenForo_Model_Alert
	 */
	protected $_alertModel = null;
	
	/**
	 * @var XenForo_Model_Moderator
	 */
	protected $_moderatorModel = null;
	
	/**
	 * @var XenForo_Model_ModerationQueue
	 */
	protected $_moderationQueueModel = null;
	
	/**
	 * Gets visible moderation queue entries for specified user.
	 *
	 * @see XenForo_ModerationQueueHandler_Abstract::getVisibleModerationQueueEntriesForUser()
	 */
	public function getVisibleModerationQueueEntriesForUser(array $contentIds, array $viewingUser)
	{
		$output = array();
		
		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'general', 'manageUserNameChange'))
		{
			$uncs = $this->_getUNCModel()->getUNCByIds($contentIds);
		
			foreach ($uncs AS $unc)
			{
				if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'general', 'viewIps'))
					$unc['ipAddress'] = ($unc['ip_address'] ? XenForo_Helper_Ip::convertIpBinaryToString($unc['ip_address']) : '');
				else
					$unc['ipAddress'] = '';
				
				$output[$unc['unc_id']] = array(
					'message' => $unc['user_note'],
					'user' => array(
						'user_id' => $unc['user_id'],
						'username' => $unc['from_username']
					),
					'title' => $unc['to_username'],
					'link' => XenForo_Link::buildAdminLink('users/edit', $unc),
					'contentTypeTitle' => new XenForo_Phrase('useress_unc_request'),
					'titleEdit' => false,
					'ipAddress' => $unc['ipAddress']
				);
			}
		}

		return $output;
	}
	
	/**
	 * Approves the specified moderation queue entry.
	 *
	 * @see XenForo_ModerationQueueHandler_Abstract::approveModerationQueueEntry()
	 */
	public function approveModerationQueueEntry($contentId, $message, $title)
	{
		return true;
	}
	
	/**
	 * Approves the specified moderation queue entry.
	 *
	 * @see XenForo_ModerationQueueHandler_Abstract::approveModerationQueueEntry()
	 */
	public function approvedModerationQueueEntry($contentId, $note)
	{
		$dw = XenForo_DataWriter::create('UserEss_DataWriter_UNC', XenForo_DataWriter::ERROR_SILENT);
		$dw->setExistingData($contentId);
		
		$params = array(
			'status' => 'approved',
			'moderator_id' => XenForo_Visitor::getUserId(),
			'moderation_date' => XenForo_Application::$time
		);
		
		if (!empty($note))
			$params['moderator_note'] = $note; // note from the moderator to the user
		
		$dw->bulkSet($params);
		$dw->setOption(UserEss_DataWriter_UNC::OPTION_SET_IP_ADDRESS, false);
		
		if ($dw->save())
		{
			$unc = $dw->getMergedData();
			
			$writer = XenForo_DataWriter::create('XenForo_DataWriter_User', XenForo_DataWriter::ERROR_SILENT);
			$writer->setExistingData($unc['user_id']);
			$writer->set('username', $unc['to_username']);
			
			// set new custom title to display previous user name
			if (XenForo_Application::get('options')->uncCustomTitleDays)
			{
				$oldCustomTitle = $writer->getExisting('custom_title');
				$newCustomTitle = new XenForo_Phrase('useress_unc_custom_title', array('name' => $unc['from_username']));
				$writer->set('custom_title', $newCustomTitle);
			}
			
			if ($writer->save())
			{
				// REMOVE FROM QUEUE
				$this->_getModerationQueueModel()->deleteFromModerationQueue('unc', $contentId);
				
				// ALERT USER
				$this->_alertUser('approved', $unc);
				
				// ALERT ALL MODERATORS
				$this->_alertModerators('approved', $unc);
				
				// EMAIL USER
				// always send so user remembers which user name to log in with next time
				$this->_emailUser('approved', $unc);
				
				// UPDATE CUSTOM TITLE CRON
				if (XenForo_Application::get('options')->uncCustomTitleDays)
				{
					$db = XenForo_Application::get('db');
					$cronEntry = $db->fetchRow("SELECT * FROM useress_unc_cron WHERE user_id = ? LIMIT 1", $unc['user_id']);
					
					if (!empty($cronEntry)) // update
					{
						$cronValues = array(
							'date' => XenForo_Application::$time,
							'old_custom_title' => $cronEntry['old_custom_title'],
							'new_custom_title' => $newCustomTitle
						);
						
						$condition = 'user_id = ' . $db->quote($unc['user_id']);
						
						$db->update('useress_unc_cron', $cronValues, $condition);
					}
					else // insert
					{
						$cronValues = array(
							'user_id' => $unc['user_id'],
							'date' => XenForo_Application::$time,
							'old_custom_title' => $oldCustomTitle,
							'new_custom_title' => $newCustomTitle
						);
						
						$db->insert('useress_unc_cron', $cronValues);
					}
				}
				
				if (isset($unc['ip_address']))
					$unc['ip_address'] = ($unc['ip_address'] ? XenForo_Helper_Ip::convertIpBinaryToString($unc['ip_address']) : '');
				
				// LOG MODERATOR ACTION
				XenForo_Model_Log::logModeratorAction(
					'unc', $writer->getMergedData(),
					'approved',
					$unc
				);
				
				return true;
			}
			else
			{
				return false;
			}
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Delete the specified moderation queue entry.
	 *
	 * @see XenForo_ModerationQueueHandler_Abstract::deleteModerationQueueEntry()
	 */
	public function deleteModerationQueueEntry($contentId)
	{
		return true;
	}
	
	/**
	 * Denies the specified moderation queue entry.
	 *
	 * @see XenForo_ModerationQueueHandler_Abstract::deleteModerationQueueEntry()
	 */
	public function deniedModerationQueueEntry($contentId, $note)
	{
		$dw = XenForo_DataWriter::create('UserEss_DataWriter_UNC', XenForo_DataWriter::ERROR_SILENT);
		$dw->setExistingData($contentId);
		
		$params = array(
			'status' => 'denied',
			'moderator_id' => XenForo_Visitor::getUserId(),
			'moderation_date' => XenForo_Application::$time
		);
		
		if (!empty($note))
			$params['moderator_note'] = $note; // note from the moderator to the user
		
		$dw->bulkSet($params);
		$dw->setOption(UserEss_DataWriter_UNC::OPTION_SET_IP_ADDRESS, false);
		
		if ($dw->save())
		{
			$unc = $dw->getMergedData();
			$unc['username'] = $unc['from_username']; // instead of fetching the whole user table
			
			// REMOVE FROM QUEUE
			$this->_getModerationQueueModel()->deleteFromModerationQueue('unc', $contentId);
			
			// ALERT USER
			$this->_alertUser('denied', $unc);
			
			// EMAIL USER
			// always send so user may properly see moderator comments (when provided)
			$this->_emailUser('denied', $unc);
			
			if (isset($unc['ip_address']))
				$unc['ip_address'] = ($unc['ip_address'] ? XenForo_Helper_Ip::convertIpBinaryToString($unc['ip_address']) : '');
			
			// LOG MODERATOR ACTION
			XenForo_Model_Log::logModeratorAction(
				'unc', $unc,
				'denied',
				$unc
			);
			
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Alert the user of the action
	 *
	 * @param string $action - approved/denied
	 * @param array $unc - name change request details
	 */
	protected function _alertUser($action, array $unc)
	{
		if (empty($unc) || ($action != 'approved' && $action != 'denied'))
			return;
		
		$alertModel = $this->_getAlertModel();
		
				   //alert($alertUserId, $userId, $username, $contentType, $contentId, $action, array $extraData = null)
		$alertModel->alert($unc['user_id'], $unc['user_id'], $unc['from_username'], 'unc', $unc['unc_id'], $action);
	}
	
	/**
	 * Alert all moderators of the action
	 *
	 * @param string $action - approved/denied
	 * @param array $unc - name change request details
	 */
	protected function _alertModerators($action, array $unc)
	{
		if (empty($unc) || $action != 'approved')
			return;
		
		if (XenForo_Application::get('options')->uncModerateAlert)
		{
			$action = 'mod_' . $action;
			$alertModel = $this->_getAlertModel();
			$moderatorModel = $this->_getModeratorModel();
			$mods = $moderatorModel->getAllGeneralModerators();
			unset($mods[XenForo_Visitor::getUserId()]); // remove this moderator as he surely knows by now :-)
			
			foreach ($mods AS $mod)
			{
						   //alert($alertUserId, $userId, $username, $contentType, $contentId, $action, array $extraData = null)
				$alertModel->alert($mod['user_id'], $mod['user_id'], $mod['username'], 'unc', $unc['unc_id'], $action);			
			}
		}
	}
	
	/**
	 * Deletes the specified moderation queue entry.
	 *
	 * @param string $action - approved/denied
	 * @param array $unc - name change request details
	 */
	protected function _emailUser($action, array $unc)
	{
		if (empty($unc) || ($action != 'approved' && $action != 'denied'))
			return;
		
		$user = $this->_getUserModel()->getUserById($unc['user_id']);
		if (!empty($user) && !empty($user['email']))
		{
			$params = array(
				'user' => array('user_id' => $user['user_id'], 'username' => $unc['from_username']),
				'requestedName' => $unc['to_username'],
				'moderatorNote' => XenForo_Helper_String::censorString($unc['moderator_note']),
				'boardTitle' => XenForo_Application::get('options')->boardTitle,
				'boardUrl' => XenForo_Application::get('options')->boardUrl
			);
			
			$mail = XenForo_Mail::create('useress_unc_email_' . $action, $params, $user['language_id']);
			$mail->enableAllLanguagePreCache();
			$mail->queue($user['email'], $user['username']);
		}			
	}

	/**
	 * @return XenForo_Model_User
	 */
	protected function _getUserModel()
	{
		if (!isset($this->_userModel))
		{
			$this->_userModel = XenForo_Model::create('XenForo_Model_User');
		}
		
		return $this->_userModel;
	}
	
	/**
	 * @return UserEss_Model_UNC
	 */
	protected function _getUNCModel()
	{
		if (!isset($this->_uncModel))
		{
			$this->_uncModel = XenForo_Model::create('UserEss_Model_UNC');
		}
		
		return $this->_uncModel;
	}
		
	/**
	 * @return XenForo_Model_Alert
	 */
	protected function _getAlertModel()
	{
		if (!isset($this->_alertModel))
		{
			$this->_alertModel = XenForo_Model::create('XenForo_Model_Alert');
		}
		
		return $this->_alertModel;
	}
	
	/**
	 * @return XenForo_Model_Moderator
	 */
	protected function _getModeratorModel()
	{
		if (!isset($this->_moderatorModel))
		{
			$this->_moderatorModel = XenForo_Model::create('XenForo_Model_Moderator');
		}
		
		return $this->_moderatorModel;
	}
	
	/**
	 * @return XenForo_Model_ModerationQueue
	 */
	protected function _getModerationQueueModel()
	{
		if (!isset($this->_moderationQueueModel))
		{
			$this->_moderationQueueModel = XenForo_Model::create('XenForo_Model_ModerationQueue');
		}
		
		return $this->_moderationQueueModel;
	}
}