<?php

class Waindigo_InstallUpgrade_Model_InstallUpgrade extends XenForo_Model
{

    protected static $_skippedFiles = array();

    public function extractFromFile($fileName, $type = '')
    {
        if (!file_exists($fileName) || !is_readable($fileName)) {
            throw new XenForo_Exception(new XenForo_Phrase('please_enter_valid_file_name_requested_file_not_read'), true);
        }

        $fh = fopen($fileName, "r");

        $blob = fgets($fh, 5);

        if (strpos($blob, 'Rar') !== false) {
            $fileName = $this->_extractFromArchive($fileName, $type, 'Rar');
        } elseif (strpos($blob, 'PK') !== false) {
            $fileName = $this->_extractFromArchive($fileName, $type, 'Zip');
        }

        return $fileName;
    } /* END extractFromFile */

    protected function _extractFromArchive($source, $type, $adapter)
    {
        @set_time_limit(120);
        ignore_user_abort(true);
        XenForo_Application::get('db')->setProfiler(false);

        $destination = XenForo_Helper_File::getTempDir() . DIRECTORY_SEPARATOR . pathinfo($source, PATHINFO_FILENAME) .
             XenForo_Application::$time;

        try {
            XenForo_Helper_File::createDirectory($destination);
        } catch (Exception $e) {
            throw new XenForo_Exception(new XenForo_Phrase('waindigo_unable_to_create_temp_dir_installupgrade'), true);
        }

        $archive = new Zend_Filter_Decompress(
            array(
                'adapter' => $adapter,
                'options' => array(
                    'target' => $destination
                )
            ));

        $archive->filter($source);
        @unlink($source);

        $xmlFileNames = array();
        $noUnlink = array();
        if ($type) {
            $fileNames = $this->_sortXmlFiles($this->_getAllXmlFilesOfType($destination, $type));

            foreach ($fileNames as $xmlFileName) {
                $rootDir = XenForo_Application::getInstance()->getRootDir();
                $title = $this->_getTitleFromXmlFile($xmlFileName);
                if ($type == 'addon') {
                    $addOnId = $this->_getAddOnIdFromXmlFile($xmlFileName);
                    $fileName = 'install/data/addon-' . $addOnId . '.xml';
                } else {
                    $fileName = 'install/data/' . $type . '-' . $title . '.xml';
                }
                if (@copy($xmlFileName, $rootDir . '/' . $fileName)) {
                    $xmlFileNames[md5($fileName)] = array(
                        $fileName,
                        $title
                    );
                } else {
                    $xmlFileNames[md5($fileName)] = array(
                        $xmlFileName,
                        $title
                    );
                    $noUnlink[] = $xmlFileName;
                }
            }
        }

        $uploadDirectory = $this->_getBestGuessUploadDirectoryFromList($this->_getAllDirectories($destination));

        if ($uploadDirectory) {
            $this->_copyRecursively($uploadDirectory, XenForo_Application::getInstance()->getRootDir());
        }

        $this->_unlinkRecursively($destination, $noUnlink);

        if (count($xmlFileNames) == 1) {
            $xmlFileNames = reset($xmlFileNames);
            $xmlFileNames = $xmlFileNames[0];
        } elseif (empty($xmlFileNames) && $type) {
            throw new XenForo_Exception(new XenForo_Phrase('provided_file_was_not_valid_xml_file'), true);
        }

        return $xmlFileNames;
    } /* END _extractFromArchive */

    protected function _getAllDirectories($path)
    {
        $directory = new Waindigo_InstallUpgrade_Helper_SkipDotsRecursiveDirectoryIterator($path);
        $iterator = new RecursiveIteratorIterator($directory, RecursiveIteratorIterator::SELF_FIRST);

        $directories = array();
        foreach ($iterator as $path => $directory) {
            if ($directory->isDir()) {
                $directories[substr_count(str_replace('\\', '/', $path), '/')][] = $path;
            }
        }

        return $directories;
    } /* END _getAllDirectories */

    protected function _getAllXmlFilesOfType($path, $type = '')
    {
        if (substr($path, -1) != '/') {
            $path .= '/';
        }

        $xmlFiles = array();

        if ($type == 'kml') {
            $ext = 'kml';
        } else {
            $ext = 'xml';
        }

        $directory = new RecursiveDirectoryIterator($path);
        $iterator = new RecursiveIteratorIterator($directory);
        foreach (new RegexIterator($iterator, '/^.+\.' . $ext . '$/i', RecursiveRegexIterator::GET_MATCH) as $xmlFile) {
            $fileName = $xmlFile[0];
            if ($this->_getXmlType($fileName) == $type) {
                $xmlFiles[substr_count(str_replace('\\', '/', $fileName), '/')][] = $fileName;
            }
        }
        return $xmlFiles;
    } /* END _getAllXmlFilesOfType */

    protected function _getXmlType($fileName)
    {
        try {
            $xml = new SimpleXMLElement($fileName, 0, true);

            return (string) $xml->getName();
        } catch (Exception $e) {
            return false;
        }
    } /* END _getXmlType */

    protected function _getAddOnIdFromXmlFile($fileName)
    {
        try {
            $xml = new SimpleXMLElement($fileName, 0, true);

            return (string) $xml['addon_id'];
        } catch (Exception $e) {
            return false;
        }
    } /* END _getAddOnIdFromXmlFile */

    protected function _getTitleFromXmlFile($fileName)
    {
        try {
            $xml = new SimpleXMLElement($fileName, 0, true);

            return (string) $xml['title'];
        } catch (Exception $e) {
            return false;
        }
    } /* END _getTitleFromXmlFile */

    protected function _sortXmlFiles(array $xmlFiles)
    {
        ksort($xmlFiles);

        $sortedXmlFiles = array();
        foreach ($xmlFiles as $depth => &$depthXmlFiles) {
            foreach ($depthXmlFiles as $fileName) {
                $sortedXmlFiles[] = $fileName;
            }
        }

        return $sortedXmlFiles;
    } /* END _sortXmlFiles */

    protected function _getBestGuessUploadDirectoryFromList(array $directories)
    {
        $possibleUploadDirectories = $this->_getPossibleUploadDirectories($directories);
        $bestGuesses = reset($possibleUploadDirectories);

        if (!$bestGuesses) {
            return false;
        }

        return reset($bestGuesses);
    } /* END _getBestGuessUploadDirectoryFromList */

    protected function _getPossibleUploadDirectories($directories)
    {
        foreach ($directories as $level => $levelDirectories) {
            foreach ($levelDirectories as $key => $path) {
                $fileName = pathinfo($path, PATHINFO_FILENAME);
                if ($fileName == 'upload') {
                    continue;
                }
                if (in_array($fileName,
                    array(
                        'data',
                        'install',
                        'internal_data',
                        'js',
                        'library',
                        'styles'
                    ))) {
                    $directories[$level - 1][] = pathinfo($path, PATHINFO_DIRNAME);
                }
                unset($directories[$level][$key]);
            }
            $directories[$level] = array_unique($directories[$level]);
        }
        return array_filter($directories);
    } /* END _getPossibleUploadDirectories */

    protected function _copyRecursively($source, $destination)
    {
        foreach ($iterator = new RecursiveIteratorIterator(
            new Waindigo_InstallUpgrade_Helper_SkipDotsRecursiveDirectoryIterator($source),
            RecursiveIteratorIterator::SELF_FIRST) as $item) {
            @set_time_limit(120);
            if ($item->isDir()) {
                @mkdir($destination . DIRECTORY_SEPARATOR . $iterator->getSubPathName());
            } else {
                $fileName = $destination . DIRECTORY_SEPARATOR . $iterator->getSubPathName();
                if (!@copy($item, $fileName)) {
                    self::$_skippedFiles[md5($fileName)] = substr($fileName, strlen($destination) + 1);
                }
            }
        }
    } /* END _copyRecursively */

    protected function _unlinkRecursively($path, array $noUnlink)
    {
        foreach (new RecursiveIteratorIterator(
            new Waindigo_InstallUpgrade_Helper_SkipDotsRecursiveDirectoryIterator($path),
            RecursiveIteratorIterator::CHILD_FIRST) as $item) {
            @set_time_limit(120);
            $pathName = $item->getPathName();
            if (!in_array($pathName, $noUnlink)) {
                if ($item->isFile()) {
                    @unlink($pathName);
                } else {
                    @rmdir($pathName);
                }
            }
        }
        @rmdir($path);
    } /* END _unlinkRecursively */

    public function getSkippedFiles()
    {
        return self::$_skippedFiles;
    } /* END getSkippedFiles */

    public function addCredentials($username, $password, $host, $path)
    {
        $uniqueKey = md5($host . $path);

        $this->_getDb()->query(
            '
            INSERT INTO xf_install_upgrade_login
            (username, password, host, path, unique_key)
            VALUES (?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE username = VALUES(username), password = VALUES(password)
        ',
            array(
                $username,
                $password,
                $host,
                $path,
                $uniqueKey
            ));
    } /* END addCredentials */

    public function getCredentials($fileName)
    {
        $host = parse_url($fileName, PHP_URL_HOST);

        $credentials = $this->fetchAllKeyed(
            '
            SELECT *
            FROM xf_install_upgrade_login
            WHERE host = ?
        ', 'login_id', $host);

        $selectedCredential = array(
            'username' => '',
            'password' => '',
            'path' => ''
        );

        foreach ($credentials as $credential) {
            if (strlen($credential['path']) < strlen($selectedCredential['path'])) {
                continue;
            }
            if (strpos($fileName, $host . $credential['path']) !== false) {
                $selectedCredential = $credential;
            }
        }

        return array(
            $selectedCredential['username'],
            $selectedCredential['password']
        );
    } /* END getCredentials */

    public function checkForUrl(&$options = array(), &$errorPhraseKey = '')
    {
        @set_time_limit(120);
        $options = array_merge(
            array(
                'original_server_file' => '',
                'server_file' => '',
                'status' => '',
                'redirect' => '',
                'post_params' => array(),
                'username' => '',
                'password' => '',
                'auth_type' => '',
                'store_credentials' => false,
                'cookie_path' => '',
                'return' => '',
                'headers' => array(),
                'redirects' => -1
            ), $options);

        $options['redirects']++;

        if (filter_var($options['server_file'], FILTER_VALIDATE_URL)) {
            $errors = array();

            $client = XenForo_Helper_Http::getClient($options['server_file']);
            if (isset($options['cookie_jar'])) {
                $client->setCookieJar($options['cookie_jar']);
            }
            if ($options['status'] == 401) {
                if ($options['auth_type'] == 'basic') {
                    $client->setAuth($options['username'], $options['password'], Zend_Http_Client::AUTH_BASIC);
                }
            } elseif ($options['status'] == 403) {
                $client->setCookieJar();
                $client->setParameterPost(
                    array(
                        'login' => $options['username'],
                        'username' => $options['username'],
                        'email' => $options['username'],
                        'password' => $options['password'],
                        'redirect' => $options['redirect']
                    ));
            } elseif (!empty($options['post_params'])) {
                $client->setParameterPost($options['post_params']);
            }

            if ($options['status'] != 403 && empty($options['post_params'])) {
                $head = $client->request('HEAD');
                $options['status'] = $head->getStatus();
            } else {
                $response = $client->request('POST');
                $options['status'] = $response->getStatus();
            }

            $this->_storeCredentials($client, $options);

            if ($options['status'] == 200) {
                $contentType = $client->getLastResponse()->getHeader('Content-Type');
                if (stripos($contentType, 'text/html') !== false) {
                    return $this->_processHtmlResponse($client, $options, $errorPhraseKey);
                } else {
                    return $this->_processNonHtmlResponse($client, $options, $errorPhraseKey);
                }
            }
            if ($options['status'] == 401 || $options['status'] == 403) {
                return $this->_processNoAuthResponse($client, $options, $errorPhraseKey);
            }
        }
    } /* END checkForUrl */

    protected function _processHtmlResponse(Zend_Http_Client $client, &$options = array(), &$errorPhraseKey = '')
    {
        $response = $client->getLastResponse();

        if (!$response->getRawBody()) {
            if (!empty($options['post_params'])) {
                $response = $client->request('POST');
            } else {
                $response = $client->request('GET');
            }
        }

        $body = $response->getBody();

        $downloadUrl = $this->_getDownloadUrlFromContent($body, $options['server_file']);
        if (!$downloadUrl && $options['redirect'] != $options['server_file']) {
            $downloadUrl = $options['redirect'];
        }
        if ($downloadUrl) {
            // TODO: check if hosts match
            // TODO: check if we have already visited this URL
            $options['server_file'] = $downloadUrl;
            $options['status'] = '';
            $options['cookie_jar'] = $client->getCookieJar();
            return $this->checkForUrl($options, $errorPhraseKey);
        }

        $loginUrl = $this->_getLoginUrlFromLoginForm($body, $options['server_file']);
        if ($loginUrl) {
            $options['status'] = 403;
            return $this->_processNoAuthResponse($client, $options, $errorPhraseKey);
        }

        if ($options['return'] == 'body') {
            return $body;
        }

        $errorPhraseKey = 'waindigo_no_resource_found_at_specified_url_installupgrade';
        return false;
    } /* END _processHtmlResponse */

    protected function _processNonHtmlResponse(Zend_Http_Client $client, &$options = array(), &$errorPhraseKey = '')
    {
        $response = $client->getLastResponse();

        if ($options['return'] == 'headers') {
            return $response->getHeaders();
        }

        if (!$response->getRawBody()) {
            if (!empty($options['post_params'])) {
                $response = $client->request('POST');
            } else {
                $response = $client->request('GET');
            }
        }

        $tempFile = tempnam(XenForo_Helper_File::getTempDir(), md5($options['server_file']));
        $fp = fopen($tempFile, 'w');
        if (isset($response)) {
            fwrite($fp, $response->getRawBody());
        } else {
            fwrite($fp, $client->request('GET')->getRawBody());
        }
        fclose($fp);

        $options['server_file'] = $tempFile;

        return true;
    } /* END _processNonHtmlResponse */

    protected function _processNoAuthResponse(Zend_Http_Client $client, &$options = array(), &$errorPhraseKey = '')
    {
        $response = $client->getLastResponse();

        $options['host'] = parse_url($options['server_file'], PHP_URL_HOST);
        if ($options['status'] == 401) {
            if (isset($options['username']) || isset($options['password'])) {
                $options['errors'][] = new XenForo_Phrase('incorrect_password');
            }
            $authenticate = $response->getHeader('WWW-Authenticate');
            preg_match('#(Basic) realm="(.*)"#s', $authenticate, $matches);
            if (isset($matches[1])) {
                $options['auth_type'] = 'basic';
            }
            if (isset($matches[2])) {
                $options['realm'] = $matches[2];
            }
        } elseif ($options['status'] == 403) {
            $body = $client->request('GET')->getBody();
            $loginUrl = $this->_getLoginUrlFromLoginForm($body, $options['server_file']);
            if ($options['redirect']) {
                if (isset($options['username']) || isset($options['password'])) {
                    if (!$loginUrl && !$options['redirects']) {
                        $options['server_file'] = $options['redirect'];
                        return $this->checkForUrl($options, $errorPhraseKey);
                    }
                    $options['errors'][] = new XenForo_Phrase('incorrect_password');
                }
            } else {
                $options['redirect'] = $options['server_file'];
                $options['server_file'] = $loginUrl;
            }
        }
        if (!$options['username'] || !$options['password']) {
            list($username, $password) = $this->getCredentials($options['server_file']);
            if ($username) {
                $options['username'] = $username;
                $options['password'] = $password;
                if (XenForo_Application::get('options')->waindigo_installUpgrade_useStoredCredentials) {
                    return $this->checkForUrl($options, $errorPhraseKey);
                }
            }
        }
        if ($options['status'] == 403 && $options['redirect'] == $options['server_file'] &&
             $loginUrl != $options['server_file'] && !$options['redirects']) {
            $options['redirect'] = $options['server_file'];
            $options['server_file'] = $loginUrl;
            return $this->checkForUrl($options, $errorPhraseKey);
        }
        $errorPhraseKey = 'do_not_have_permission';
        return false;
    } /* END _processNoAuthResponse */

    protected function _getLoginUrlFromLoginForm($body, $fileName)
    {
        preg_match('#<form[^>]*action="([^"]*)"[^>]*id="pageLogin"[^>]*>#Us', $body, $matches);
        if (isset($matches[1])) {
            return $this->_getBasedUrlFromContent($body, $fileName, $matches[1]);
        }
        preg_match_all('#<form[^>]*action="([^"]*)"[^>]*>#Us', $body, $matches);
        if (!empty($matches[1])) {
            foreach ($matches[1] as $match) {
                if (strpos($match, 'login') !== false) {
                    return $this->_getBasedUrlFromContent($body, $fileName, $match);
                }
            }
        }

        return false;
    } /* END _getLoginUrlFromLoginForm */

    protected function _getDownloadUrlFromContent($content, $currentUrl)
    {
        $dom = new Zend_Dom_Query($content);
        $downloadButtons = $dom->query('label.downloadButton a');
        if ($downloadButtons->count()) {
            foreach ($downloadButtons as $downloadButton) {
                $href = $downloadButton->getAttribute('href');
                if ($href != $currentUrl) {
                    return $this->_getBasedUrlFromContent($content, $currentUrl, $href);
                }
            }
        }
    } /* END _getDownloadUrlFromContent */

    protected function _getBasedUrlFromContent($content, $currentUrl, $link)
    {
        if (filter_var($link, FILTER_VALIDATE_URL)) {
            return $link;
        }
        preg_match('#<base[^>]*href="([^"]*)"[^>]*>#Us', $content, $matches);
        if (isset($matches[1])) {
            return $matches[1] . $link;
        }

        $parsedUrl = parse_url($currentUrl);
        $baseUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
        if (strlen($link) && substr($link, 0, 1) == '/') {
            return $baseUrl;
        }

        return $baseUrl . '/' . pathinfo($parsedUrl['path'], PATHINFO_DIRNAME);
    } /* END _getBasedUrlFromContent */

    /**
     *
     * @param Zend_Http_Client $client
     * @param string $fileName
     */
    protected function _storeCredentials(Zend_Http_Client $client, &$options = array())
    {
        if (!$options['store_credentials']) {
            return false;
        }

        $path = $options['cookie_path'];
        if (!$path) {
            $path = '/';
        }

        if ($client->getCookieJar()) {
            $cookieJar = $client->getCookieJar();
            $cookies = $cookieJar->getAllCookies();
            foreach ($cookies as $cookie) {
                /* @var $cookie Zend_Http_Cookie */
                if (strpos($options['server_file'], $cookie->getPath()) !== false &&
                     strlen($cookie->getPath()) > strlen($path)) {
                    $path = $cookie->getPath();
                }
            }
        }

        $username = $options['username'];
        $password = $options['password'];
        $host = parse_url($options['server_file'], PHP_URL_HOST);

        $this->addCredentials($username, $password, $host, $path);
    } /* END _storeCredentials */

    public function checkForAddOnUpdates(array &$addOns, &$errorString = null)
    {
        $xenOptions = XenForo_Application::get('options');

        if (!$xenOptions->waindigo_installUpgrade_checkForOutdatedAddOns) {
            return false;
        }

        if (!$xenOptions->waindigo_installUpgrade_uniqueIdentifier) {
            $uniqueIdentifier = XenForo_Application::generateRandomString(30);
            $dw = XenForo_DataWriter::create('XenForo_DataWriter_Option');
            $dw->setExistingData('waindigo_installUpgrade_uniqueIdentifier');
            $dw->set('option_value', $uniqueIdentifier);
            $dw->save();
            XenForo_Application::get('options')->set('waindigo_installUpgrade_uniqueIdentifier', $uniqueIdentifier);
        }

        /* @var $optionModel XenForo_Model_Option */
        $optionModel = $this->getModelFromCache('XenForo_Model_Option');

        $installUpgradeOptions = $optionModel->getOptionsByAddOn('Waindigo_InstallUpgrade');
        if (XenForo_Application::$versionId > 1020000) {
            $installUpgradeOptionValues = XenForo_Application::arrayColumn($installUpgradeOptions, 'option_value',
                'option_id');
        } else {
            $installUpgradeOptionValues = Waindigo_InstallUpgrade_Application::arrayColumn($installUpgradeOptions,
                'option_value', 'option_id');
        }

        try {
            $checker = XenForo_Helper_Http::getClient('https://waindigo.org/updates/index.json');
            $checker->setParameterPost('addons', $addOns);
            $checker->setParameterPost('board_url', $xenOptions->boardUrl);
            $checker->setParameterPost('current_version_id', $xenOptions->currentVersionId);
            $checker->setParameterPost('options', $installUpgradeOptionValues);
            $checker->setParameterPost('debug_mode', XenForo_Application::debugMode());
            $checkerResponse = $checker->request('POST');

            if (!$checkerResponse || $checkerResponse->getStatus() != 200) {
                $errorString = 'Error with Waindigo update server';
                return false;
            }
        } catch (Zend_Http_Client_Exception $e) {
            $errorString = 'Connection to Waindigo failed';
            return false;
        }

        $body = $checkerResponse->getBody();
        $updatedAddOnIds = array();
        if ($body) {
            $responseArray = json_decode($body);

            if (!empty($responseArray->addOnIds)) {
                $updatedAddOnIds = $responseArray->addOnIds;
            }
            if (!empty($responseArray->errorString)) {
                $errorString = $responseArray->errorString;
            }
        }

        $this->_getDb()->query('
                UPDATE xf_addon SET install_upgrade_updated = 0
            ');

        if (!empty($updatedAddOnIds)) {
            $this->_getDb()->query(
                '
                UPDATE xf_addon SET install_upgrade_updated = 1
                WHERE addon_id IN (' . $this->_getDb()
                    ->quote($updatedAddOnIds) . ')
            ');

            foreach ($addOns as $addOnId => $addOn) {
                if (in_array($addOnId, $updatedAddOnIds)) {
                    $addOns[$addOnId]['install_upgrade_updated'] = 1;
                } else {
                    $addOns[$addOnId]['install_upgrade_updated'] = 0;
                }
            }
        }

        return true;
    } /* END checkForAddOnUpdates */

    public function validateLicenseToken($licenseValidationToken)
    {
        $xenOptions = XenForo_Application::get('options');

        $domain = parse_url($xenOptions->boardUrl, PHP_URL_HOST);

        try {
            $checker = XenForo_Helper_Http::getClient('http://xenforo.com/api/license-lookup.json');
            $checker->setParameterPost('token', $licenseValidationToken);
            $checker->setParameterPost('domain', $domain);
            $checkerResponse = $checker->request('POST');

            if (!$checkerResponse || $checkerResponse->getStatus() != 200) {
                return false;
            }
        } catch (Zend_Http_Client_Exception $e) {
            return false;
        }

        $body = $checkerResponse->getBody();
        if (!$body) {
            return false;
        }

        $responseObject = json_decode($body);

        if (empty($responseObject->is_valid)) {
            return false;
        }

        if (empty($responseObject->domain_match) && $domain != 'localhost') {
            return false;
        }

        return true;
    } /* END validateLicenseToken */
}