<?php

/**
 * Helper for forum, thread, and post related pages.
 * Provides validation methods, amongst other things.
 */
class Waindigo_SocialGroups_ControllerHelper_SocialForumThreadPost extends XenForo_ControllerHelper_ForumThreadPost
{
	protected $_thread = array();
	
	/**
	 * Checks that a forum is valid and viewable, before returning the forum's info.
	 *
	 * @param integer|string $forumIdOrName ID or node name of forum
	 * @param array $fetchOptions Extra data to fetch wtih the forum
	 *
	 * @return array Forum info
	 */
	public function assertForumValidAndViewable($forumIdOrName, array $fetchOptions = array())
	{
		$forum = parent::assertForumValidAndViewable($forumIdOrName, $fetchOptions);
		
		$visitor = XenForo_Visitor::getInstance();
		
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();

		/* @var $socialForumModel Waindigo_SocialGroups_Model_SocialForum */
		$socialForumModel = XenForo_Model::create('Waindigo_SocialGroups_Model_SocialForum');
		
		if (isset($socialForum['social_forum_id'])) {
			$forum['social_forum_title'] = $socialForum['title'];
			$forum['social_forum_id'] = $socialForum['social_forum_id'];
			$nodePermissions = $socialForumModel->getNodePermissions($socialForum->toArray(), $socialForum->getSocialForumMembers());
			$this->_visitor->setNodePermissions($forum['node_id'], $nodePermissions);
		}
		return $forum;
	}
	
	/**
	* Checks that a thread is valid and viewable, before returning the thread
	* and containing forum's info.
	*
	* @param integer $threadId
	* @param array $threadFetchOptions Extra data to fetch with the thread
	* @param array $forumFetchOptions Extra data to fetch wtih the forum
	*
	* @return array Format: [0] => thread info, [1] => forum info
	*/
	public function assertThreadValidAndViewable($threadId,
	array $threadFetchOptions = array(), array $forumFetchOptions = array()
	)
	{
		$thread = $this->getThreadOrError($threadId, $threadFetchOptions);

		if ($thread['social_forum_id'])
		{
			$socialForum = Waindigo_SocialGroups_SocialForum::setup($thread['social_forum_id']);
			$this->_thread['social_forum_user_id'] = $socialForum['user_id'];
		}
		
		return parent::assertThreadValidAndViewable($threadId, $threadFetchOptions, $forumFetchOptions);
	}
	
	/**
	* Gets the specified thread or throws an error.
	*
	* @param integer $threadId
	* @param array $fetchOptions Options that control the data fetched with the thread
	*
	* @return array
	*/
	public function getThreadOrError($threadId, array $fetchOptions = array())
	{
		if (!$this->_thread)
		{
			$this->_thread = parent::getThreadOrError($threadId, $fetchOptions);
		}
		return $this->_thread;
	}
	
	/**
	 * Gets the breadcrumbs that relate to the specified node, including
	 * the supplied node.
	 *
	 * @param array $forum Information about the node
	 * @param boolean $includeSelf Whether to include this node in the bread crumbs
	 *
	 * @return unknown_type
	 */
	public function getNodeBreadCrumbs(array $forum, $includeSelf = true)
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		if (isset($socialForum['social_forum_id']))
		{
			$nodeBreadCrumbs = $this->_controller->getModelFromCache('XenForo_Model_Node')->getNodeBreadCrumbs($forum, true);
	
			if ($includeSelf)
			{
				$nodeBreadCrumbs[] = array(
					'href' => XenForo_Link::buildPublicLink('social-forums', $socialForum),
					'value' => $socialForum['title'],
				);
			}
			return $nodeBreadCrumbs;
		}
		return parent::getNodeBreadCrumbs($forum, $includeSelf);
	}
	
}