<?php

/**
 * Controller for handling actions on social forums.
 */
class Waindigo_SocialGroups_ControllerPublic_SocialForum extends XenForo_ControllerPublic_Forum
{
	/**
	 * Method designed to be overridden by child classes to add pre-dispatch
	 * behaviors. This method should only be overridden by specific, concrete
	 * controllers.
	 *
	 * @param string Action that is requested
	 */
	protected function _preDispatch($action)
	{
		if ($socialForumId = $this->_input->filterSingle('social_forum_id', XenForo_Input::UINT))
		{
			if ($socialForum = Waindigo_SocialGroups_SocialForum::setup($socialForumId))
			{
				$this->_request->setParam('node_id', $socialForum['node_id']);
			}
		}
	}
	
	protected function _postDispatch($controllerResponse, $controllerName, $action)
	{
		parent::_postDispatch($controllerResponse, $controllerName, $action);
	
		if (class_exists('NodesAsTabs_Listen') && method_exists('NodesAsTabs_Listen', 'breadCrumbs'))
		{
			try {
				$optionsModel = XenForo_Model::create('NodesAsTabs_Model_Options');
				$routeMatch = $this->getRouteMatch();
				$request = $this->getRequest();
				$viewParams = (isset($controllerResponse->params) ? $controllerResponse->params : array());
	
				// DO ROUTE STUFF
				// THIS IS DONE IN CONTROLLER INSTEAD OF ROUTER
				// SO WE CAN ACCESS NODE RECORD TO SAVE A QUERY
	
				$nodeId = 0;
				if (!empty($viewParams['forum']['node_id']))
				$nodeId = $viewParams['forum']['node_id'];
				$nodeId = ($nodeId ? $nodeId : $optionsModel->getNodeIdFromRequest($request));
	
				$nodeTabId = $optionsModel->handleRoute($nodeId, $routeMatch);
	
				// USED LATER FOR BREADCRUMBS
				$controllerResponse->containerParams['nodeTabId'] = $nodeTabId;
			} catch (Exception $e) {
				// do nothing
			}
		}
	}
	
	public function actionIndex()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
	
		$visitor = XenForo_Visitor::getInstance();
		$response = parent::actionIndex();
		
		if ($this->_routeMatch->getResponseType() == 'rss')
		{
			return $response;	
		}
		
		if (get_class($response) == "XenForo_ControllerResponse_View")
		{
			if (XenForo_Application::get('options')->waindigo_socialGroups_showSidebar)
			{
				$response->templateName = 'waindigo_social_forum_view_social_groups';
			}
			$response->viewName = 'Waindigo_SocialGroups_ViewPublic_SocialForum_View';

			$viewParams =& $response->params;
			$forum =& $viewParams['forum'];
			$forum['description'] = '';
			$forum['social_forum_id'] = $socialForum['social_forum_id'];
			$forum['title'] = $socialForum['title'];
			$viewParams['socialForum'] = $socialForum->toArray();
			
			$viewParams['canPostEvent'] = XenForo_Permission::hasPermission($visitor['permissions'], 'EWRatendo', 'canPost')
				&& in_array($forum['node_id'], XenForo_Application::get('options')->EWRatendo_eventforums);
			$viewParams['canViewSocialForumMembers'] = $this->_getForumModel()->canViewSocialForumMembers($socialForum);
			$viewParams['canJoinSocialForum'] = $this->_getForumModel()->canJoinSocialForum($socialForum);
			$viewParams['canLeaveSocialForum'] = $this->_getForumModel()->canLeaveSocialForum($socialForum);
			$viewParams['canEditSocialForum'] = $this->_getForumModel()->canEditSocialForum($socialForum);
			$viewParams['canMoveSocialForum'] = $this->_getForumModel()->canMoveSocialForum($socialForum);
			$viewParams['canDeleteSocialForum'] = $this->_getForumModel()->canDeleteSocialForum($socialForum);
			$viewParams['canApproveSocialForumJoinRequest'] = $this->_getForumModel()->canApproveSocialForumJoinRequest($socialForum);
				
			$viewParams['onlineUsers']['records'] = $this->_getSocialForumMemberModel()->getSocialForumUsers(array(
				'social_forum_id' => $socialForum['social_forum_id'],
			));
			
			$viewParams['moderatorCount'] = 0;
			$viewParams['memberCount'] = 0;
			$viewParams['unapprovedCount'] = 0;
			foreach ($viewParams['onlineUsers']['records'] as $member)
			{
				if ($member['is_invited'])
				{
					
				}
				else if ($member['is_approved'])
				{
					if ($member['is_social_forum_moderator'])
					{
						$viewParams['moderatorCount']++;
					}
					else
					{
						$viewParams['memberCount']++;
					}
				}
				else
				{
					$viewParams['unapprovedCount']++;
				}							
			}
				
			if (array_key_exists($visitor['user_id'], $viewParams['onlineUsers']['records']))
			{
				$viewParams['member'] = $viewParams['onlineUsers']['records'][$visitor['user_id']];
			}
				
			// if we've read everything on the first page of a normal sort order, probably need to mark as read
			if ($visitor['user_id'] && $viewParams['page'] == 1 && !$viewParams['displayConditions']
				&& $viewParams['order'] == 'last_post_date' && $viewParams['orderDirection'] == 'desc'
				&& $socialForum['social_forum_read_date'] < $socialForum['last_post_date']
				)
			{
				$hasNew = false;
				foreach ($viewParams['threads'] AS $thread)
				{
					if ($thread['isNew'] && !$thread['isIgnored'])
					{
						$hasNew = true;
						break;
					}
				}
	
				if (!$hasNew)
				{
					// everything read, but forum not marked as read. Let's check.
					$this->_getForumModel()->markSocialForumReadIfNeeded($socialForum->toArray());
				}
			}
		}
		
		return $response;
	}
	
	public function actionEdit()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		$member = $socialForum->getMember();
		
		$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);
			
		$ftpHelper = $this->getHelper('ForumThreadPost');
		$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
				$forumId,
				$this->_getForumFetchOptions()
		);

		$this->_assertCanEditSocialForum($socialForum);
		
		$viewParams = array(
			'socialForum' => $socialForum,
			'forum' => $forum,
			'nodeBreadCrumbs' => $ftpHelper->getNodeBreadCrumbs($forum),
			'member' => $member,
		);
		return $this->responseView('Waindigo_SocialGroups_ViewPublic_SocialForum_Edit', 'waindigo_social_forum_edit_social_groups', $viewParams);
	}

	public function actionSave()
	{
		$this->_assertPostOnly();
	
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		$member = $socialForum->getMember();
		
		$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);
			
		$ftpHelper = $this->getHelper('ForumThreadPost');
		$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
				$forumId,
				$this->_getForumFetchOptions()
		);

		$this->_assertCanEditSocialForum($socialForum);
		
		$input = $this->_input->filter(array(
			'title' => XenForo_Input::STRING,
			'social_forum_open' => XenForo_Input::UINT,
			'social_forum_moderated' => XenForo_Input::UINT,
		));
	
		$input['description'] = $this->getHelper('Editor')->getMessageText('description', $this->_input);
		$input['description'] = XenForo_Helper_String::autoLinkBbCode($input['description']);

		// note: assumes that the message dw will pick up the username issues
		$writer = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForum');
		$writer->setExistingData($socialForum['social_forum_id']);
		$writer->bulkSet($input);
		$writer->save();
	
		$socialForum = $writer->getMergedData();

		$return = XenForo_Link::buildPublicLink('social-forums', $socialForum);

		return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				$return,
				new XenForo_Phrase('waindigo_your_social_forum_has_been_updated_social_groups')
		);
	}
	
	/**
	 * Deletes an existing social forum.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionDelete()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		$member = $socialForum->getMember();

		$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);

		$ftpHelper = $this->getHelper('ForumThreadPost');
		$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
				$forumId,
				$this->_getForumFetchOptions()
		);

		$fetchElements = $this->_getThreadFetchElements($forum, array());
		$threadFetchConditions = $fetchElements['conditions'];
		
		$totalThreads = $this->_getThreadModel()->countThreadsInForum($forumId, $threadFetchConditions);
		
		$this->_assertCanDeleteSocialForum($socialForum);

		if ($totalThreads) // must delete threads first
		{
			return $this->responseError(new XenForo_Phrase('waindigo_social_forum_not_empty_social_groups'));
		}
		else if ($this->isConfirmedPost()) // delete the social forum
		{
			/* @var $dw Waindigo_SocialGroups_DataWriter_SocialForum */
			$dw = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForum');
			
			$dw->setExistingData($socialForum['social_forum_id']);
			
			$dw->delete();

			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('social-categories', $forum)
			);
		}
		else // show a deletion confirmation dialog
		{
			$viewParams = array(
				'forum' => $forum,
				'nodeBreadCrumbs' => $ftpHelper->getNodeBreadCrumbs($forum),
				'socialForum' => $socialForum,
			);

			return $this->responseView(
				'Waindigo_SocialGroups_ViewPublic_SocialForum_Delete',
				'waindigo_social_forum_delete_social_groups',
				$viewParams
			);
		}
	}
	
	/**
	 * Moves a social forum to a different social category.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */	
	public function actionMove()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();

		$this->_assertCanMoveSocialForum($socialForum);
		
		$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);

		$ftpHelper = $this->getHelper('ForumThreadPost');
		$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
				$forumId,
				$this->_getForumFetchOptions()
		);
		
		if ($this->isConfirmedPost()) // move the thread
		{
			$input = $this->_input->filter(array(
				'move_to_node_id' => XenForo_Input::UINT,
	
//				'create_redirect' => XenForo_Input::STRING,
//				'redirect_ttl_value' => XenForo_Input::UINT,
//				'redirect_ttl_unit' => XenForo_Input::STRING
			));
		
			$viewableNodes = $this->getModelFromCache('XenForo_Model_Node')->getViewableNodeList();
			if (isset($viewableNodes[$input['move_to_node_id']]))
			{
				$targetNode = $viewableNodes[$input['move_to_node_id']];
			}
			else
			{
				return $this->responseNoPermission();
			}
		
/*			if ($input['create_redirect'] == 'permanent')
			{
				$options = array('redirect' => true, 'redirectExpiry' => 0);
			}
			else if ($input['create_redirect'] == 'expiring')
			{
				$expiryDate = strtotime('+' . $input['redirect_ttl_value'] . ' ' . $input['redirect_ttl_unit']);
				$options = array('redirect' => true, 'redirectExpiry' => $expiryDate);
			}
			else
			{
				$options = array('redirect' => false);
			}*/
			
			XenForo_Db::beginTransaction();
			$dw = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForum');
			$dw->setExistingData($socialForum['social_forum_id']);
			$dw->set('node_id', $input['move_to_node_id']);
			$dw->save();
			$socialForum = $dw->getMergedData();
			$this->_getForumModel()->moveThreads($socialForum);
			XenForo_Db::commit();
			
			XenForo_Model_Log::logModeratorAction('social_forum', $socialForum, 'move', array('from' => $forum['title']));
		
//			if ($dw->isChanged('node_id') && $options['redirect'])
//			{
//				$this->getModelFromCache('XenForo_Model_ThreadRedirect')->createRedirectThread(
//					XenForo_Link::buildPublicLink('threads', $thread), $thread,
//					"thread-$thread[thread_id]-$thread[node_id]-", $options['redirectExpiry']
//				);
//			}
		
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('social-forums', $socialForum)
			);
		}
		else
		{
			return $this->responseView(
				'Waindigo_SocialGroups_ViewPublic_SocialForum_Move',
				'waindigo_social_forum_move_social_groups',
				array(
					'socialForum' => $socialForum,
					'forum' => $forum,
					'nodeBreadCrumbs' => $ftpHelper->getNodeBreadCrumbs($forum),

					'nodeOptions' => $this->getModelFromCache('XenForo_Model_Node')->getViewableNodeList(),
				)
			);
		}
	}
		
	public function actionAvatar()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		
		$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);
			
		$ftpHelper = $this->getHelper('ForumThreadPost');
		$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
				$forumId,
				$this->_getForumFetchOptions()
		);
		
		$this->_assertCanEditSocialForum($socialForum);
			
		$maxWidth = Waindigo_SocialGroups_Model_SocialForumAvatar::getSizeFromCode('m');
		
		$viewParams = array(
				'socialForum' => $socialForum->toArray(),
				'sizeCode' => 'm',
				'maxWidth' => $maxWidth,
				'maxDimension' => ($socialForum['logo_width'] > $socialForum['logo_height'] ? 'height' : 'width'),
				'width' => $socialForum['logo_width'],
				'height' => $socialForum['logo_height'],
				'cropX' => $socialForum['logo_crop_x'],
				'cropY' => $socialForum['logo_crop_y'],
		);
	
		return $this->responseView('Waindigo_SocialGroups_ViewPublic_SocialForum_Avatar', 'waindigo_social_forum_avatar_social_groups', $viewParams);
	}

	public function actionAvatarUpload()
	{
		$this->_assertPostOnly();

		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		
		$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);
			
		$ftpHelper = $this->getHelper('ForumThreadPost');
		$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
				$forumId,
				$this->_getForumFetchOptions()
		);
		
		$this->_assertCanEditSocialForum($socialForum);
	
		$avatar = XenForo_Upload::getUploadedFile('avatar');
	
		/* @var $avatarModel Waindigo_SocialGroups_Model_SocialForumAvatar */
		$avatarModel = $this->getModelFromCache('Waindigo_SocialGroups_Model_SocialForumAvatar');
		
		$inputData = $this->_input->filter(array(
				'delete' => XenForo_Input::UINT,
		));

		$inputData['logo_crop_x'] = $this->_input->filterSingle('avatar_crop_x', XenForo_Input::UINT);
		$inputData['logo_crop_y'] = $this->_input->filterSingle('avatar_crop_y', XenForo_Input::UINT);
				
		// upload new avatar
		if ($avatar)
		{
			$avatarData = $avatarModel->uploadAvatar($avatar, $socialForum['social_forum_id'], XenForo_Visitor::getInstance()->getNodePermissions($socialForum['node_id']));
		}
		// delete avatar
		else if ($inputData['delete'])
		{
			$avatarData = $avatarModel->deleteAvatar($socialForum['social_forum_id']);
		}
		// re-crop avatar thumbnail
		else if ($inputData['logo_crop_x'] != $socialForum['logo_crop_x'] || $inputData['logo_crop_y'] != $socialForum['logo_crop_y'])
		{
			$avatarData = $avatarModel->recropAvatar($socialForum['social_forum_id'], $inputData['logo_crop_x'], $inputData['logo_crop_y']);
		}
		
		// merge new data into $socialForum, if there is any
		if (isset($avatarData) && is_array($avatarData))
		{
			foreach ($avatarData AS $key => $val)
			{
				$socialForum[$key] = $val;
			}
		}
		
		$message = new XenForo_Phrase('upload_completed_successfully');
		
		// return a view if noredirect has been requested and we are not deleting
		if ($this->_noRedirect())
		{
			return $this->responseView(
					'Waindigo_SocialGroups_ViewPublic_SocialForum_AvatarUpload',
					'',
					array(
							'socialForum' => $socialForum->toArray(),
							'sizeCode' => 'm',
							'maxWidth' => Waindigo_SocialGroups_Model_SocialForumAvatar::getSizeFromCode('m'),
							'maxDimension' => ($socialForum['logo_width'] > $socialForum['logo_height'] ? 'height' : 'width'),
							'width' => $socialForum['logo_width'],
							'height' => $socialForum['logo_height'],
							'cropX' => $socialForum['logo_crop_x'],
							'cropY' => $socialForum['logo_crop_y'],
							'social_forum_id' => $socialForum['social_forum_id'],
							'logo_date' => $socialForum['logo_date'],
							'message' => $message
					)
			);
		}
		else
		{
			return $this->responseRedirect(
					XenForo_ControllerResponse_Redirect::SUCCESS,
					XenForo_Link::buildPublicLink('account/personal-details'),
					$message
			);
		}		
	}	
	
	public function actionModerator()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		$member = $socialForum->getMember();
		
		$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);
			
		$ftpHelper = $this->getHelper('ForumThreadPost');
		$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
				$forumId,
				$this->_getForumFetchOptions()
		);
			
		$userId = $this->_input->filterSingle('user_id', XenForo_Input::UINT);
		
		$writer = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForumMember');
		$socialForumMember = $this->_getSocialForumMemberModel()->getSocialForumMemberByUserId($socialForum['social_forum_id'], $userId);
		if ($socialForumMember)
		{
			$writer->setExistingData($socialForumMember['social_forum_member_id']);
			if ($socialForumMember['is_social_forum_moderator'])
			{
				$this->_assertCanRemoveSocialForumModerator($socialForum);
				if (XenForo_Model_Alert::userReceivesAlert($socialForumMember, 'social_forum', 'remove_moderator'))
				{
					$visitor = XenForo_Visitor::getInstance();
					XenForo_Model_Alert::alert(
							$userId,
							$visitor['user_id'],
							$visitor['username'],
							'social_forum',
							$socialForum['social_forum_id'],
							'remove_moderator'
					);
				}
			}
			else
			{
				$this->_assertCanAddSocialForumModerator($socialForum);
				if (XenForo_Model_Alert::userReceivesAlert($socialForumMember, 'social_forum', 'moderator'))
				{
					$visitor = XenForo_Visitor::getInstance();
					XenForo_Model_Alert::alert(
							$userId,
							$visitor['user_id'],
							$visitor['username'],
							'social_forum',
							$socialForum['social_forum_id'],
							'moderator'
					);
				}
			}
			$writer->set('is_social_forum_moderator', !$socialForumMember['is_social_forum_moderator']);
			$writer->save();
		}
	
		return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('social-forums/members', $socialForum)
		);
	}	
	
	public function actionAcceptInvite()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		$member = $socialForum->getMember();
		
		if (isset($member['social_forum_member_id']) && $member['is_invited'])
		{
			$writer = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForumMember');
			$writer->setExistingData($member['social_forum_member_id']);
			$writer->set('is_invited', false);
			$writer->set('join_date', XenForo_Application::$time);
			$writer->save();
		}
		
		return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('social-forums', $socialForum),
				new XenForo_Phrase('waindigo_invite_accepted_social_groups')
		);
	}
	
	public function actionJoin()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		$member = $socialForum->getMember();
		
		$forumModel = $this->_getForumModel();
		
		$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);
			
		$ftpHelper = $this->getHelper('ForumThreadPost');
		$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
				$forumId,
				$this->_getForumFetchOptions()
		);
				
		$visitor = XenForo_Visitor::getInstance();
		$userId = $visitor['user_id'];

		if ($forumModel->canApproveSocialForumJoinRequest($socialForum) ||
				$forumModel->canRejectSocialForumJoinRequest($socialForum) ||
				$forumModel->canRevokeSocialForumMembership($socialForum))
		{
			if ($this->_input->filterSingle('user_id', XenForo_Input::UINT))
			{
				$userId = $this->_input->filterSingle('user_id', XenForo_Input::UINT);
			}
		}
		
		$user = $this->_getUserModel()->getUserById($userId);
		
		$writer = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForumMember');
		$socialForumMember = $this->_getSocialForumMemberModel()->getSocialForumMemberByUserId($socialForum['social_forum_id'], $userId);
		if ($socialForumMember)
		{
			$writer->setExistingData($socialForumMember['social_forum_member_id']);
			if ($this->_input->filterSingle('approved', XenForo_Input::UINT))
			{
				$this->_assertCanApproveSocialForumJoinRequest($socialForum);
				$writer->set('is_approved', true);
				$writer->save();
				if (XenForo_Model_Alert::userReceivesAlert($user, 'social_forum', 'approve'))
				{
					$visitor = XenForo_Visitor::getInstance();
					XenForo_Model_Alert::alert(
							$userId,
							$visitor['user_id'],
							$visitor['username'],
							'social_forum',
							$socialForum['social_forum_id'],
							'approve'
					);
				}
			}
			else
			{
				if (isset($member['user_id']) && $member['user_id'] == $userId)
				{
					$this->_assertCanLeaveSocialForum($socialForum);
				}
				elseif ($socialForumMember['is_approved'])
				{
					$this->_assertCanRevokeSocialForumMembership($socialForum);
					if (XenForo_Model_Alert::userReceivesAlert($user, 'social_forum', 'revoke'))
					{
						$visitor = XenForo_Visitor::getInstance();
						XenForo_Model_Alert::alert(
								$userId,
								$visitor['user_id'],
								$visitor['username'],
								'social_forum',
								$socialForum['social_forum_id'],
								'revoke'
						);
					}
				}
				else
				{
					if ($socialForumMember['is_social_forum_moderator'])
					{
						$this->_assertCanRemoveSocialForumModerator($socialForum);
						if (XenForo_Model_Alert::userReceivesAlert($user, 'social_forum', 'remove_moderator'))
						{
							$visitor = XenForo_Visitor::getInstance();
							XenForo_Model_Alert::alert(
									$userId,
									$visitor['user_id'],
									$visitor['username'],
									'social_forum',
									$socialForum['social_forum_id'],
									'remove_moderator'
							);
						}
					}
					$this->_assertCanRejectSocialForumJoinRequest($socialForum);
					if (XenForo_Model_Alert::userReceivesAlert($user, 'social_forum', 'reject'))
					{
						$visitor = XenForo_Visitor::getInstance();
						XenForo_Model_Alert::alert(
								$userId,
								$visitor['user_id'],
								$visitor['username'],
								'social_forum',
								$socialForum['social_forum_id'],
								'reject'
						);
					}
				}
				$writer->delete();
			}
			if (!isset($member['user_id']) || $member['user_id'] != $userId)
			{
				return $this->responseRedirect(
						XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
						XenForo_Link::buildPublicLink('social-forums/members', $socialForum)
				);
			}
		}
		else
		{
			$this->_assertCanJoinSocialForum($socialForum);
			
			$writer->bulkSet(array(
				'social_forum_id'	=> $socialForum['social_forum_id'],
				'user_id'			=> $userId,
			));
			if (!$socialForum['social_forum_moderated'])
			{
				$writer->set('is_approved', true);
			}
			else
			{
				if (XenForo_Model_Alert::userReceivesAlert($user, 'social_forum', 'request'))
				{
					$visitor = XenForo_Visitor::getInstance();
					XenForo_Model_Alert::alert(
							$socialForum['user_id'],
							$visitor['user_id'],
							$visitor['username'],
							'social_forum',
							$socialForum['social_forum_id'],
							'request'
					);
				}
			}
			$writer->save();
		}
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
			XenForo_Link::buildPublicLink('social-forums', $socialForum)
		);
	}

	public function actionCreateThread()
	{
		$response = parent::actionCreateThread();

		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
	
		if (get_class($response) == "XenForo_ControllerResponse_View")
		{
			$response->params['forum']['social_forum_id'] = $socialForum['social_forum_id'];
			$response->params['forum']['description'] = '';
		}
	
		return $response;
	}
	
	/**
	 * Member list
	 *
	 * @return XenForo_ControllerResponse_View
	 */
	public function actionMembers()
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		$viewParams['socialForum'] = $socialForum->toArray();
		if ($socialForum)
		{
			$socialForumMemberModel = $this->_getSocialForumMemberModel();

			$forumId = $this->_input->filterSingle('node_id', XenForo_Input::UINT);
			
			$ftpHelper = $this->getHelper('ForumThreadPost');
			$forum = $this->getHelper('ForumThreadPost')->assertForumValidAndViewable(
					$forumId,
					$this->_getForumFetchOptions()
			);

			$this->_assertCanViewSocialForumMembers($socialForum);
				
			$page = $this->_input->filterSingle('page', XenForo_Input::UINT);
			$usersPerPage = XenForo_Application::get('options')->membersPerPage;

			$member = $socialForum->getMember();
				
			$criteria = array(
				'social_forum_id' => $socialForum['social_forum_id'],
				'user_state' => 'valid',
				'is_banned' => 0
			);
			
			if (!$this->_getForumModel()->canApproveSocialForumJoinRequest($socialForum))
			{
				$criteria['is_approved'] = true;
			}
			
			// users for the member list
			$users = $socialForumMemberModel->getSocialForumUsers($criteria, array(
				'join' => XenForo_Model_User::FETCH_USER_FULL,
				'perPage' => $usersPerPage,
				'page' => $page
			));
			
			$viewParams = array(
				'users' => $users,
				'totalUsers' => $socialForumMemberModel->countSocialForumMembers($criteria),
				'page' => $page,
				'usersPerPage' => $usersPerPage,
				'socialForum' => $socialForum,
				'nodeBreadCrumbs' => $ftpHelper->getNodeBreadCrumbs($forum, true),
				'member' => $member,
					
				'canAddSocialForumModerator' => $this->_getForumModel()->canAddSocialForumModerator($socialForum),
				'canRemoveSocialForumModerator' => $this->_getForumModel()->canRemoveSocialForumModerator($socialForum),
				'canApproveSocialForumJoinRequest' => $this->_getForumModel()->canApproveSocialForumJoinRequest($socialForum),
				'canRejectSocialForumJoinRequest' => $this->_getForumModel()->canRejectSocialForumJoinRequest($socialForum),
				'canRevokeSocialForumMembership' => $this->_getForumModel()->canRevokeSocialForumMembership($socialForum),
			);
							
			return $this->responseView('Waindigo_SocialGroups_ViewPublic_SocialForumMember_List', 'waindigo_member_list_social_groups', $viewParams);
		}
	
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
			XenForo_Link::buildPublicLink('index')
		);
	}
	
	public function actionInvite()
	{
		$this->_assertPostOnly();
		
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		
		$username = $this->_input->filterSingle('username', XenForo_Input::STRING);
		$user = $this->_getUserModel()->getUserByName($username);
		
		$this->_getSocialForumMemberModel()->invite($user, $socialForum);

		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('social-forums/members', $socialForum),
			new XenForo_Phrase('waindigo_invite_sent_social_groups')
		);
	}
		
	public function canonicalizeRequestUrl($linkUrl)
	{
		$page = max(1, $this->_input->filterSingle('page', XenForo_Input::UINT));
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		return parent::canonicalizeRequestUrl(
			XenForo_Link::buildPublicLink('social-forums', $socialForum, array('page' => $page))
		);
	}
	
	public function getSocialForumId()
	{
		$socialForumId = $this->_input->filterSingle('social_forum_id', XenForo_Input::UINT);
		return ($socialForumId ? $socialForumId : false);
	}
	
	/**
	* Creates the specified helper class. If no underscore is present in the class
	* name, "XenForo_ControllerHelper_" is prefixed. Otherwise, a full class name
	* is assumed.
	*
	* @param string $class Full class name, or partial suffix (if no underscore)
	*
	* @return XenForo_ControllerHelper_Abstract
	*/
	public function getHelper($class)
	{
		if ($class == "ForumThreadPost")
		{
			$class = 'Waindigo_SocialGroups_ControllerHelper_SocialForumThreadPost';
		}
	
		return parent::getHelper($class);
	}
	
	/**
	 * Asserts that the currently browsing user can view members of the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanViewSocialForumMembers(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canViewSocialForumMembers($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}
	
	/**
	 * Asserts that the currently browsing user can join the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanJoinSocialForum(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canJoinSocialForum($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}	
	
	/**
	 * Asserts that the currently browsing user can leave the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanLeaveSocialForum(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canLeaveSocialForum($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}	
	
	/**
	 * Asserts that the currently browsing user can approve a join request in the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanApproveSocialForumJoinRequest(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canApproveSocialForumJoinRequest($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}	
	
	/**
	 * Asserts that the currently browsing user can reject a join request in the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanRejectSocialForumJoinRequest(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canRejectSocialForumJoinRequest($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}
	
	/**
	 * Asserts that the currently browsing user can revoke membership of the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanRevokeSocialForumMembership(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canRevokeSocialForumMembership($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}
	
	/**
	 * Asserts that the currently browsing user can edit the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanEditSocialForum(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canEditSocialForum($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}	

	/**
	 * Asserts that the currently browsing user can move the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanMoveSocialForum(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canMoveSocialForum($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}
	
	/**
	 * Asserts that the currently browsing user can delete the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanDeleteSocialForum(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canDeleteSocialForum($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}
	
	/**
	 * Asserts that the currently browsing user can add a moderator to the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanAddSocialForumModerator(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canAddSocialForumModerator($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}	

	/**
	 * Asserts that the currently browsing user can remove a moderator from the specified social forum.
	 *
	 * @param Waindigo_SocialGroups_SocialForum $socialForum
	 */
	protected function _assertCanRemoveSocialForumModerator(Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$errorPhraseKey = '';
		if (!$this->_getForumModel()->canRemoveSocialForumModerator($socialForum, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
	}
	
	/**
	* @return Waindigo_SocialGroups_Model_SocialForumMember
	*/
	protected function _getSocialForumMemberModel()
	{
		return $this->getModelFromCache('Waindigo_SocialGroups_Model_SocialForumMember');
	}
	
	/**
	 * @return XenForo_Model_User
	 */
	protected function _getUserModel()
	{
		return $this->getModelFromCache('XenForo_Model_User');
	}
	
	/**
	* @return Waindigo_SocialGroups_Model_SocialForum
	*/
	protected function _getForumModel()
	{
		return $this->getModelFromCache('Waindigo_SocialGroups_Model_SocialForum');
	}
}