<?php
/**
* Data writer for Social Forums.
*/
class Waindigo_SocialGroups_DataWriter_SocialForum extends XenForo_DataWriter
{
	protected function _getFields()
	{
		return array('xf_social_forum' => array(
			'social_forum_id'	 => array('type' => self::TYPE_UINT, 'autoIncrement' => true),
			'node_id' 			 => array('type' => self::TYPE_UINT, 'required' => true),
			'title' 			 => array('type' => self::TYPE_STRING, 'required' => true),
			'description' 		 => array('type' => self::TYPE_STRING),
			'user_id'			 => array('type' => self::TYPE_UINT, 'default' => 0),
			'created_date'		 => array('type' => self::TYPE_UINT, 'default' => XenForo_Application::$time),

			// denormalized counters
			'discussion_count'   => array('type' => self::TYPE_UINT_FORCED, 'default' => 0),
			'message_count'      => array('type' => self::TYPE_UINT_FORCED, 'default' => 0),

			// denormalized last post info
			'last_post_id'       => array('type' => self::TYPE_UINT,   'default' => 0),
			'last_post_date'     => array('type' => self::TYPE_UINT,   'default' => 0),
			'last_post_user_id'  => array('type' => self::TYPE_UINT,   'default' => 0),
			'last_post_username' => array('type' => self::TYPE_STRING, 'maxLength' => 50, 'default' => ''),
			'last_thread_title'  => array('type' => self::TYPE_STRING, 'maxLength' => 150, 'default' => ''),

			// options
			'social_forum_open'			=> array('type' => self::TYPE_BOOLEAN, 'default' => 1),
			'social_forum_moderated'	=> array('type' => self::TYPE_BOOLEAN, 'default' => 0),
			'moderate_messages'			=> array('type' => self::TYPE_BOOLEAN, 'default' => 0),
			'allow_posting'				=> array('type' => self::TYPE_BOOLEAN, 'default' => 1),
			'count_messages'			=> array('type' => self::TYPE_BOOLEAN, 'default' => 1),
			'find_new'					=> array('type' => self::TYPE_BOOLEAN, 'default' => 1),
				
			// avatar
			'logo_date'	=> array('type' => self::TYPE_UINT, 'default' => 0),
			'logo_width'	=> array('type' => self::TYPE_UINT, 'max' => 65535, 'default' => 0),
			'logo_height' => array('type' => self::TYPE_UINT, 'max' => 65535, 'default' => 0),
			'logo_crop_x' => array('type' => self::TYPE_UINT, 'default' => 0),
			'logo_crop_y' => array('type' => self::TYPE_UINT, 'default' => 0),
		));
	}

	/**
	* Gets the actual existing data out of data that was passed in. See parent for explanation.
	*
	* @param mixed
	*
	* @return array|false
	*/
	protected function _getExistingData($data)
	{
		if (!$socialForumId = $this->_getExistingPrimaryKey($data))
		{
			return false;
		}

		$socialForum = $this->getModelFromCache('Waindigo_SocialGroups_Model_SocialForum')->getSocialForumById($socialForumId);
		if (!$socialForum)
		{
			return false;
		}

		return $this->getTablesDataFromArray($socialForum);
	}
	
	protected function _postSave()
	{
		Waindigo_SocialGroups_SocialForum::setup($this->getMergedData());
		
		if ($this->isInsert())
		{
			$this->addCreatorAsMember();
		}
	}
	
	public function addCreatorAsMember()
	{
		$writer = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForumMember');
		$writer->bulkSet(array(
			'social_forum_id' => $this->get('social_forum_id'),
			'user_id' => $this->get('user_id'),
			'is_social_forum_moderator' => true,
			'is_social_forum_creator' => true,
			'is_approved' => true,
		));
		$writer->save();
	}
	
	/**
	* Gets SQL condition to update the existing record.
	*
	* @return string
	*/
	protected function _getUpdateCondition($tableName)
	{
		return 'social_forum_id = ' . $this->_db->quote($this->getExisting('social_forum_id'));
	}
	
	public function updateCountersAfterDiscussionSave(XenForo_DataWriter_Discussion $discussionDw, $forceInsert = false)
	{
		if ($discussionDw->get('discussion_type') == 'redirect')
		{
			// note: this assumes the discussion type will never change to/from this except at creation
			return;
		}
	
		if ($discussionDw->get('discussion_state') == 'visible'
		&& ($discussionDw->getExisting('discussion_state') != 'visible' || $forceInsert)
		)
		{
			$this->set('discussion_count', $this->get('discussion_count') + 1);
			$this->set('message_count', $this->get('message_count') + $discussionDw->get('reply_count') + 1);
		}
		else if ($discussionDw->getExisting('discussion_state') == 'visible' && $discussionDw->get('discussion_state') != 'visible')
		{
			$this->set('discussion_count', $this->get('discussion_count') - 1);
			$this->set('message_count', $this->get('message_count') - $discussionDw->get('reply_count') - 1);
	
			if ($discussionDw->get('last_post_id') == $this->get('last_post_id'))
			{
				$this->updateLastPost();
			}
		}
		else if ($discussionDw->get('discussion_state') == 'visible' && $discussionDw->getExisting('discussion_state') == 'visible')
		{
			// no state change, probably just a reply
			$messageChange = $discussionDw->get('reply_count') - $discussionDw->getExisting('reply_count');
			$this->set('message_count', $this->get('message_count') + $messageChange);
		}
	
		if ($discussionDw->get('discussion_state') == 'visible' && $discussionDw->get('last_post_date') >= $this->get('last_post_date'))
		{
			$this->set('last_post_date', $discussionDw->get('last_post_date'));
			$this->set('last_post_id', $discussionDw->get('last_post_id'));
			$this->set('last_post_user_id', $discussionDw->get('last_post_user_id'));
			$this->set('last_post_username', $discussionDw->get('last_post_username'));
			$this->set('last_thread_title', $discussionDw->get('title'));
		}
		else if ($discussionDw->get('discussion_state') == 'visible'
		&& $discussionDw->getExisting('discussion_state') == 'visible'
		&& $discussionDw->getExisting('last_post_id') == $this->get('last_post_id')
		&& ($discussionDw->isChanged('last_post_id') || $discussionDw->isChanged('title'))
		)
		{
			$this->updateLastPost();
		}
		
		$categoryDw = $this->_getSocialCategoryDataWriter($this->get('node_id'), $this->_errorHandler);
		$categoryDw->updateCountersAfterDiscussionSave($discussionDw);
		if ($categoryDw->hasChanges())
		{
			$categoryDw->save();
		}
	}
	
	/**
	 * Implemented for {@see XenForo_DataWriter_DiscussionContainerInterface}.
	 */
	public function updateCountersAfterDiscussionDelete(XenForo_DataWriter_Discussion $discussionDw)
	{
		if ($discussionDw->get('discussion_type') == 'redirect')
		{
			// note: this assumes the discussion type will never change to/from this except at creation
			return;
		}
	
		if ($discussionDw->get('discussion_state') == 'visible')
		{
			$this->set('discussion_count', $this->get('discussion_count') - 1);
			$this->set('message_count', $this->get('message_count') - $discussionDw->get('reply_count') - 1);
	
			if ($discussionDw->get('last_post_id') == $this->get('last_post_id'))
			{
				$this->updateLastPost();
			}
		}
		
		$categoryDw = $this->_getSocialCategoryDataWriter($this->get('node_id'), $this->_errorHandler);
		$categoryDw->updateCountersAfterDiscussionDelete($discussionDw);
		if ($categoryDw->hasChanges())
		{
			$categoryDw->save();
		}		
	}

	/**
	* Updates the last post information for this forum.
	*/
	public function updateLastPost()
	{
		$lastPost = $this->getModelFromCache('XenForo_Model_Thread')->getLastUpdatedThreadInSocialForum($this->get('social_forum_id'));
		if ($lastPost)
		{
			$this->set('last_post_id', $lastPost['last_post_id']);
			$this->set('last_post_date', $lastPost['last_post_date']);
			$this->set('last_post_user_id', $lastPost['last_post_user_id']);
			$this->set('last_post_username', $lastPost['last_post_username']);
			$this->set('last_thread_title', $lastPost['title']);
		}
		else
		{
			$this->set('last_post_id',0);
			$this->set('last_post_date', 0);
			$this->set('last_post_user_id', 0);
			$this->set('last_post_username', '');
			$this->set('last_thread_title', '');
		}
	}
	
	/**
	 * Rebuilds the counters for this forum.
	 */
	public function rebuildCounters()
	{
		$this->updateLastPost();
		$this->bulkSet($this->getModelFromCache('Waindigo_SocialGroups_Model_SocialForum')->getSocialForumCounters($this->get('social_forum_id')));
	}
	
	/**
	 * @return Waindigo_SocialGroups_DataWriter_SocialCategory
	 */
	protected function _getSocialCategoryDataWriter($socialCategoryId, $errorHandler)
	{
		$dw = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialCategory', $errorHandler);
		$dw->setExistingData($socialCategoryId);
		return $dw;
	}	
}