<?php

/**
 * Model for social_forum_members.
 */
class Waindigo_SocialGroups_Model_SocialForumMember extends XenForo_Model
{
	/**
	* Constants to allow joins to extra tables in certain queries
	*
	* @var integer Join user
	*/
	const FETCH_USER = 0x01;
	const FETCH_USER_PROFILE = 0x02;
	const FETCH_USER_OPTION = 0x04;
	const FETCH_USER_PRIVACY = 0x08;
	const FETCH_USER_ALL = 0x0F;
	
	/**
	* Gets social_forum_member that matches the given conditions.
	*
	* @param array $conditions Conditions to apply to the fetching
	* @param array $fetchOptions Collection of options that relate to fetching
	*
	* @return array|false
	*/
	public function getSocialForumMember(array $conditions = array(), array $fetchOptions = array())
	{
		$fetchOptions['limit'] = 1;
		
		$whereClause = $this->prepareSocialForumMemberConditions($conditions, $fetchOptions);
		$sqlClauses = $this->prepareSocialForumMemberFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		
		return $this->_getDb()->fetchRow($this->limitQueryResults('
				SELECT social_forum_member.*
					' . $sqlClauses['selectFields'] . '
				FROM xf_social_forum_member AS social_forum_member
				' . $sqlClauses['joinTables'] . '
				WHERE ' . $whereClause . '
			', $limitOptions['limit'], $limitOptions['offset'])
		);
	}

	/**
	* Gets the specified social_forum_member if it exists.
	*
	* @param string $social_forum_member_id
	*
	* @return array|false
	*/
	public function getSocialForumMemberById($id)
	{
		$fetchOptions = array();
		$conditions = array('social_forum_member_id' => $id);
		return $this->getSocialForumMember($conditions, $fetchOptions);
	}

	/**
	* Gets the specified social_forum_member if it exists.
	*
	* @param string $user_id
	*
	* @return array|false
	*/
	public function getSocialForumMemberByUserId($socialForumId, $id)
	{
		$fetchOptions = array();
		$conditions = array(
			'social_forum_id' => $socialForumId,
			'user_id' => $id,
		);
		return $this->getSocialForumMember($conditions, $fetchOptions);
	}
	
	/**
	* Gets social_forum_members that match the given conditions.
	*
	* @param array $conditions Conditions to apply to the fetching
	* @param array $fetchOptions Collection of options that relate to fetching
	*
	* @return array Format: [social_forum_member id] => info
	*/
	public function getSocialForumMembers(array $conditions = array(), array $fetchOptions = array())
	{
		$whereClause = $this->prepareSocialForumMemberConditions($conditions, $fetchOptions);
		$sqlClauses = $this->prepareSocialForumMemberFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
	
		return $this->fetchAllKeyed($this->limitQueryResults('
				SELECT social_forum_member.*
					' . $sqlClauses['selectFields'] . '
				FROM xf_social_forum_member AS social_forum_member
				' . $sqlClauses['joinTables'] . '
				WHERE ' . $whereClause . '
				' . $sqlClauses['orderClause'] . '
			', $limitOptions['limit'], $limitOptions['offset']
		), 'social_forum_member_id');
	}
	
	/**
	* Gets social forum users that match the given conditions.
	*
	* @param array $conditions Conditions to apply to the fetching
	* @param array $fetchOptions Collection of options that relate to fetching
	*
	* @return array Format: [user_id] => info
	*/
	public function getSocialForumUsers(array $conditions = array(), array $fetchOptions = array())
	{
		if (!isset($fetchOptions['join'])) $fetchOptions['join'] = 0;
		$fetchOptions['join'] |= self::FETCH_USER;
		
		$whereClause = $this->prepareSocialForumMemberConditions($conditions, $fetchOptions);
		$sqlClauses = $this->prepareSocialForumMemberFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
	
		return $this->fetchAllKeyed($this->limitQueryResults('
					SELECT social_forum_member.*
						' . $sqlClauses['selectFields'] . '
					FROM xf_social_forum_member AS social_forum_member
					' . $sqlClauses['joinTables'] . '
					WHERE ' . $whereClause . '
					' . $sqlClauses['orderClause'] . '
				', $limitOptions['limit'], $limitOptions['offset']
		), 'user_id');
	}
	
	/**
	 * Gets the count of social forum members with the specified criteria.
	 *
	 * @param array $conditions Conditions to apply to the fetching
	 *
	 * @return integer
	 */
	public function countSocialForumMembers(array $conditions)
	{
		$fetchOptions = array();
		$whereConditions = $this->prepareSocialForumMemberConditions($conditions, $fetchOptions);
	
		$sqlClauses = $this->prepareSocialForumMemberFetchOptions($fetchOptions);
			
		return $this->_getDb()->fetchOne('
				SELECT COUNT(*)
				FROM xf_social_forum_member AS social_forum_member
				' . $sqlClauses['joinTables'] . '
				WHERE ' . $whereConditions . '
			');
	}
	
	/**
	* Prepares join-related fetch options.
	*
	* @param array $fetchOptions
	*
	* @return array Containing 'selectFields' and 'joinTables' keys.
	*/	
	public function prepareSocialForumMemberFetchOptions(array $fetchOptions)
	{
		$selectFields = '';
		$joinTables = '';
		$orderClause = 'ORDER BY social_forum_member.is_social_forum_moderator DESC,
				social_forum_member.is_approved ASC ';
		
		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_USER)
			{
				$selectFields .= ',
								user.*';
				$joinTables .= '
					LEFT JOIN xf_user AS user ON
						(user.user_id = social_forum_member.user_id)';
				$orderClause .= ', user.username ASC';
			}

			if ($fetchOptions['join'] & self::FETCH_USER_PROFILE)
			{
				$selectFields .= ',
					user_profile.*';
				$joinTables .= '
					INNER JOIN xf_user_profile AS user_profile ON
						(user_profile.user_id = user.user_id)';
			}
			
			if ($fetchOptions['join'] & self::FETCH_USER_OPTION)
			{
				$selectFields .= ',
					user_option.*';
				$joinTables .= '
					INNER JOIN xf_user_option AS user_option ON
						(user_option.user_id = user.user_id)';
			}
			
			if ($fetchOptions['join'] & self::FETCH_USER_PRIVACY)
			{
				$selectFields .= ',
					user_privacy.*';
				$joinTables .= '
					INNER JOIN xf_user_privacy AS user_privacy ON
						(user_privacy.user_id = user.user_id)';
			}
				
		}
				
		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables,
			'orderClause'   => $orderClause,
		);
	}
	
	/**
	 * Prepares a set of conditions to select social_forum_members against.
	 *
	 * @param array $conditions List of conditions.
	 * @param array $fetchOptions The fetch options that have been provided. May be edited if criteria requires.
	 *
	 * @return string Criteria as SQL for where clause
	 */
	public function prepareSocialForumMemberConditions(array $conditions, array &$fetchOptions)
	{
		$db = $this->_getDb();
		$sqlConditions = array();
		
		if (isset($conditions['social_forum_member_id']))
		{
			$sqlConditions[] = 'social_forum_member.social_forum_member_id = ' . $db->quote($conditions['social_forum_member_id']);
		}		

		if (isset($conditions['user_id']))
		{
			$sqlConditions[] = 'social_forum_member.user_id = ' . $db->quote($conditions['user_id']);
		}

		if (isset($conditions['social_forum_id']))
		{
			$sqlConditions[] = 'social_forum_member.social_forum_id = ' . $db->quote($conditions['social_forum_id']);
		}

		if (isset($conditions['is_approved']))
		{
			$sqlConditions[] = 'social_forum_member.is_approved = ' . $db->quote($conditions['is_approved']);
		}

		if (isset($conditions['is_invited']))
		{
			$sqlConditions[] = 'social_forum_member.is_invited = ' . $db->quote($conditions['is_invited']);
		}
		
		return $this->getConditionsForClause($sqlConditions);
	}
	
	public function getMaximumMembershipForUserId($userId)
	{
		return $this->_getDb()->fetchRow('
				SELECT (max(social_forum_member.is_social_forum_moderator)*max(social_forum_member.is_approved) + max(social_forum_member.is_approved)) AS level
				FROM xf_social_forum_member AS social_forum_member
				WHERE user_id = ?
			', $userId
		);
	}
	
	public function approve(array $socialForumMember)
	{
		if (!$socialForumMember['is_approved'])
		{
			$writer = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForumMember');
			$writer->setExistingData($socialForumMember);
			$writer->set('is_approved', true);
			$writer->save();
			
			if (XenForo_Model_Alert::userReceivesAlert($socialForumMember, 'social_forum', 'approve'))
			{
				$visitor = XenForo_Visitor::getInstance();
				XenForo_Model_Alert::alert(
					$socialForumMember['user_id'],
					$visitor['user_id'],
					$visitor['username'],
					'social_forum',
					$socialForumMember['social_forum_id'],
					'approve'
				);
			}
			return true;
		}
		return false;
	}
	
	public function invite(array $user, Waindigo_SocialGroups_SocialForum $socialForum)
	{
		$member = $socialForum->getMember();
		
		$socialForumMember = $this->getSocialForumMemberByUserId($socialForum['social_forum_id'], $user['user_id']);
		if ($socialForumMember)
		{
			if ($this->_getSocialForumModel()->canApproveSocialForumJoinRequest($socialForum))
			{
				return $this->approve($socialForumMember);
			}
			return false;
		}

		$writer = XenForo_DataWriter::create('Waindigo_SocialGroups_DataWriter_SocialForumMember');
		$writer->set('user_id', $user['user_id']);
		$writer->set('social_forum_id', $socialForum['social_forum_id']);
		if (!$socialForum['social_forum_moderated'] || $this->_getSocialForumModel()->canApproveSocialForumJoinRequest($socialForum))
		{
			$writer->set('is_approved', true);
		}
		$writer->set('is_invited', true);
		$writer->save();
		
		if (XenForo_Model_Alert::userReceivesAlert($user, 'social_forum', 'invite'))
		{
			$visitor = XenForo_Visitor::getInstance();
			XenForo_Model_Alert::alert(
				$user['user_id'],
				$visitor['user_id'],
				$visitor['username'],
				'social_forum',
				$socialForum['social_forum_id'],
				'invite'
			);
		}
		
		return true;
	}
	
	/**
	 * @return Waindigo_SocialGroups_Model_SocialForum
	 */
	protected function _getSocialForumModel()
	{
		return $this->getModelFromCache('Waindigo_SocialGroups_Model_SocialForum');
	}
	
	/**
	 * @return XenForo_Model_Thread
	 */
	protected function _getThreadModel()
	{
		return $this->getModelFromCache('XenForo_Model_Thread');
	}
}