<?php

/**
 * Model for threads.
 */
class Waindigo_SocialGroups_Model_Thread extends XFCP_Waindigo_SocialGroups_Model_Thread
{
	const FETCH_SOCIAL_FORUM = 0x40;
	const FETCH_SOCIAL_FORUM_CREATOR = 0x80;

	public function prepareThreadFetchOptions(array $fetchOptions)
	{
		$selectFields = '';
		$joinTables = '';
		
		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_SOCIAL_FORUM)
			{
				$selectFields .= ',
								social_forum.title AS social_forum_title, social_forum.user_id AS social_forum_user_id';
				$joinTables .= '
								LEFT JOIN xf_social_forum AS social_forum ON
									(social_forum.social_forum_id = thread.social_forum_id)';
			}
			if ($fetchOptions['join'] & self::FETCH_SOCIAL_FORUM_CREATOR)
			{
				$selectFields .= ',
								social_forum_user.username AS social_forum_username';
				$joinTables .= '
								LEFT JOIN xf_user AS social_forum_user ON
									(social_forum.user_id = social_forum_user.user_id)';
			}
		}
		
		$threadFetchOptions = parent::prepareThreadFetchOptions($fetchOptions);
		
		return array(
			'selectFields' => $threadFetchOptions['selectFields'] . $selectFields,
			'joinTables'   => $joinTables . $threadFetchOptions['joinTables'],
			'orderClause'  => $threadFetchOptions['orderClause'],
		);
	}
	
	/**
	* Prepares a collection of thread fetching related conditions into an SQL clause
	*
	* @param array $conditions List of conditions
	* @param array $fetchOptions Modifiable set of fetch options (may have joins pushed on to it)
	*
	* @return string SQL clause (at least 1=1)
	*/
	public function prepareThreadConditions(array $conditions, array &$fetchOptions)
	{
		$sqlConditions = array();
		$db = $this->_getDb();
	
		$sqlConditions[] = parent::prepareThreadConditions($conditions, $fetchOptions);

		if (isset($conditions['social_forum_id']))
		{
			$sqlConditions[] = 'thread.social_forum_id = ' . ($conditions['social_forum_id']);
		}
	
		return $this->getConditionsForClause($sqlConditions);
	}
	
	/**
	* Gets the count of threads in the specified forum.
	*
	* @param integer $forumId
	* @param array $conditions Conditions to apply to the fetching
	*
	* @return integer
	*/
	public function countThreadsInForum($forumId, array $conditions = array())
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		if (!empty($socialForum))
		{
			if (isset($socialForum['social_forum_id']))
			{
				$conditions['social_forum_id'] = $socialForum['social_forum_id'];
			}
		}
		return parent::countThreadsInForum($forumId, $conditions);
	}
	
	/**
	 * Gets the thread with the most recent post in the specified social forum.
	 * Doesn't include redirects.
	 *
	 * @param integer $socialForumId
	 * @param array $fetchOptions Collection of options that relate to fetching
	 *
	 * @return array|false
	 */
	public function getLastUpdatedThreadInSocialForum($socialForumId, array $fetchOptions = array())
	{
		$db = $this->_getDb();
	
		$stateLimit = $this->prepareStateLimitFromConditions($fetchOptions, '', 'discussion_state');
	
		return $db->fetchRow($db->limit('
				SELECT *
				FROM xf_thread
				WHERE social_forum_id = ?
				AND discussion_type <> \'redirect\'
				AND (' . $stateLimit . ')
				ORDER BY last_post_date DESC
				', 1), $socialForumId);
	}
	
	/**
	* Gets threads that belong to the specified forum.
	*
	* @param integer $forumId
	* @param array $conditions Conditions to apply to the fetching
	* @param array $fetchOptions Collection of options that relate to fetching
	*
	* @return array Format: [thread id] => info
	*/
	public function getThreadsInForum($forumId, array $conditions = array(), array $fetchOptions = array())
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		if (!empty($socialForum))
		{
			if (isset($socialForum['social_forum_id']))
			{
				$conditions['social_forum_id'] = $socialForum['social_forum_id'];
			}
		}
		return parent::getThreadsInForum($forumId, $conditions, $fetchOptions);
	}
	
	/**
	 * Gets all sticky threads in a particular forum.
	 *
	 * @param integer $forumId
	 * @param array $fetchOptions Collection of options that relate to fetching
	 *
	 * @return array Format: [thread id] => info
	 */
	public function getStickyThreadsInForum($forumId, array $conditions = array(), array $fetchOptions = array())
	{
		$socialForum = Waindigo_SocialGroups_SocialForum::getInstance();
		if (!empty($socialForum))
		{
			if (isset($socialForum['social_forum_id']))
			{
				$conditions['social_forum_id'] = $socialForum['social_forum_id'];
			}
		}
		return parent::getStickyThreadsInForum($forumId, $conditions, $fetchOptions);
	}	
	
	/**
	* @return Waindigo_SocialGroups_Model_SocialForum
	*/
	protected function _getForumModel()
	{
		return $this->getModelFromCache('Waindigo_SocialGroups_Model_SocialForum');
	}	
}