<?php

class Waindigo_Thumbnails_Model_Thumbnails extends Xenforo_Model
{

    protected $_defaultThumbnail = array();

    protected static $_attachments = array();

    protected static $_posts = array();

    protected static $_articlePages = array();

    /**
     *
     * @param array $forum
     * @return array
     */
    public function getOptionValue(array &$forum)
    {
        if (!isset($forum['node_id'])) {
            return array();
        }
        if (!isset($forum['thumb_sources'])) {
            $dw = XenForo_DataWriter::create('XenForo_DataWriter_Forum');
            $dw->setExistingData($forum['node_id']);
            $forum['thumb_sources'] = $dw->getExisting('thumb_sources');
            $forum['thumb_width'] = $dw->getExisting('thumb_width');
            $forum['thumb_height'] = $dw->getExisting('thumb_height');
        }
        $options = array_filter(explode(",", $forum['thumb_sources']));
        if (empty($options)) {
            return array(
                "0" => "1"
            );
        } else {
            return $options;
        }
    } /* END getOptionValue */

    /**
     *
     * @param array $preparedOption
     * @return array
     */
    public function getPreparedOption(array $preparedOption)
    {
        $preparedOption['count'] = array(
            1,
            2,
            3,
            4
        );

        $preparedOption['sources'] = array(
            1 => new XenForo_Phrase('waindigo_user_entered_url_thumbnails'),
            2 => new XenForo_Phrase('waindigo_first_attachment_thumbnails'),
            3 => new XenForo_Phrase('waindigo_first_image_in_post_thumbnails'),
            4 => new XenForo_Phrase('waindigo_no_thumbnail_image_thumbnails'),
            5 => new XenForo_Phrase('waindigo_author_avatar_thumbnails')
        );

        $columns = array(
            1 => null,
            2 => null,
            3 => null,
            4 => null
        );
        foreach ($preparedOption['option_value'] as $row => $column) {
            $columns[$column] = $row;
        }
        $columnClasses = array(
            1 => "",
            2 => "",
            3 => "",
            4 => ""
        );
        foreach ($columns as $column => $row) {
            if (!isset($columnClasses[$column - 1])) {
                if (isset($row)) {
                    $columnClasses[$column] = "selected";
                }
            } else {
                if ($columnClasses[$column - 1] == "disabled") {
                    $columnClasses[$column] = "disabled";
                } elseif ($row) {
                    $columnClasses[$column] = "selected";
                } elseif ($columnClasses[$column - 1] == "") {
                    $columnClasses[$column] = "disabled";
                }
            }
        }
        $preparedOption['columnClasses'] = $columnClasses;

        return $preparedOption;
    } /* END getPreparedOption */

    /**
     *
     * @param array $nodesGrouped
     */
    public function addThumbsToNodesGrouped(array &$nodesGrouped)
    {
        $threads = array();
        $articles = array();

        foreach ($nodesGrouped as $parentNodeId => $parentNode) {
            foreach ($parentNode as $nodeId => $node) {
                if (isset($node['thread'])) {
                    $threadIds[$parentNodeId][$nodeId] = $node['thread']['thread_id'];
                    $threads[$node['thread']['thread_id']] = $node['thread'];
                    $forums[$nodeId] = $node;
                }
                if (isset($node['article'])) {
                    $articleIds[$parentNodeId][$nodeId] = $node['article']['article_id'];
                    $articles[$node['article']['article_id']] = $node['article'];
                    $libraries[$nodeId] = $node;
                }
            }
        }
        if ($threads) {
            $threads = $this->addThumbsToSearchThreads($threads, $forums);
            foreach ($threadIds as $parentNodeId => $nodeIds) {
                foreach ($nodeIds as $nodeId => $threadId) {
                    $nodesGrouped[$parentNodeId][$nodeId]['thread'] = $threads[$threadId];
                }
            }
        }
        if ($articles) {
            $articles = $this->addThumbsToSearchArticles($articles, $libraries);
            foreach ($articleIds as $parentNodeId => $nodeIds) {
                foreach ($nodeIds as $nodeId => $articleId) {
                    $nodesGrouped[$parentNodeId][$nodeId]['article'] = $articles[$articleId];
                }
            }
        }
    } /* END addThumbsToNodesGrouped */

    /**
     *
     * @param array $threads
     * @param array $nodes
     * @return array
     */
    public function addThumbsToSearchThreads(array $threads, array $nodes = array())
    {
        $threadIds = array_keys($threads);
        $postIds = $this->_getFirstPosts($threadIds, $threads);
        if (class_exists('Waindigo_LastPostAvatar_Listener_TemplateHook')) {
            if (XenForo_Application::get('options')->waindigo_lastPostAvatar_thumbnails) {
                $postIds = array_unique(array_merge($postIds, $this->_getLastPosts($threadIds, $threads)));
            }
        }
        $attachments = $this->_getAttachments('post', $postIds);
        $posts = $this->_getPosts($postIds);

        $forumThreads = $this->_arrangeThreadsByForum($threads);
        $oldThreads = $threads;
        $newThreads = array();
        foreach ($forumThreads as $forumId => $threads) {
            if (isset($nodes[$forumId])) {
                $node = $nodes[$forumId];
            } else {
                $node = array(
                    "node_id" => $forumId
                );
            }
            $newThreads[$forumId] = $this->addThumbsToThreads($threads, $node);
        }
        $threads = array();
        foreach ($oldThreads as $threadId => $thread) {
            $threads[$threadId] = $newThreads[$thread['node_id']][$threadId];
        }
        return $threads;
    } /* END addThumbsToSearchThreads */

    /**
     *
     * @param array $articles
     * @param array $nodes
     * @return array
     */
    public function addThumbsToSearchArticles(array $articles, array $nodes = array())
    {
        $articleIds = array_keys($articles);
        $articlePageIds = $this->_getFirstArticlePages($articleIds, $articles);
        $attachments = $this->_getAttachments('library-article', $articlePageIds);
        $articlePages = $this->_getArticlePages($articlePageIds);

        $libraryArticles = $this->_arrangeArticlesByLibrary($articles);
        $oldArticles = $articles;
        $newArticles = array();
        foreach ($libraryArticles as $libraryId => $articles) {
            if (isset($nodes[$libraryId])) {
                $node = $nodes[$libraryId];
            } else {
                $node = array(
                    "node_id" => $libraryId
                );
            }
            $newArticles[$libraryId] = $this->addThumbsToArticles($articles, $node);
        }
        $articles = array();
        foreach ($oldArticles as $articleId => $article) {
            $articles[$articleId] = $newArticles[$article['node_id']][$articleId];
        }
        return $articles;
    } /* END addThumbsToSearchArticles */

    /**
     *
     * @param array $threads
     * @return array
     */
    protected function _arrangeThreadsByForum(array $threads)
    {
        $forumThreads = array();
        foreach ($threads as $threadId => $thread) {
            $forumThreads[$thread['node_id']][$threadId] = $thread;
        }
        return $forumThreads;
    } /* END _arrangeThreadsByForum */

    /**
     *
     * @param array $articles
     * @return array
     */
    protected function _arrangeArticlesByLibrary(array $articles)
    {
        $libraryArticles = array();
        foreach ($articles as $articleId => $article) {
            $libraryArticles[$article['node_id']][$articleId] = $article;
        }
        return $libraryArticles;
    } /* END _arrangeArticlesByLibrary */

    /**
     *
     * @param string $original
     * @return string
     */
    protected function _urlMap($original)
    {
        $newUrl = $original;
        $requestPaths = XenForo_Application::get('requestPaths');
        $baseUrl = $requestPaths['fullBasePath'];

        if (strpos($newUrl, "http://") === false) {
            $newUrl = $baseUrl . $newUrl;
        }

        return $newUrl;
    } /* END _urlMap */

    /**
     *
     * @param array $thread
     * @param array $forum
     * @param array $options
     * @return array
     */
    public function addOpenGraphToThread(array $thread, array $forum, array $options = null)
    {
        $thread = $this->addThumbToThread($thread, $forum, $options);
        if (isset($thread['thumbnail']['thumbnail_url'])) {
            $thread['open_graph'] = $this->_urlMap($thread['thumbnail']['thumbnail_url']);
        }
        return $thread;
    } /* END addOpenGraphToThread */

    /**
     *
     * @param array $thread
     * @param array $forum
     * @param array $options
     * @return array
     */
    public function addThumbToThread(array $thread, array $forum, array $options = null)
    {
        $threads = $this->addThumbsToThreads(array(
            $thread['thread_id'] => $thread
        ), $forum, $options);
        return $threads[$thread['thread_id']];
    } /* END addThumbToThread */

    /**
     *
     * @param array $article
     * @param array $library
     * @param array $options
     * @return array
     */
    public function addThumbToArticle(array $article, array $library, array $options = null)
    {
        $articles = $this->addThumbsToArticles(array(
            $article['article_id'] => $article
        ), $library, $options);
        return $articles[$article['article_id']];
    } /* END addThumbToArticle */

    /**
     *
     * @param array $articles
     * @param array $library
     * @param array $options
     * @return array
     */
    public function addThumbsToArticles(array $articles, array $library, array $options = null)
    {
        if (!isset($options)) {
            $options = $this->getOptions($library);
        }

        $contents = array(
            'articles' => $articles
        );

        $contents = $this->addThumbsToContents($contents, $library, $options);

        return $contents['articles'];
    } /* END addThumbsToArticles */

    /**
     *
     * @param array $threads
     * @param array $forum
     * @param array $options
     * @return array
     */
    public function addThumbsToThreads(array $threads, array $forum, array $options = null)
    {
        if (!isset($options)) {
            $options = $this->getOptions($forum);
        }

        $contents = array(
            'threads' => $threads
        );

        if (class_exists('Waindigo_LastPostAvatar_Listener_TemplateHook')) {
            if (XenForo_Application::get('options')->waindigo_lastPostAvatar_thumbnails) {
                $contents['lastPostThreads'] = $this->_getLastPostThreads($threads);
            }
        }

        $contents = $this->addThumbsToContents($contents, $forum, $options);

        if (XenForo_Application::get('options')->waindigo_enableThumbnailCache_thumbnails) {
            $this->updateThreadThumbnailCache($contents['threads']);
        }

        if (isset($contents['lastPostThreads'])) {
            $contents['threads'] = $this->_combineLastPostThumbsIntoThreads($contents['lastPostThreads'],
                $contents['threads']);
        }

        return $contents['threads'];
    } /* END addThumbsToThreads */

    public function updateThreadThumbnailCache(array $threads)
    {
        foreach ($threads as $threadId => $thread) {
            if (empty($thread['thumbnail_cache_waindigo']) && !empty($thread['thumbnail'])) {
                $this->_getDb()->update('xf_thread', array(
                    'thumbnail_cache_waindigo' => serialize($thread['thumbnail'])),
                'thread_id = ' . $threadId);
            }
        }
    } /* END updateThreadThumbnailCache */

    /**
     *
     * @param array $contents
     * @param array $node
     * @param array $options
     * @return array
     */
    public function addThumbsToContents(array $contents, array $node, array $options = null)
    {
        $unmatched = array();
        foreach ($contents as $contentType => $content) {
            foreach ($content as $contentItemId => $contentItem) {
                if (XenForo_Application::get('options')->waindigo_enableThumbnailCache_thumbnails && !empty($contentItem['thumbnail_cache_waindigo'])) {
                    $contents[$contentType][$contentItemId]['thumbnail'] = unserialize($contentItem['thumbnail_cache_waindigo']);
                } else {
                    $unmatched[$contentType][] = $contentItemId;
                }
            }
        }
        foreach ($options as $option) {
            foreach ($unmatched as $unmatchedContentType => $unmatchedContent) {
                if (empty($unmatchedContent)) {
                    unset($unmatched[$unmatchedContentType]);
                }
            }
            if (empty($unmatched)) {
                break;
            }
            switch ($option) {
                case 1:
                    list($unmatched, $contents) = $this->_addUserEnteredURLs($unmatched, $contents, $node);
                    break;
                case 2:
                    list($unmatched, $contents) = $this->_addFirstAttachments($unmatched, $contents, $node);
                    break;
                case 3:
                    list($unmatched, $contents) = $this->_addFirstImages($unmatched, $contents, $node);
                    break;
                case 4:
                    $contents = $this->_addNoThumbnailImages($unmatched, $contents, $node);
                case 5:
                    $unmatched = array();
            }
        }
        return $contents;
    } /* END addThumbsToContents */

    /**
     *
     * @param array $threads
     * @return array
     */
    protected function _getLastPostThreads(array $threads)
    {
        foreach ($threads as $threadId => $thread) {
            $threads[$threadId]['first_post_id'] = $thread['last_post_id'];
        }
        return $threads;
    } /* END _getLastPostThreads */

    /**
     *
     * @param array $lastPostThreads
     * @param array $threads
     * @return array
     */
    protected function _combineLastPostThumbsIntoThreads(array $lastPostThreads, array $threads)
    {
        foreach ($lastPostThreads as $threadId => $thread) {
            if (isset($thread['thumbnail'])) {
                $threads[$threadId]['lastPostInfo']['thumbnail'] = $thread['thumbnail'];
            }
        }
        return $threads;
    } /* END _combineLastPostThumbsIntoThreads */

    /**
     *
     * @param array $forum
     * @return array
     */
    public function getOptions(array $forum)
    {
        $forumOptions = $this->_sortOptions($this->getOptionValue($forum));
        $defaultOptions = $this->_sortOptions(XenForo_Application::get('options')->waindigo_thumbSources_thumbnails);
        $options = array();
        foreach ($forumOptions as $option) {
            if ($option == 0) {
                $options = array_merge_recursive($options, $defaultOptions);
            } else {
                $options[] = $option;
            }
        }
        return array_unique($options);
    } /* END getOptions */

    /**
     *
     * @param array $unmatched
     * @param array $contents
     * @param array $node
     * @return array
     */
    protected function _addUserEnteredURLs(array $unmatched, array $contents, array $node = array())
    {
        if (isset($contents['threads'])) {
            list($unmatched['threads'], $contents['threads']) = $this->_addUserEnteredURLsToThreads(
                $unmatched['threads'], $contents['threads'], $node);
        }
        if (isset($contents['lastPostThreads'])) {
            list($unmatched['lastPostThreads'], $contents['lastPostThreads']) = $this->_addUserEnteredURLsToThreads(
                $unmatched['lastPostThreads'], $contents['lastPostThreads'], $node);
        }
        if (isset($contents['articles'])) {
            list($unmatched['articles'], $contents['articles']) = $this->_addUserEnteredURLsToArticles(
                $unmatched['articles'], $contents['articles'], $node);
        }

        return array(
            $unmatched,
            $contents
        );
    } /* END _addUserEnteredURLs */

    /**
     *
     * @param array $unmatched
     * @param array $threads
     * @param array $forum
     * @return array
     */
    protected function _addUserEnteredURLsToThreads(array $unmatched, array $threads, array $forum = array())
    {
        return $this->_addUserEnteredURLsToContents($unmatched, $threads, $forum);
    } /* END _addUserEnteredURLsToThreads */

    /**
     *
     * @param array $unmatched
     * @param array $articles
     * @param array $library
     * @return array
     */
    protected function _addUserEnteredURLsToArticles(array $unmatched, array $articles, array $library = array())
    {
        return $this->_addUserEnteredURLsToContents($unmatched, $articles, $library);
    } /* END _addUserEnteredURLsToArticles */

    /**
     *
     * @param array $unmatched
     * @param array $contents
     * @param array $node
     * @return array
     */
    protected function _addUserEnteredURLsToContents(array $unmatched, array $contents, array $node = array())
    {
        $newUnmatched = array();

        foreach ($unmatched as $contentId) {
            if (isset($contents[$contentId]['thumbnail_url']) and $contents[$contentId]['thumbnail_url']) {
                $newThumbnail = array(
                    'thumbnail_url' => $contents[$contentId]['thumbnail_url']
                );
                if (XenForo_Application::$versionId >= 1030033) {
                    $newThumbnail['thumbnailUrl'] = $this->_handleThumbnailProxyOption($newThumbnail['thumbnail_url']);
                }

                $contents[$contentId]['thumbnail'] = array_merge($newThumbnail,
                    $this->_calculateDimensions(array(), $node));
            } else {
                $newUnmatched[] = $contentId;
            }
        }

        return array(
            $newUnmatched,
            $contents
        );
    } /* END _addUserEnteredURLsToContents */

    /**
     *
     * @param array $unmatched
     * @param array $contents
     * @param array $node
     * @return array
     */
    protected function _addFirstAttachments(array $unmatched, array $contents, array $node)
    {
        if (isset($unmatched['threads']) || isset($unmatched['lastPostThreads'])) {
            $firstPostIds = array();
            if (isset($unmatched['threads'])) {
                $firstPostIds = $this->_getFirstPosts($unmatched['threads'], $contents['threads']);
            }
            if (isset($unmatched['lastPostThreads'])) {
                $firstPostIds = array_merge($firstPostIds,
                    $this->_getFirstPosts($unmatched['lastPostThreads'], $contents['lastPostThreads']));
            }
            if ($firstPostIds) {
                $attachments = $this->_getAttachments('post', $firstPostIds);
                if (isset($unmatched['threads'])) {
                    list($unmatched['threads'], $contents['threads']) = $this->_addFirstAttachmentsToThreads(
                        $unmatched['threads'], $contents['threads'], $attachments, $node);
                }
                if (isset($unmatched['lastPostThreads'])) {
                    list($unmatched['lastPostThreads'], $contents['lastPostThreads']) = $this->_addFirstAttachmentsToThreads(
                        $unmatched['lastPostThreads'], $contents['lastPostThreads'], $attachments, $node, true);
                }
            }
        }

        if (!empty($unmatched['articles'])) {
            $attachments = $this->_getAttachments('library_article', $unmatched['articles']);
            list($unmatched['articles'], $contents['articles']) = $this->_addFirstAttachmentsToArticles(
                $unmatched['articles'], $contents['articles'], $attachments, $node);
        }

        return array(
            $unmatched,
            $contents
        );
    } /* END _addFirstAttachments */

    /**
     *
     * @param array $unmatched
     * @param array $threads
     * @param array $attachments
     * @param array $forum
     * @param boolean $lastPost
     * @return array
     */
    protected function _addFirstAttachmentsToThreads(array $unmatched, array $threads, array $attachments,
        array $forum = array(), $lastPost = false)
    {
        $attachmentModel = $this->_getAttachmentModel();
        $newUnmatched = array();

        foreach ($unmatched as $threadId) {
            $thread = $threads[$threadId];
            if ($lastPost) {
                $postId = $thread['last_post_id'];
            } else {
                $postId = $thread['first_post_id'];
            }

            if (isset($attachments[$postId])) {
                $threads[$threadId]['thumbnail'] = array_merge(
                    array(
                        'thumbnail_url' => $attachmentModel->getAttachmentThumbnailUrl($attachments[$postId])
                    ), $this->_calculateDimensions($attachments[$postId], $forum));
            } else {
                $newUnmatched[] = $threadId;
            }
        }

        return array(
            $newUnmatched,
            $threads
        );
    } /* END _addFirstAttachmentsToThreads */

    /**
     *
     * @param array $unmatched
     * @param array $articles
     * @param array $attachments
     * @param array $library
     * @return array
     */
    protected function _addFirstAttachmentsToArticles(array $unmatched, array $articles, array $attachments,
        array $library = array())
    {
        $attachmentModel = $this->_getAttachmentModel();
        $newUnmatched = array();

        foreach ($unmatched as $articleId) {
            $article = $articles[$articleId];
            if (isset($attachments[$articleId])) {
                $articles[$articleId]['thumbnail'] = array_merge(
                    array(
                        'thumbnail_url' => $attachmentModel->getAttachmentThumbnailUrl($attachments[$articleId])
                    ), $this->_calculateDimensions($attachments[$article['article_id']], $library));
            } else {
                $newUnmatched[] = $articleId;
            }
        }

        return array(
            $newUnmatched,
            $articles
        );
    } /* END _addFirstAttachmentsToArticles */

    /**
     *
     * @param string $message
     * @param array $forum
     * @return array
     */
    public function getImagesFromMessage($message, array $forum = array())
    {
        preg_match_all('/\[img\](.*?)\[\/img\]/i', $message, $matches);
        $thumbnails = array();
        foreach ($matches[1] as $thumbnailUrl) {
            $newThumbnail = array(
                'thumbnail_url' => $thumbnailUrl
            );
            if (XenForo_Application::$versionId >= 1030033) {
                $newThumbnail['thumbnailUrl'] = $this->_handleThumbnailProxyOption($thumbnailUrl);
            }
            $thumbnails[] = array_merge($newThumbnail, $this->_calculateDimensions(array(), $forum));
        }
        return $thumbnails;
    } /* END getImagesFromMessage */

    /**
     *
     * @param string $message
     * @return void unknown
     */
    public function getImageFromMessage($message)
    {
        preg_match('/\[img\](.*?)\[\/img\]/i', $message, $matches);
        if (isset($matches[1])) {
            return $matches[1];
        } else {
            return;
        }
    } /* END getImageFromMessage */

    /**
     *
     * @param array $unmatched
     * @param array $contents
     * @param array $node
     * @return array
     */
    protected function _addFirstImages(array $unmatched, array $contents, array $node)
    {
        if (isset($unmatched['threads']) || isset($unmatched['lastPostThreads'])) {
            $firstPostIds = array();
            if (isset($unmatched['threads'])) {
                $firstPostIds = $this->_getFirstPosts($unmatched['threads'], $contents['threads']);
            }
            if (isset($unmatched['lastPostThreads'])) {
                $firstPostIds = array_merge($firstPostIds,
                    $this->_getFirstPosts($unmatched['lastPostThreads'], $contents['lastPostThreads']));
            }
            $firstPosts = $this->_getPosts($firstPostIds);

            if (isset($unmatched['threads'])) {
                list($unmatched['threads'], $contents['threads']) = $this->_addFirstImageInPostToThreads(
                    $unmatched['threads'], $contents['threads'], $firstPosts, $node);
            }
            if (isset($unmatched['lastPostThreads'])) {
                list($unmatched['lastPostThreads'], $contents['lastPostThreads']) = $this->_addFirstImageInPostToThreads(
                    $unmatched['lastPostThreads'], $contents['lastPostThreads'], $firstPosts, $node);
            }
        }
        if (isset($unmatched['articles'])) {
            $articlePageModel = $this->_getArticlePageModel();
            $firstArticlePageIds = $this->_getFirstArticlePages($unmatched['articles'], $contents['articles']);
            $firstArticlePages = $articlePageModel->getArticlePagesByIds($firstArticlePageIds);
            list($unmatched['articles'], $contents['articles']) = $this->_addFirstImageInPostToArticles(
                $unmatched['articles'], $contents['articles'], $firstArticlePages, $node);
        }
        return array(
            $unmatched,
            $contents
        );
    } /* END _addFirstImages */

    /**
     *
     * @param array $unmatched
     * @param array $threads
     * @param array $firstPosts
     * @param array $forum
     * @return array
     */
    protected function _addFirstImageInPostToThreads(array $unmatched, array $threads, array $firstPosts,
        array $forum = array())
    {
        foreach ($firstPosts as $postId => $post) {
            if (isset($post['message']) and $thumbnailUrl = $this->getImageFromMessage($post['message'])) {
                $newThumbnail = array(
                    'thumbnail_url' => $thumbnailUrl
                );
                if (XenForo_Application::$versionId >= 1030033) {
                    $newThumbnail['thumbnailUrl'] = $this->_handleThumbnailProxyOption($newThumbnail['thumbnail_url']);
                }

                $threads[$post['thread_id']]['thumbnail'] = array_merge($newThumbnail,
                    $this->_calculateDimensions(array(), $forum));
                unset($unmatched[array_search($post['thread_id'], $unmatched)]);
            }
        }

        return array(
            $unmatched,
            $threads
        );
    } /* END _addFirstImageInPostToThreads */

    /**
     *
     * @param array $unmatched
     * @param array $articles
     * @param array $firstArticlePages
     * @param array $library
     * @return array
     */
    protected function _addFirstImageInPostToArticles(array $unmatched, array $articles, array $firstArticlePages,
        array $library = array())
    {
        foreach ($firstArticlePages as $articlePageId => $articlePage) {
            if (isset($articlePage['message']) and $thumbnailUrl = $this->getImageFromMessage($articlePage['message'])) {
                $newThumbnail = array(
                    'thumbnail_url' => $thumbnailUrl
                );
                if (XenForo_Application::$versionId >= 1030033) {
                    $newThumbnail['thumbnailUrl'] = $this->_handleThumbnailProxyOption($newThumbnail['thumbnail_url']);
                }

                $articles[$articlePage['article_id']]['thumbnail'] = array_merge($newThumbnail,
                    $this->_calculateDimensions(array(), $library));
                unset($unmatched[array_search($articlePage['article_id'], $unmatched)]);
            }
        }

        return array(
            $unmatched,
            $articles
        );
    } /* END _addFirstImageInPostToArticles */

    /**
     *
     * @param array $unmatched
     * @param array $contents
     * @param array $node
     * @return array
     */
    protected function _addNoThumbnailImages(array $unmatched, array $contents, array $node)
    {
        foreach ($unmatched as $unmatchedTypeId => $unmatchedItem) {
            foreach ($unmatchedItem as $contentId) {
                $contents[$unmatchedTypeId][$contentId]['thumbnail'] = $this->_getDefaultThumbnail($node);
            }
        }

        return $contents;
    } /* END _addNoThumbnailImages */

    /**
     *
     * @param array $node
     * @return array
     */
    protected function _getDefaultThumbnail(array $node = array())
    {
        if (!isset($this->_defaultThumbnail[$node['node_id']])) {
            list($width, $height) = $this->_getDefaultDimensions($node);
            $maxDimension = max($width, $height);
            $attachment = array();
            $noThumbImage = XenForo_Application::get('options')->waindigo_noThumbImage_thumbnails;
            if ($noThumbImage) {
                $attachment['thumbnail_width'] = $width;
                $attachment['thumbnail_height'] = $height;
                $newThumbnail = array(
                    'thumbnail_url' => $noThumbImage
                );
                if (XenForo_Application::$versionId >= 1030033) {
                    $newThumbnail['thumbnailUrl'] = $this->_handleThumbnailProxyOption($newThumbnail['thumbnail_url']);
                }
                $this->_defaultThumbnail[$node['node_id']] = array_merge($newThumbnail,
                    $this->_calculateDimensions($attachment, $node));
            } elseif ($maxDimension <= 48) {
                $attachment['thumbnail_width'] = 48;
                $attachment['thumbnail_height'] = 48;
                $this->_defaultThumbnail[$node['node_id']] = array_merge(
                    array(
                        'thumbnail_url' => "styles/default/xenforo/avatars/avatar_s.png"
                    ), $this->_calculateDimensions($attachment, $node));
            } elseif ($maxDimension <= 96) {
                $attachment['thumbnail_width'] = 96;
                $attachment['thumbnail_height'] = 96;
                $this->_defaultThumbnail[$node['node_id']] = array_merge(
                    array(
                        'thumbnail_url' => "styles/default/xenforo/avatars/avatar_m.png"
                    ), $this->_calculateDimensions($attachment, $node));
            } else {
                $attachment['thumbnail_width'] = 192;
                $attachment['thumbnail_height'] = 192;
                $this->_defaultThumbnail[$node['node_id']] = array_merge(
                    array(
                        'thumbnail_url' => "styles/default/xenforo/avatars/avatar_l.png"
                    ), $this->_calculateDimensions($attachment, $node));
            }
        }

        return $this->_defaultThumbnail[$node['node_id']];
    } /* END _getDefaultThumbnail */

    /**
     *
     * @param array $forum
     * @return array
     */
    protected function _getDefaultDimensions(array $forum = array())
    {
        $width = XenForo_Application::get('options')->waindigo_thumbWidth_thumbnails;
        $height = XenForo_Application::get('options')->waindigo_thumbHeight_thumbnails;
        if (isset($forum['thumb_width']) and $forum['thumb_width']) {
            $width = $forum['thumb_width'];
        }
        if (isset($forum['thumb_height']) and $forum['thumb_height']) {
            $height = $forum['thumb_height'];
        }
        return array(
            $width,
            $height
        );
    } /* END _getDefaultDimensions */

    /**
     *
     * @param array $attachment
     * @param array $forum
     * @return array
     */
    protected function _calculateDimensions(array $attachment = array(), array $forum = array())
    {
        list($width, $height) = $this->_getDefaultDimensions($forum);
        $newWidth = $width;
        $newHeight = $height;
        $newRatio = $newWidth / $newHeight;

        $oldWidth = $newWidth;
        $oldHeight = $newHeight;

        if (isset($attachment['thumbnail_width'])) {
            $oldWidth = $attachment['thumbnail_width'];
        }

        if (isset($attachment['thumbnail_height'])) {
            $oldHeight = $attachment['thumbnail_height'];
        }

        if ($oldWidth != 0 and $oldHeight != 0) {
            $oldRatio = $oldWidth / $oldHeight;
        } else {
            $oldRatio = 1;
        }

        $verticalPosition = 0;
        $horizontalPosition = 0;

        if ($newRatio > $oldRatio) {
            $newHeight = round($newWidth / $oldRatio);
            $verticalPosition = round(($height - $newHeight) / 2);
        } elseif ($newRatio < $oldRatio) {
            $newWidth = round($newHeight * $oldRatio);
            $horizontalPosition = round(($width - $newWidth) / 2);
        }

        return array(
            'width' => $newWidth,
            'height' => $newHeight,
            'max-width' => $width,
            'max-height' => $height,
            'vertical-position' => $verticalPosition,
            'horizontal-position' => $horizontalPosition
        );
    } /* END _calculateDimensions */

    /**
     *
     * @param array $unmatched
     * @param array $threads
     * @return array
     */
    protected function _getFirstPosts(array $unmatched, array $threads)
    {
        $firstPosts = array();
        foreach ($unmatched as $threadId) {
            $firstPosts[$threadId] = $threads[$threadId]['first_post_id'];
        }
        return $firstPosts;
    } /* END _getFirstPosts */

    /**
     *
     * @param array $unmatched
     * @param array $threads
     * @return array
     */
    protected function _getLastPosts(array $unmatched, array $threads)
    {
        $lastPosts = array();
        foreach ($unmatched as $threadId) {
            $lastPosts[$threadId] = $threads[$threadId]['last_post_id'];
        }
        return $lastPosts;
    } /* END _getLastPosts */

    /**
     *
     * @param array $postIds
     * @return array
     */
    protected function _getPosts(array $postIds)
    {
        $postModel = $this->_getPostModel();

        $posts = array();
        $checkPostIds = $postIds;
        foreach ($checkPostIds as $key => $postId) {
            if (isset(self::$_posts[$postId])) {
                $posts[$postId] = self::$_posts[$postId];
                unset($postIds[$key]);
            }
        }

        if (!empty($postIds)) {
            $posts = $posts + $postModel->getPostsByIds($postIds);
        }

        self::$_posts = self::$_posts + $posts;

        return $posts;
    } /* END _getPosts */

    /**
     *
     * @param array $articlePageIds
     * @return array
     */
    protected function _getArticlePages(array $articlePageIds)
    {
        $articlePageModel = $this->_getArticlePageModel();

        $articlePages = array();
        $checkArticlePageIds = $articlePageIds;
        foreach ($checkArticlePageIds as $key => $articlePageId) {
            if (isset(self::$_articlePages[$articlePageId])) {
                $articlePages[$articlePageId] = self::$_articlePages[$articlePageId];
                unset($articlePageIds[$key]);
            }
        }

        if (!empty($articlePageIds)) {
            $articlePages = $articlePages + $articlePageModel->getArticlePagesByIds($articlePageIds);
        }

        self::$_articlePages = self::$_articlePages + $articlePages;

        return $articlePages;
    } /* END _getArticlePages */

    /**
     *
     * @param array $unmatched
     * @param array $articles
     * @return array
     */
    protected function _getFirstArticlePages(array $unmatched, array $articles)
    {
        $firstArticlePages = array();
        foreach ($unmatched as $articleId) {
            if ($articles[$articleId]['discussion_type'] != 'redirect') {
                $firstArticlePages[$articleId] = $articles[$articleId]['first_article_page_id'];
            }
        }
        return $firstArticlePages;
    } /* END _getFirstArticlePages */

    /**
     *
     * @param string $contentType
     * @param array $contentIds
     * @return array
     */
    protected function _getAttachments($contentType, array $contentIds)
    {
        $attachmentModel = $this->_getAttachmentModel();

        $attachments = array();
        $checkContentIds = $contentIds;
        foreach ($checkContentIds as $key => $contentId) {
            if (isset(self::$_attachments[$contentType][$contentId])) {
                if (!empty(self::$_attachments[$contentType][$contentId])) {
                    $attachments[$contentId] = self::$_attachments[$contentType][$contentId];
                }
                unset($contentIds[$key]);
            } else {
                self::$_attachments[$contentType][$contentId] = array();
            }
        }

        if (!empty($contentIds)) {
            $attachments = $attachments + $attachmentModel->getFirstAttachmentsByContentIds($contentType, $contentIds);
        }

        if (!empty(self::$_attachments[$contentType])) {
            self::$_attachments[$contentType] = array_merge($attachments, self::$_attachments[$contentType]);
        }

        return $attachments;
    } /* END _getAttachments */

    /**
     *
     * @param array $thumbSources
     * @return array
     */
    protected function _sortOptions(array $thumbSources)
    {
        $thumbSources = array_flip($thumbSources);
        ksort($thumbSources);
        for ($i = 1; $i <= 4; $i++) {
            if (!isset($thumbSources[$i])) {
                $thumbSources = array_slice($thumbSources, 0, 4);
            }
        }
        return $thumbSources;
    } /* END _sortOptions */

    /**
     *
     * @param int $sourceId
     * @param array $forum
     * @return boolean
     */
    public function isInThumbSources($sourceId, array $forum)
    {
        if (!$this->canShowThumbs($forum)) {
            return false;
        }
        $thumbSources = $this->getOptions($forum);
        return in_array($sourceId, $thumbSources);
    } /* END isInThumbSources */

    /**
     * Determines if the specified forum shows thumbnails with the given
     * permissions.
     *
     * @param array $node Info about the node
     * @param string $errorPhraseKey Returned phrase key for a specific error
     * @param array|null $nodePermissions List of permissions for this page; if
     * not provided, use visitor's permissions
     * @param array|null $viewingUser
     *
     * @return boolean
     */
    public function canShowThumbs(array $node, &$errorPhraseKey = '', array $nodePermissions = null, array $viewingUser = null)
    {
        if (isset($node['node_type_id'])) {
            if ($node['node_type_id'] == 'Forum' || $node['node_type_id'] == 'SocialCategory') {
                $showInForums = XenForo_Application::get('options')->waindigo_showInForums_thumbnails;
                if (($showInForums['_type'] == "_some" and !isset($showInForums[$node['node_id']]))) {
                    return false;
                } elseif ($showInForums['_type'] == "_permissions") {
                    $this->standardizeViewingUserReferenceForNode($node['node_id'], $viewingUser, $nodePermissions);

                    return XenForo_Permission::hasContentPermission($nodePermissions, 'thumbnails');
                }
                return true;
            } elseif ($node['node_type_id'] == 'Library') {
                $showInLibraries = XenForo_Application::get('options')->waindigo_showInLibraries_thumbnails;
                if (($showInLibraries['_type'] == "_some" and !isset($showInLibraries[$node['node_id']]))) {
                    return false;
                } elseif ($showInLibraries['_type'] == "_permissions") {
                    $this->standardizeViewingUserReferenceForNode($node['node_id'], $viewingUser, $nodePermissions);

                    return XenForo_Permission::hasContentPermission($nodePermissions, 'articleThumbs');
                }
                return true;
            }
        }
        return false;
    } /* END canShowThumbs */

    protected function _handleThumbnailProxyOption($url)
    {
        list($class, $target, $type, $schemeMatch) = XenForo_Helper_String::getLinkClassTarget($url);

        if ($type == 'external' && $schemeMatch) {
            $options = XenForo_Application::getOptions();
            if (!empty($options->imageLinkProxy['thumbnails'])) {
                $url = $this->_generateProxyLink('thumbnail', $url);
            } elseif (!empty($options->imageLinkProxy['images'])) {
                $url = $this->_generateProxyLink('image', $url);
            }
        }

        return $url;
    } /* END _handleThumbnailProxyOption */

    protected function _generateProxyLink($proxyType, $url)
    {
        $hash = hash_hmac('md5', $url,
            XenForo_Application::getConfig()->globalSalt . XenForo_Application::getOptions()->imageLinkProxyKey);
        return 'proxy.php?' . $proxyType . '=' . urlencode($url) . '&hash=' . $hash;
    } /* END _generateProxyLink */

    /**
     *
     * @return Waindigo_Library_Model_ArticlePage
     */
    protected function _getArticlePageModel()
    {
        return $this->getModelFromCache('Waindigo_Library_Model_ArticlePage');
    } /* END _getArticlePageModel */

    /**
     *
     * @return XenForo_Model_Post
     */
    protected function _getPostModel()
    {
        return $this->getModelFromCache('XenForo_Model_Post');
    } /* END _getPostModel */

    /**
     *
     * @return XenForo_Model_Attachment
     */
    protected function _getAttachmentModel()
    {
        return $this->getModelFromCache('XenForo_Model_Attachment');
    } /* END _getAttachmentModel */
}