<?php

/**
 * Controller for managing upgrade coupons.
 *
 * @package Waindigo_UpgradeCoupons
 */
class Waindigo_UpgradeCoupons_ControllerAdmin_UpgradeCoupons extends XenForo_ControllerAdmin_Abstract
{

    /**
     * Displays a list of user coupons, and the related option configuration.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionIndex()
    {
        $couponModel = $this->_getCouponModel();
        
        $viewParams = array(
            'coupons' => $couponModel->getAllCoupons()
        );
        
        return $this->responseView('XenForo_ViewAdmin_UpgradeCoupons_List', 
            'waindigo_upgrade_coupons_list_upgradecoupons', $viewParams);
    } /* END actionIndex */

    /**
     * Gets the coupon add/edit form response.
     *
     * @param array $coupon
     *
     * @return XenForo_ControllerResponse_View
     */
    protected function _getCouponAddEditResponse(array $coupon)
    {
        if (!empty($coupon['user_ids'])) {
            $userIds = explode(',', $coupon['user_ids']);
            $userNames = $this->_getUserModel()->getNamesByIds($userIds);
            if ($userNames) {
                $coupon['usernames'] = implode(', ', $userNames);
            }
        }
        $viewParams = array(
            'coupon' => $coupon,
            'userUpgradesActive' => $this->_getUpgradeModel()->userUpgradesCheck(),
            'upgradeOptions' => $this->_getUpgradeModel()->getUserUpgradeOptions($coupon['upgrade_ids'])
        );
        
        return $this->responseView('XenForo_ViewAdmin_UpgradeCoupons_Edit', 
            'waindigo_upgrade_coupon_edit_upgradecoupon', $viewParams);
    } /* END _getCouponAddEditResponse */

    /**
     * Displays a form to add a coupon.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionAdd()
    {
        return $this->_getCouponAddEditResponse(
            array(
                'coupon_id' => null,
                'title' => '',
                'description' => '',
                'amount' => 5,
                'coupon_type' => 'flat_rate',
                'coupon_key' => '',
                'user_ids' => '',
                'upgrade_ids' => '',
                'multiple_use' => 0,
                're_entry_required' => 1,
                'single_use' => 0,
                'initial_payment_only' => 0,
                'is_gift' => 0,
                'is_active' => 1
            ));
    } /* END actionAdd */

    /**
     * Displays a form to edit a coupon.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionEdit()
    {
        $couponId = $this->_input->filterSingle('coupon_id', XenForo_Input::UINT);
        $coupon = $this->_getCouponOrError($couponId);
        
        return $this->_getCouponAddEditResponse($coupon);
    } /* END actionEdit */

    /**
     * Inserts a new coupon or updates an existing one.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionSave()
    {
        $this->_assertPostOnly();
        
        $couponId = $this->_input->filterSingle('coupon_id', XenForo_Input::UINT);
        $input = $this->_input->filter(
            array(
                'title' => XenForo_Input::STRING,
                'amount' => XenForo_Input::UNUM,
                'coupon_type' => XenForo_Input::STRING,
                'coupon_key' => XenForo_Input::STRING,
                'upgrade_ids' => array(
                    XenForo_Input::UINT,
                    'array' => true
                ),
                'multiple_use' => XenForo_Input::UINT,
                're_entry_required' => XenForo_Input::UINT,
                'single_use' => XenForo_Input::UINT,
                'initial_payment_only' => XenForo_Input::UINT,
                'is_active' => XenForo_Input::UINT
            ));
        
        $dw = XenForo_DataWriter::create('Waindigo_UpgradeCoupons_DataWriter_UpgradeCoupons');
        if ($couponId) {
            $dw->setExistingData($couponId);
        }
        $dw->bulkSet($input);
        $dw->save();
        
        return $this->responseRedirect(XenForo_ControllerResponse_Redirect::SUCCESS, 
            XenForo_Link::buildAdminLink('upgrade-coupons') . $this->getLastHash($dw->get('coupon_id')));
    } /* END actionSave */

    /**
     * Deletes a coupon.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionDelete()
    {
        if ($this->isConfirmedPost()) {
            return $this->_deleteData('Waindigo_UpgradeCoupons_DataWriter_UpgradeCoupons', 'coupon_id', 
                XenForo_Link::buildAdminLink('upgrade-coupons'));
        } else {
            $couponId = $this->_input->filterSingle('coupon_id', XenForo_Input::UINT);
            $coupon = $this->_getCouponOrError($couponId);
            
            $dw = XenForo_DataWriter::create('Waindigo_UpgradeCoupons_DataWriter_UpgradeCoupons', 
                XenForo_DataWriter::ERROR_EXCEPTION);
            $dw->setExistingData($coupon, true);
            $dw->preDelete();
            
            $viewParams = array(
                'coupon' => $coupon
            );
            
            return $this->responseView('XenForo_ViewAdmin_UpgradeCoupons_Delete', 
                'waindigo_coupon_delete_upgradecoupons', $viewParams);
        }
    } /* END actionDelete */

    /**
     * Displays a list of active coupons, either across all coupons or a
     * specific one.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionActive()
    {
        $couponId = $this->_input->filterSingle('coupon_id', XenForo_Input::UINT);
        $couponModel = $this->_getCouponModel();
        
        $page = $this->_input->filterSingle('page', XenForo_Input::UINT);
        $perPage = 20;
        
        $fetchOptions = array(
            'page' => $page,
            'perPage' => $perPage
        );
        
        if ($couponId) {
            $coupon = $this->_getCouponOrError($couponId);
            
            $conditions = array(
                'coupon_id' => $coupon['coupon_id']
            );
            
            $viewParams = array(
                'coupon' => $coupon,
                'couponRecords' => $couponModel->getGiftCouponRecords($conditions, $fetchOptions),
                
                'totalRecords' => $couponModel->countGiftCouponRecords($conditions),
                'perPage' => $perPage,
                'page' => $page
            );
            
            return $this->responseView('Waindigo_UpgradeCoupons_ViewAdmin_UpgradeCoupons_Active', 
                'waindigo_gift_coupon_active_single_upgradecoupons', $viewParams);
        } else {
            $conditions = array();
            
            $couponRecords = $couponModel->getGiftCouponRecords($conditions, $fetchOptions);
            $couponModel->prepareUsernamesForRecords($couponRecords);
            
            $viewParams = array(
                'couponRecords' => $couponRecords,
                
                'totalRecords' => $couponModel->countGiftCouponRecords($conditions),
                'perPage' => $perPage,
                'page' => $page
            );
            
            return $this->responseView('Waindigo_UpgradeCoupons_ViewAdmin_UpgradeCoupons_Active', 
                'waindigo_gift_coupons_active_upgradecoupons', $viewParams);
        }
    } /* END actionActive */

    /**
     * Selectively enables or disables specified coupons
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionToggle()
    {
        return $this->_getToggleResponse($this->_getCouponModel()
            ->getAllCoupons(), 'Waindigo_UpgradeCoupons_DataWriter_UpgradeCoupons', 'coupons', 'is_active');
    } /* END actionToggle */

        /**
     * Gets the specified coupon or throws an exception.
     *
     * @param string $id
     *
     * @return array
     */
    protected function _getCouponOrError($id)
    {
        $couponModel = $this->_getCouponModel();
        $coupon = $couponModel->getCouponById($id);
        if (!$coupon) {
            throw $this->responseException(
                $this->responseError(new XenForo_Phrase('waindigo_cannot_find_requested_coupon_upgradecoupons')));
        }
        return $coupon;
    } /* END _getCouponOrError */

    /**
     *
     * @return XenForo_Model_User
     */
    protected function _getUserModel()
    {
        return $this->getModelFromCache('XenForo_Model_User');
    } /* END _getUserModel */

    /**
     *
     * @return Waindigo_Model_UpgradeCoupons
     */
    protected function _getCouponModel()
    {
        return $this->getModelFromCache('Waindigo_UpgradeCoupons_Model_UpgradeCoupons');
    } /* END _getCouponModel */

    /**
     *
     * @return XenForo_Model_UserUpgrade
     */
    protected function _getUpgradeModel()
    {
        return $this->getModelFromCache('XenForo_Model_UserUpgrade');
    } /* END _getUpgradeModel */
}