<?php

/**
 * Model for user coupons.
 *
 * @package XenForo_Coupon
 */
class Waindigo_UpgradeCoupons_Model_UpgradeCoupons extends XenForo_Model
{

    /**
     * Joins the coupon details to a user-specific coupon record.
     *
     * @var int
     */
    const JOIN_UPGRADE = 1;

    /**
     * Gets the specified coupon.
     *
     * @param integer $id
     *
     * @return array false
     */
    public function getCouponById($id)
    {
        return $this->_getDb()->fetchRow(
            '
			SELECT *
			FROM xf_upgrade_coupon_waindigo
			WHERE coupon_id = ?
		', $id);
    } /* END getCouponById */

    /**
     * Gets a users coupon if they have one applied to their account.
     *
     * @return array false
     */
    public function getCouponFromUserId($userId)
    {
        return $this->_getDb()->fetchRow(
            '
			SELECT *
			FROM xf_upgrade_coupon_waindigo AS coupon
            INNER JOIN xf_user_profile AS user_profile ON (coupon.coupon_id=user_profile.coupon_id)
			WHERE user_id = ?
		', $userId);
    } /* END getCouponFromUserId */

    /**
     * Returns coupon from database based on a given coupon key
     */
    public function getCouponFromInput($couponKey)
    {
        return $this->_getDb()->fetchRow(
            '
            SELECT *
            FROM xf_upgrade_coupon_waindigo
            WHERE coupon_key LIKE ?', $couponKey);
    } /* END getCouponFromInput */

    /**
     * Gets all coupons in display order.
     *
     * @return array [coupon id] => info
     */
    public function getAllCoupons()
    {
        return $this->fetchAllKeyed('
			SELECT *
			FROM xf_upgrade_coupon_waindigo
            WHERE is_gift != 1
		', 'coupon_id');
    } /* END getAllCoupons */
    
    /**
     * Gets all gift coupons in display order.
     *
     * @return array [coupon id] => info
     */
    public function getAllGiftCoupons()
    {
        return $this->fetchAllKeyed('
			SELECT *
			FROM xf_upgrade_coupon_waindigo
            WHERE is_gift = 1
		', 'coupon_id');
    } /* END getAllGiftCoupons */

    public function getGiftCoupon($upgradeId)
    {
        return $this->_getDb()->fetchRow(
            '
			SELECT *
			FROM xf_upgrade_coupon_waindigo AS coupon
            WHERE is_gift = 1
                AND upgrade_ids = ?
                AND gift_status = \'created\'
		', $upgradeId
            );
    } /* END getGiftCoupon */
    
    /**
     * Gets the specified upgrade coupon records. Queries active and expired records.
     *
     * @param array $conditions
     * @param array $fetchOptions
     *
     * @return array [coupon id]
     */
    public function getGiftCouponRecords(array $conditions = array(), array $fetchOptions = array())
    {
        $whereClause = $this->prepareGiftCouponRecordConditions($conditions, $fetchOptions);
        $limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
    
        return $this->fetchAllKeyed($this->limitQueryResults(
            '
				SELECT coupon_waindigo.*
				FROM xf_upgrade_coupon_waindigo AS coupon_waindigo
		        WHERE is_gift = 1
				AND ' . $whereClause . '
			', $limitOptions['limit'], $limitOptions['offset']
        ), 'coupon_id');
    } /* END getGiftCouponRecords */
    
    /**
     * Counts the number of upgrade coupon records matching the conditions.
     *
     * @param array $conditions
     *
     * @return integer
     */
    public function countGiftCouponRecords(array $conditions = array())
    {
        $fetchOptions = array();
        $whereClause = $this->prepareGiftCouponRecordConditions($conditions, $fetchOptions);
    
        return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM xf_upgrade_coupon_waindigo AS coupon_waindigo
            WHERE is_gift = 1
			AND ' . $whereClause
        );
    } /* END countGiftCouponRecords */
    
    /**
     * Adds username information to the gift coupon records. 
     * 
     * @param array $couponRecords
     */
    public function prepareUsernamesForRecords(&$couponRecords = array())
    {
        $couponUserData = array();
        foreach ($couponRecords as $key => $record) {
            $extraData = unserialize($record['extra_data']);
            if (isset($extraData['redeemer_user_id'])) {
                if (!in_array($extraData['redeemer_user_id'], $couponUserData)) {
                    $couponUserData[] = $extraData['redeemer_user_id'];
                }
            }
            if (!in_array($extraData['purchase_user_id'], $couponUserData)) {
                $couponUserData[] = $extraData['purchase_user_id'];
            }
        }
        $couponUserNames = $this->_getUserModel()->getNamesByIds($couponUserData);
        foreach ($couponRecords as $key => &$record) {
            $extraData = unserialize($record['extra_data']);
            if (isset($extraData['redeemer_user_id'])) {
                $record['redeemer_username'] = $couponUserNames[$extraData['redeemer_user_id']];
                $record['purchase_username'] = $couponUserNames[$extraData['purchase_user_id']];
            } else {
                $record['purchase_username'] = $couponUserNames[$extraData['purchase_user_id']];
            }
        }
    } /* END prepareUsernamesForRecords */ /* END _prepareRecords */
    
    
    /**
     * Prepares a list of upgrade coupon gift record conditions.
     *
     * @param array $conditions
     * @param array $fetchOptions
     *
     * @return string
     */
    public function prepareGiftCouponRecordConditions(array $conditions, array &$fetchOptions)
    {
        $db = $this->_getDb();
        $sqlConditions = array();
    
        if (!empty($conditions['coupon_id']))
        {
            if (is_array($conditions['coupon_id']))
            {
                $sqlConditions[] = 'coupon_waindigo.coupon_id IN (' . $db->quote($conditions['coupon_id']) . ')';
            }
            else
            {
                $sqlConditions[] = 'coupon_waindigo.coupon_id = ' . $db->quote($conditions['coupon_id']);
            }
        }
    
        return $this->getConditionsForClause($sqlConditions);
    } /* END prepareGiftCouponRecordConditions */

    public function updateGiftCoupon(array $upgrade, $userId, $giftStatus)
    {
        $coupon = $this->getGiftCoupon($upgrade['user_upgrade_id']);
        if ($coupon['coupon_id']) {
            $dw = XenForo_DataWriter::create('Waindigo_UpgradeCoupons_DataWriter_UpgradeCoupons');
            $dw->setExistingData($coupon['coupon_id']);
            $dw->set('gift_status', $giftStatus);
            $dw->save();
        }
        if ($giftStatus == 'activated') {
            $this->sendUserCoupon($coupon['coupon_id'], $userId);
        }
    } /* END updateGiftCoupon */

    public function createGiftCoupon(array $upgrade)
    {
        $coupon = array(
            'title' => new XenForo_Phrase('waindigo_gift_coupon_upgradecoupons') . $upgrade['title'],
            'amount' => $upgrade['cost_amount'],
            'coupon_type' => 'flat_rate',
            'coupon_key' => XenForo_Application::generateRandomString(10),
            'upgrade_ids' => $upgrade['user_upgrade_id'],
            'multiple_use' => 0,
            're_entry_required' => 1,
            'single_use' => 1,
            'is_gift' => 1,
            'gift_status' => 'created',
            'is_active' => 1,
            'extra_data' => array('upgrade_id' => $upgrade['user_upgrade_id'],
                'purchase_user_id' => XenForo_Visitor::getUserId(),
                'cost_amount' => $upgrade['cost_amount'])
        );
        $dw = XenForo_DataWriter::create('Waindigo_UpgradeCoupons_DataWriter_UpgradeCoupons');
        $dw->bulkSet($coupon);
        $dw->save();
    } /* END createGiftCoupon */

    public function sendUserCoupon($couponId, $userId)
    {
        $coupon = $this->getCouponById($couponId);
        $user = $this->_getUserModel()->getUserById($userId);
        
        if (!$coupon || !$user) {
            return false;
        }
        
        //TODO - add more ways to inform members of coupon details. 
        if (true) {
            $this->_emailUser($user, $coupon);
        }
    } /* END sendUserCoupon */

    /**
     * Send an email to notify the user the coupon details
     *
     * @param array $user
     * @param array $coupon
     */
    protected function _emailUser(array $user, array $coupon)
    {
        $mail = XenForo_Mail::create('waindigo_email_coupon_upgradecoupons', 
            array(
                'user' => $user,
                'coupon' => $coupon
            ), $user['language_id']);
        
        $mail->send($user['email'], $user['username']);
        return true;
    } /* END _emailUser */

    /**
     * Upgrades the user with the specified coupon.
     *
     * @return integer false coupon record ID
     */
    public function updateUsersCouponRecord(array $user, array $coupon)
    {
        if ($user['coupon_id'] == $coupon['coupon_id']) {
            return true;
        }
        
        if (!$coupon['is_active']) {
            return false;
        }
        
        $db = $this->_getDb();
        XenForo_Db::beginTransaction($db);
        
        $db->update('xf_user_profile', array(
            'coupon_id' => $coupon['coupon_id']
        ), 'user_id = ' . $this->_getDb()
            ->quote($user['user_id']));
        
        XenForo_Db::commit($db);
        
        return true;
    } /* END updateUsersCouponRecord */

    /**
     * Performs the necessary checks to make sure the upgrade coupon can be used
     * with the given user upgrade.
     *
     * @param array $coupon
     * @param string $userId
     * @return boolean
     */
    public function checkCouponWithUser(array $coupon, $userId = null)
    {
        if (!$userId) {
            $userId = XenForo_Visitor::getUserId();
        }
        if (!$coupon['is_active']) {
            return false;
        }
        if (!empty($coupon['user_ids'])) {
            $couponUserIds = explode(',', $coupon['user_ids']);
            if (in_array($userId, $couponUserIds) && !$coupon['multiple_use']) {
                return false;
            }
        }
        return true;
    } /* END checkCouponWithUser */

    /**
     * Performs the necessary checks to make sure the upgrade coupon can be used
     * with the given user upgrade.
     *
     * @param array $coupon
     * @param array $upgrade
     * @param string $userId
     * @return boolean
     */
    public function checkCouponWithUpgrade(array $coupon, array $upgrade, $userId = null)
    {
        if (!$userId) {
            $userId = XenForo_Visitor::getUserId();
        }
        if (!$coupon['is_active'] || !$upgrade['allow_coupons']) {
            return false;
        }
        
        $userUpgradesCheck = $this->_getUpgradeModel()->userUpgradesCheck();
        //removes recurring upgrades if required
        if ($upgrade['recurring']) {
            if (!XenForo_Application::get('options')->waindigo_upgradeCoupons_allowRecurring) {
                return false;
            }
            if ($userUpgradesCheck) {
                if (!$coupon['initial_payment_only'] && $upgrade['length_amount_post_trial']) {
                    return false;
                }
            }
        }
        
        if ($userUpgradesCheck) {
            if ($upgrade['tiered_upgrade']) {
                return false;
            }
        }
        
        if (!empty($coupon['upgrade_ids'])) {
            $upgradeIds = explode(',', $coupon['upgrade_ids']);
            if (!in_array($upgrade['user_upgrade_id'], $upgradeIds)) {                
                return false;
            }
        }
        if (!empty($coupon['user_ids'])) {
            $couponUserIds = explode(',', $coupon['user_ids']);
            if (in_array($userId, $couponUserIds) && !$coupon['multiple_use']) {
                return false;
            }
        }
        return true;
    } /* END checkCouponWithUpgrade */

    /**
     * Applies the necessary discount to the given upgrade.
     *
     * @param array $upgrade
     * @param array $coupon
     * @return string
     */
    public function applyCouponToUpgrade(array $upgrade, array $coupon)
    {
        if (isset($upgrade['coupon_applied'])) {
            return $upgrade;
        }
        if ($upgrade['recurring'] && $coupon['initial_payment_only'] && XenForo_Application::get('options')->waindigo_upgradeCoupons_allowRecurring && $this->_getUpgradeModel()->userUpgradesCheck()) {
            if ($upgrade['length_amount_post_trial'] && $upgrade['length_unit_post_trial']) {
                if ($upgrade['cost_amount_trial'] > 0) {
                    $upgrade['cost_amount_trial'] = $upgrade['cost_amount_trial'] - $coupon['amount'];
                    $upgrade['cost_amount'] = number_format($upgrade['cost_amount'] - $coupon['amount'],2);
                    $upgrade['coupon_applied'] = true;
                    return $upgrade;
                }
                $upgrade['cost_amount'] = number_format($upgrade['cost_amount'] - $coupon['amount'],2);
                $upgrade['cost_amount_trial'] = $upgrade['cost_amount_trial'] - $coupon['amount'];
                $upgrade['coupon_applied'] = true;
                return $upgrade;
            }
            if (!$upgrade['length_amount_post_trial'] && !$upgrade['length_unit_post_trial']) {
                $upgrade['length_amount_post_trial'] = $upgrade['length_amount'];
                $upgrade['length_unit_post_trial'] = $upgrade['length_unit'];
                $costAmount = $upgrade['cost_amount'];
                $upgrade['cost_amount_trial'] = -$coupon['amount'];
                $upgrade['cost_amount'] = number_format($costAmount - $coupon['amount'], 2);
                $upgrade['coupon_applied'] = true;
            }
            return $upgrade;
        }
        if ($coupon['coupon_type'] == 'flat_rate') {
            $upgrade['cost_amount'] = number_format($upgrade['cost_amount'] - $coupon['amount'], 2);
        } else {
            $upgrade['cost_amount'] = number_format($upgrade['cost_amount'] * ((100 - $coupon['amount']) / 100), 2);
        }
        $upgrade['coupon_applied'] = true;
        return $upgrade;
    } /* END applyCouponToUpgrade */

    /**
     * Updates the count of purchasable coupons.
     *
     * @return integer
     */
    public function updateCouponCount()
    {
        $coupons = $this->getAllCoupons();
        foreach ($coupons as $couponId => $coupon) {
            if (!$coupon['is_active']) {
                unset($coupons[$couponId]);
            }
        }
        $couponCount = count($coupons);
        
        XenForo_Application::setSimpleCacheData('couponCount', $couponCount);
        
        return $couponCount;
    } /* END updateCouponCount */

    /**
     * Gets a list of user upgrade coupons as options.
     *
     * @param string|array $selectedOptions List of selected options
     * @param integer|false $skip ID to skip
     *
     * @return array
     */
    public function getCouponOptions($selectedOptions, $skip = false)
    {
        if (!is_array($selectedOptions)) {
            $selectedOptions = ($selectedOptions ? explode(',', $selectedOptions) : array());
        }
        
        $options = array();
        foreach ($this->getAllCoupons() as $coupon) {
            if ($coupon['coupon_id'] == $skip) {
                continue;
            }
            
            $options[] = array(
                'label' => $coupon['title'],
                'value' => $coupon['coupon_id'],
                'selected' => in_array($coupon['coupon_id'], $selectedOptions)
            );
        }
        
        return $options;
    } /* END getCouponOptions */

    public function getCouponsForAdminQuickSearch($searchText)
    {
        $quotedText = XenForo_Db::quoteLike($searchText, 'lr', $this->_getDb());
        
        return $this->fetchAllKeyed(
            '
			SELECT *
			FROM xf_upgrade_coupon_waindigo
			WHERE title LIKE ' . $quotedText . '
				OR description LIKE ' . $quotedText, 'coupon_id');
    } /* END getCouponsForAdminQuickSearch */

    protected function _verifyCouponOrError(array $coupon, array $upgrade, array $user)
    {
        return $coupon;
    } /* END _verifyCouponOrError */
    
    /**
     *
     * @return XenForo_Model_User
     */
    protected function _getUserModel()
    {
        return $this->getModelFromCache('XenForo_Model_User');
    } /* END _getUserModel */
    
    /**
     *
     * @return XenForo_Model_UserUpgrade
     */
    protected function _getUpgradeModel()
    {
        return $this->getModelFromCache('XenForo_Model_UserUpgrade');
    } /* END _getUpgradeModel */
}