<?php

/**
 * Admin controller for handling actions on upgrade categories.
 */
class Waindigo_UserUpgrades_ControllerAdmin_UpgradeCategory extends XenForo_ControllerAdmin_Abstract
{

    protected function _preDispatch($action)
    {
        $this->assertAdminPermission('upgrade');
    } /* END _preDispatch */

    /**
     * Shows a list of upgrade categories.
     *
     * @return XenForo_ControllerResponse_View
     */
    public function actionIndex()
    {
        $upgradeCategoryModel = $this->_getUpgradeCategoryModel();

        $upgradeCategories = $upgradeCategoryModel->getUpgradeCategories(array(),
            array(
                'order' => 'default',
                'orderDirection' => 'asc'
            ));

        $upgradeCategories = $upgradeCategoryModel->groupUpgradeCategoriesByParent($upgradeCategories);

        $viewParams = array(
            'upgradeCategories' => $upgradeCategories
        );

        return $this->responseView('Waindigo_UserUpgrades_ViewAdmin_UpgradeCategory_List',
            'waindigo_upgrade_category_list_userupgrades', $viewParams);
    } /* END actionIndex */

    /**
     * Helper to get the upgrade category add/edit form controller response.
     *
     * @param array $upgradeCategory
     *
     * @return XenForo_ControllerResponse_View
     */
    protected function _getUpgradeCategoryAddEditResponse(array $upgradeCategory)
    {
        $upgradeCategories = $this->_getUpgradeCategoryModel()->getUpgradeCategories(
            array(
                'parent_category_id' => 0
            ));
        
        $upgradeCategoryTitles = array();
        foreach ($upgradeCategories as $upgradeCategoryId => $category) {
            $upgradeCategoryTitles[$upgradeCategoryId] = $category['title'];
        }

        $viewParams = array(
            'upgradeCategory' => $upgradeCategory,
            'upgradeCategories' => $upgradeCategoryTitles
        );

        return $this->responseView('Waindigo_UserUpgrades_ViewAdmin_UpgradeCategory_Edit',
            'waindigo_upgrade_category_edit_userupgrades', $viewParams);
    } /* END _getUpgradeCategoryAddEditResponse */

    /**
     * Displays a form to add a new upgrade category.
     *
     * @return XenForo_ControllerResponse_View
     */
    public function actionAdd()
    {
        $upgradeCategory = $this->_getUpgradeCategoryModel()->getDefaultUpgradeCategory();

        return $this->_getUpgradeCategoryAddEditResponse($upgradeCategory);
    } /* END actionAdd */

    /**
     * Displays a form to edit an existing upgrade category.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionEdit()
    {
        $upgradeCategoryId = $this->_input->filterSingle('upgrade_category_id', XenForo_Input::STRING);

        if (!$upgradeCategoryId) {
            return $this->responseReroute('Waindigo_UserUpgrades_ControllerAdmin_UpgradeCategory', 'add');
        }

        $upgradeCategory = $this->_getUpgradeCategoryOrError($upgradeCategoryId);

        return $this->_getUpgradeCategoryAddEditResponse($upgradeCategory);
    } /* END actionEdit */

    /**
     * Inserts a new upgrade category or updates an existing one.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionSave()
    {
        $this->_assertPostOnly();

        $upgradeCategoryId = $this->_input->filterSingle('upgrade_category_id', XenForo_Input::STRING);

        $input = $this->_input->filter(
            array(
                'title' => XenForo_Input::STRING,
                'parent_category_id' => XenForo_Input::UINT,
                'display_order' => XenForo_Input::UINT,
            ));

        $visitor = XenForo_Visitor::getInstance();
        $writer = XenForo_DataWriter::create('Waindigo_UserUpgrades_DataWriter_UpgradeCategory');
        if ($upgradeCategoryId) {
            $writer->setExistingData($upgradeCategoryId);
        }
        $writer->bulkSet($input);
        $writer->save();

        if ($this->_input->filterSingle('reload', XenForo_Input::STRING)) {
            return $this->responseRedirect(XenForo_ControllerResponse_Redirect::RESOURCE_UPDATED,
                XenForo_Link::buildAdminLink('upgrade-categories/edit', $writer->getMergedData()));
        } else {
            return $this->responseRedirect(XenForo_ControllerResponse_Redirect::SUCCESS,
                XenForo_Link::buildAdminLink('upgrade-categories') .
                     $this->getLastHash($writer->get('upgrade_category_id')));
        }
    } /* END actionSave */

    /**
     * Deletes a upgrade category.
     *
     * @return XenForo_ControllerResponse_Abstract
     */
    public function actionDelete()
    {
        $upgradeCategoryId = $this->_input->filterSingle('upgrade_category_id', XenForo_Input::STRING);
        $upgradeCategory = $this->_getUpgradeCategoryOrError($upgradeCategoryId);

        $writer = XenForo_DataWriter::create('Waindigo_UserUpgrades_DataWriter_UpgradeCategory');
        $writer->setExistingData($upgradeCategory);

        if ($this->isConfirmedPost()) { // delete upgrade category
            $writer->delete();

            return $this->responseRedirect(XenForo_ControllerResponse_Redirect::SUCCESS,
                XenForo_Link::buildAdminLink('upgrade-categories'));
        } else { // show delete confirmation prompt
            $writer->preDelete();
            $errors = $writer->getErrors();
            if ($errors) {
                return $this->responseError($errors);
            }

            $viewParams = array(
                'upgradeCategory' => $upgradeCategory
            );

            return $this->responseView('Waindigo_UserUpgrades_ViewAdmin_UpgradeCategory_Delete',
                'waindigo_upgrade_category_delete_userupgrades', $viewParams);
        }
    } /* END actionDelete */

    /**
     * Gets a valid upgrade category or throws an exception.
     *
     * @param string $upgradeCategoryId
     *
     * @return array
     */
    protected function _getUpgradeCategoryOrError($upgradeCategoryId)
    {
        $upgradeCategory = $this->_getUpgradeCategoryModel()->getUpgradeCategoryById($upgradeCategoryId);
        if (!$upgradeCategory) {
            throw $this->responseException(
                $this->responseError(new XenForo_Phrase('waindigo_requested_upgrade_category_not_found_userupgrades'), 404));
        }

        return $upgradeCategory;
    } /* END _getUpgradeCategoryOrError */

    /**
     * Get the upgrade categories model.
     *
     * @return Waindigo_UserUpgrades_Model_UpgradeCategory
     */
    protected function _getUpgradeCategoryModel()
    {
        return $this->getModelFromCache('Waindigo_UserUpgrades_Model_UpgradeCategory');
    } /* END _getUpgradeCategoryModel */
}