<?php

/**
 * We need to modify some default model behaviour to allow for advanced
 * functionality such as allowing multiple purchases.
 *
 * @see XenForo_Model_UserUpgrade
 */
class Waindigo_UserUpgrades_Extend_XenForo_Model_UserUpgrade extends XFCP_Waindigo_UserUpgrades_Extend_XenForo_Model_UserUpgrade
{

    public static $userId;

    /**
     *
     * @see XenForo_Model::__construct()
     */
    public function __construct()
    {
        parent::__construct();

        $_request = new Zend_Controller_Request_Http();
        $_input = new XenForo_Input($_request);

        $custom = $_input->filterSingle('custom', XenForo_Input::STRING);

        if ($custom) {
            $itemParts = explode(',', $custom, 4);
            if (count($itemParts) == 4) {
                list ($userId, $userUpgradeId, $validationType, $validation) = $itemParts;
                if ($userUpgradeId) {
                    $this->overridePayPalAddressForUpgradeById($userUpgradeId);
                }
            }
        }
    } /* END __construct */

    /**
     *
     * @param int $userUpgradeId
     */
    public function overridePayPalAddressForUpgradeById($userUpgradeId)
    {
        $payPalPrimaryAccount = $this->_getDb()->fetchOne('
			SELECT paypal_email
			FROM xf_user_upgrade
			WHERE user_upgrade_id = ?
		', $userUpgradeId);

        if ($payPalPrimaryAccount) {
            XenForo_Application::get('options')->set('payPalPrimaryAccount', $payPalPrimaryAccount);
        }
    } /* END overridePayPalAddressForUpgradeById */

    /**
     *
     * @see XenForo_Model_UserUpgrade::getUserUpgradesForPurchaseList()
     */
    public function getUserUpgradesForPurchaseList(array $viewingUser = null)
    {
        $purchased = array();
        $upgrades = array();

        $this->standardizeViewingUserReference($viewingUser);
        $upgrades = $this->getAllUserUpgrades();

        if ($viewingUser['user_id'] && $upgrades) {
            $activeUpgrades = $this->getActiveUserUpgradeRecordsForUser($viewingUser['user_id']);

            foreach ($upgrades as $upgradeId => $upgrade) {
                if (isset($activeUpgrades[$upgradeId])) {
                    // purchased
                    $purchased[$upgradeId] = $upgrades[$upgradeId];
                    $purchased[$upgradeId]['record'] = $activeUpgrades[$upgradeId];

                    // remove any upgrades disabled by this
                    if ($upgrade['disabled_upgrade_ids']) {
                        foreach (explode(',', $upgrade['disabled_upgrade_ids']) as $disabledId) {
                            unset($upgrades[$disabledId]);
                        }
                    }
                    if (! $upgrade['purchase_multiple']) {
                        unset($upgrades[$upgradeId]); // can't buy again
                    }
                } else
                    if (! $upgrade['can_purchase']) {
                        unset($upgrades[$upgradeId]);
                    }
                // removes any upgrades that users don't have access to
                if (isset($upgrades[$upgradeId]) && $upgrade['required_group_ids']) {
                    if (! self::_isMemberOfAnyOfTheGivenUserGroups($viewingUser, $upgrade['required_group_ids'])) {
                        unset($upgrades[$upgradeId]);
                    }
                }
                // remove tiered upgrades with zero cost
                $upgrade['cost_amount'] = self::_getTieredCost($upgrade, $viewingUser['user_id']);
                if (isset($upgrades[$upgradeId]) && $upgrade['cost_amount'] == 0) {
                    unset($upgrades[$upgradeId]);
                }
            }
        }
        if ($viewingUser['user_id'] == 0 && $upgrades) {
            foreach ($upgrades as $upgradeId => $upgrade) {
                if (isset($upgrades[$upgradeId]) && ! $upgrade['can_purchase']) {
                    unset($upgrades[$upgradeId]);
                } elseif (isset($upgrades[$upgradeId]) && $upgrade['required_group_ids']) {
                    if (! self::_isMemberOfAnyOfTheGivenUserGroups($viewingUser, $upgrade['required_group_ids'])) {
                        unset($upgrades[$upgradeId]);
                    }
                }
            }
        }

        return array(
            'available' => $upgrades,
            'purchased' => $purchased
        );
    } /* END getUserUpgradesForPurchaseList */

    /**
     *
     * @see XenForo_Model_UserUpgrade::getUserUpgradebyId();
     */
    public function getUserUpgradeById($id)
    {
        $upgrade = parent::getUserUpgradeById($id);

        $userId = self::$userId;

        if ($upgrade['tiered_upgrade'] && $userId) {
            $upgrade = self::_getTieredCost($upgrade, $userId);
        }

        return $upgrade;
    } /* END getUserUpgradeById */

    /**
     *
     * @see XenForo_Model_UserUpgrade::prepareUserUpgradeRecordConditions()
     *
     */
    public function prepareUserUpgradeRecordConditions(array $conditions, $baseTable, array &$fetchOptions)
    {
        $db = $this->_getDb();
        $sqlConditions = array();

        if (! empty($conditions['user_id'])) {

            if (is_array($conditions['user_id'])) {
                $sqlConditions[] = $baseTable . '.user_id IN(' . $db->quote($conditions['user_id']) . ')';
            } else {
                $sqlConditions[] = $baseTable . '.user_id = ' . $db->quote($conditions['user_id']);
            }
        }

        $conditionsForClause = parent::prepareUserUpgradeRecordConditions($conditions, $baseTable, $fetchOptions);
        if ($conditionsForClause == '1=1') {
            return $this->getConditionsForClause($sqlConditions);
        } else {
            return $conditionsForClause . ' AND ' . $this->getConditionsForClause($sqlConditions);
        }
    } /* END prepareUserUpgradeRecordConditions */

    /**
     *
     * @see XenForo_Model_UserUpgrade::prepareUserUpgrade()
     */
    public function prepareUserUpgrade(array $upgrade)
    {
        $upgrade['currency'] = strtoupper($upgrade['cost_currency']);
        $upgrade['length_unit_trial'] = '';
        $upgrade['length_amount_trial'] = 0;

        if ($upgrade['length_amount_post_trial'] && $upgrade['length_unit_post_trial']) {
            // if there is a trial...
            $upgrade['costAmountPP'] = $upgrade['cost_amount'];
            if ($upgrade['length_amount_post_trial'] == $upgrade['length_amount'] && $upgrade['length_unit_post_trial'] == $upgrade['length_unit'] && $upgrade['cost_amount_trial'] > 0) {
                $upgrade['cost_amount'] = number_format($upgrade['cost_amount'] - $upgrade['cost_amount_trial'], 2);
                $cost = "$upgrade[cost_amount_trial] $upgrade[currency]";
                $costPhrase = "waindigo_x_initial_fee_and_y_userupgrades";
                $costPhraseParams = array(
                    'cost' => $cost
                );
            } else {
                $upgrade['length_amount_trial'] = $upgrade['length_amount'];
                $upgrade['length_unit_trial'] = $upgrade['length_unit'];
                $upgrade['length_amount'] = $upgrade['length_amount_post_trial'];
                $upgrade['length_unit'] = $upgrade['length_unit_post_trial'];
                $tempCostAmount = $upgrade['cost_amount'] - $upgrade['cost_amount_trial'];
                $upgrade['cost_amount_trial'] = $upgrade['cost_amount'];
                $upgrade['cost_amount'] = number_format($tempCostAmount, 2);
                unset($tempCostAmount);
                if ($upgrade['cost_amount_trial'] + 0) {
                    $cost = "$upgrade[cost_amount_trial] $upgrade[currency]";
                } else {
                    $cost = new XenForo_Phrase("waindigo_free_userupgrades");
                }
                $trial = new XenForo_Phrase('x_for_y_' . $upgrade['length_unit_trial'] . 's', array(
                    'cost' => $cost,
                    'length' => $upgrade['length_amount_trial']
                ));
                $costPhrase = 'waindigo_x_then_y_userupgrades';
                $costPhraseParams = array(
                    'trial' => $trial
                );
            }
        } elseif ($upgrade['cost_amount_trial'] + 0) {
            $upgrade['costAmountPP'] = $upgrade['cost_amount'];
            $upgrade['cost_amount'] = number_format($upgrade['cost_amount'] - $upgrade['cost_amount_trial'], 2);
            $cost = "$upgrade[cost_amount_trial] $upgrade[currency]";
            $costPhrase = "waindigo_x_initial_fee_and_y_userupgrades";
            $costPhraseParams = array(
                'cost' => $cost
            );
        }

        switch ($upgrade['length_unit_trial']) {
            case 'day':
                $upgrade['lengthUnitTrialPP'] = 'D';
                break;
            case 'month':
                $upgrade['lengthUnitTrialPP'] = 'M';
                break;
            case 'year':
                $upgrade['lengthUnitTrialPP'] = 'Y';
                break;
            default:
                $upgrade['lengthUnitTrialPP'] = '';
                break;
        }

        $userId = XenForo_Visitor::getUserId();

        if ($upgrade['tiered_upgrade'] && $userId) {
            $upgrade = self::_getTieredCost($upgrade, $userId);
        }
        $upgrade = parent::prepareUserUpgrade($upgrade);

        if ($upgrade['length_amount'] && $upgrade['specific_end_date']) {
            $upgrade = self::_prepareSpecificEndDatePhrase($upgrade);
        }

        if (isset($costPhrase, $costPhraseParams)) {
            $upgrade['costPhrase'] = new XenForo_Phrase($costPhrase, array_merge(array(
                'then' => $upgrade['costPhrase']
            ), $costPhraseParams));
        }
        return $upgrade;
    } /* END prepareUserUpgrade */

    /**
     *
     * @see XenForo_Model_UserUpgrade::upgradeUser()
     */
    public function upgradeUser($userId, array $upgrade, $allowInsertUnpurchasable = false, $endDate = null)
    {
        $db = $this->_getDb();

        $active = $this->getActiveUserUpgradeRecord($userId, $upgrade['user_upgrade_id']);
        if ($active) {
            // updating an existing upgrade - if no end date override specified,
            // extend the upgrade
            $activeExtra = unserialize($active['extra']);
            if ($endDate === null) {
                if ($active['end_date'] == 0 || ! $activeExtra['length_unit']) {
                    $endDate = 0;
                } else {
                    $endDate = intval($endDate);
                    $endDate = strtotime('+' . $activeExtra['length_amount'] . ' ' . $activeExtra['length_unit'], $active['end_date']);
                }
            } else {
                $endDate = intval($endDate);
            }

            if ($upgrade['specific_end_date']) {
                $endDate = strtotime(date($upgrade['specific_end_date']) . ' 00:00:00');
            }
            if ($endDate != $active['end_date']) {
                $db->update('xf_user_upgrade_active', array(
                    'end_date' => $endDate
                ), 'user_id = ' . $db->quote($userId) . ' AND user_upgrade_id = ' . $db->quote($upgrade['user_upgrade_id']));
            }

            $upgradeRecordId = $active['user_upgrade_record_id'];
        } else {

            $upgradeRecordId = parent::upgradeUser($userId, $upgrade, $allowInsertUnpurchasable, $endDate);

            if (! empty($upgrade['specific_end_date'])) {
                /* @var $db Zend_Db_Adapter_Abstract */
                $db = $this->_getDb();
                XenForo_Db::beginTransaction($db);
                $db->update('xf_user_upgrade_active', array(
                    'end_date' => strtotime(date($upgrade['specific_end_date']) . ' 00:00:00')
                ), 'user_upgrade_record_id = ' . $db->quote($upgradeRecordId));
                XenForo_Db::commit($db);
            }

            if (XenForo_Application::get('options')->waindigo_userUpgrades_sendConfirmationEmail) {
                $this->_sendConfirmationEmail($upgrade, $userId);
            }

            if ($upgrade['permanent_group_ids']) {
                $this->_applyPermanentUserGroupChange($upgrade, $userId);
                // TODO - remove entries from xf_user_group_change?
                // $this->_removeMatchingUserGroupChangeRecords($upgrade, $userId);
            }

            $nodeExists = $this->_checkCongratulationsNodeExists();
            if ($upgradeRecordId && $nodeExists) {
                $this->_createCongratulationsThread($upgrade, $userId);
            }

            if ($upgradeRecordId && $upgrade['length_amount_post_trial'] && $upgrade['length_unit_post_trial']) {
                $extra = array(
                    'cost_amount' => $upgrade['cost_amount'] - $upgrade['cost_amount_trial'],
                    'cost_currency' => $upgrade['cost_currency'],
                    'length_amount' => $upgrade['length_amount_post_trial'],
                    'length_unit' => $upgrade['length_unit_post_trial']
                );

                if (empty($endDate)) {
                	$endDate = strtotime('+' . $upgrade['length_amount'] . ' ' . $upgrade['length_unit']);
				}

                /* @var $db Zend_Db_Adapter_Abstract */
                $db = $this->_getDb();
                XenForo_Db::beginTransaction($db);
                $db->update('xf_user_upgrade_active', array(
                    'user_id' => $userId,
                    'user_upgrade_id' => $upgrade['user_upgrade_id'],
                    'extra' => serialize($extra),
                    'start_date' => XenForo_Application::$time,
                    'end_date' => $endDate
                ), 'user_upgrade_record_id = ' . $db->quote($upgradeRecordId));
                XenForo_Db::commit($db);
            }
            if ($upgradeRecordId) {
                $this->updateRegistrationUserStateIfRequired($userId);
            }
        }

        return $upgradeRecordId;
    } /* END upgradeUser */

    /**
     * Get transaction log
     *
     * @param array $criteria
     * @param array $fetchOptions
     * @return array
     */
    public function getTransactionLog(array $criteria, array $fetchOptions = array())
    {
        $db = $this->_getDb();

        $limitOptions = $this->prepareLimitFetchOptions($fetchOptions);

        $whereClause = '';
        if (isset($criteria['search'])) {
            $s = $db->quote($criteria['search']);
            $whereClause = 'WHERE transaction_id LIKE ' . $s . ' OR message LIKE ' . $s . ' OR transaction_details LIKE ' . $s;
        }

        return $this->fetchAllKeyed($this->limitQueryResults('
				SELECT *
				FROM xf_user_upgrade_log
				' . $whereClause . '
				ORDER BY log_date DESC
			', $limitOptions['limit'], $limitOptions['offset']), 'user_upgrade_log_id');
    } /* END getTransactionLog */

    /**
     * Get specific log entry
     *
     * @param int $idTransaction
     * @return array
     */
    public function getTransactionLogEntry($idTransaction)
    {
        if (empty($idTransaction)) {
            return false;
        }

        return $this->_getDb()->fetchRow('
			SELECT
				*
			FROM xf_user_upgrade_log
			WHERE user_upgrade_log_id = ?
		', $idTransaction);
    } /* END getTransactionLogEntry */

    /**
     * Get number of log entries
     *
     * @return int
     */
    public function getTransactionLogCount()
    {
        return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM xf_user_upgrade_log
		');
    } /* END getTransactionLogCount */

    /**
     * Gets the XML representation of a user upgrade, including permissions.
     *
     * @param array $userUpgrade
     *
     * @return DOMDocument
     */
    public function getUserUpgradeXml(array $userUpgrade)
    {
        $document = new DOMDocument('1.0', 'utf-8');
        $document->formatOutput = true;

        $extraGroupTitles = self::_getUserGroupTitlesFromIds($userUpgrade['extra_group_ids']);
        $requiredGroupTitles = self::_getUserGroupTitlesFromIds($userUpgrade['required_group_ids']);
        $permanentGroupTitles = self::_getUserGroupTitlesFromIds($userUpgrade['permanent_group_ids']);

        $rootNode = $document->createElement('user_upgrade');
        $rootNode->setAttribute('required_group_titles', serialize($requiredGroupTitles));
        $rootNode->setAttribute('permanent_group_titles', serialize($permanentGroupTitles));
        $rootNode->setAttribute('notification_day', $userUpgrade['notification_day']);
        $rootNode->setAttribute('paypal_email', $userUpgrade['paypal_email']);
        $rootNode->setAttribute('length_unit_post_trial', $userUpgrade['length_unit_post_trial']);
        $rootNode->setAttribute('length_amount_post_trial', $userUpgrade['length_amount_post_trial']);
        $rootNode->setAttribute('length_unit', $userUpgrade['length_unit']);
        $rootNode->setAttribute('length_amount', $userUpgrade['length_amount']);
        $rootNode->setAttribute('specific_end_date', $userUpgrade['specific_end_date']);
        $rootNode->setAttribute('cost_currency', $userUpgrade['cost_currency']);
        $rootNode->setAttribute('cost_amount_trial', $userUpgrade['cost_amount_trial']);
        $rootNode->setAttribute('cost_amount', $userUpgrade['cost_amount']);
        $rootNode->setAttribute('recurring', $userUpgrade['recurring']);
        $rootNode->setAttribute('extra_group_titles', serialize($extraGroupTitles));
        $rootNode->setAttribute('tiered_upgrade', $userUpgrade['tiered_upgrade']);
        $rootNode->setAttribute('hidden', $userUpgrade['hidden']);
        $rootNode->setAttribute('display_order', $userUpgrade['display_order']);
        $rootNode->setAttribute('redirect', $userUpgrade['redirect']);
        $rootNode->setAttribute('agreement', $userUpgrade['agreement']);
        $rootNode->setAttribute('purchase_multiple', $userUpgrade['purchase_multiple']);
        $rootNode->setAttribute('description', $userUpgrade['description']);
        $rootNode->setAttribute('title', $userUpgrade['title']);

        $document->appendChild($rootNode);

        return $document;
    } /* END getUserUpgradeXml */

    /**
     * Imports a user upgrade XML file.
     *
     * @param SimpleXMLElement $document
     * @param integer $overwriteUserUpgradeId
     */
    public function importUserUpgradeXml(SimpleXMLElement $document, $overwriteUserUpgradeId = 0)
    {
        if ($document->getName() != 'user_upgrade') {
            throw new XenForo_Exception(new XenForo_Phrase('waindigo_provided_file_is_not_valid_user_upgrade_xml_userupgrades'), true);
        }

        $title = (string) $document['title'];
        if ($title === '') {
            throw new XenForo_Exception(new XenForo_Phrase('waindigo_provided_file_is_not_valid_user_upgrade_xml_userupgrades'), true);
        }

        $extraGroupIds = self::_getUserGroupIdsFromTitles(unserialize((string) $document['extra_group_titles']));
        $requiredGroupIds = self::_getUserGroupIdsFromTitles(unserialize((string) $document['required_group_titles']));
        $permanentGroupIds = self::_getUserGroupIdsFromTitles(unserialize((string) $document['permanent_group_titles']));

        $input = array(
            'title' => $title,
            'description' => (string) $document['description'],
            'purchase_multiple' => (boolean) $document['purchase_multiple'],
            'agreement' => (string) $document['agreement'],
            'redirect' => (string) $document['redirect'],
            'display_order' => (integer) $document['display_order'],
            'tiered_upgrade' => (boolean) $document['tiered_upgrade'],
            'permanent_group_ids' => $permanentGroupIds,
            'hidden' => (boolean) $document['hidden'],
            'extra_group_ids' => $extraGroupIds,
            'recurring' => (integer) $document['recurring'],
            'cost_amount' => (float) $document['cost_amount'],
            'cost_amount_trial' => (float) $document['cost_amount_trial'],
            'cost_currency' => (string) $document['cost_currency'],
            'length_amount' => (integer) $document['length_amount'],
            'length_unit' => (string) $document['length_unit'],
            'specific_end_date' => (string) $document['specific_end_date'],
            'length_amount_post_trial' => (integer) $document['length_amount_post_trial'],
            'length_unit_post_trial' => (string) $document['length_unit_post_trial'],
            'paypal_email' => (string) $document['paypal_email'],
            'notification_day' => (integer) $document['notification_day'],
            'required_group_ids' => $requiredGroupIds,
            'can_purchase' => 1
        );

        $dw = XenForo_DataWriter::create('XenForo_DataWriter_UserUpgrade');
        if ($overwriteUserUpgradeId) {
            $dw->setExistingData($overwriteUserUpgradeId);
        }
        $dw->bulkSet($input);
        $dw->save();
    } /* END importUserUpgradeXml */

    /**
     * Returns a list of active User Upgrades that meet the expiration criteria
     */
    public function notifyUsersOfExpiringUpgrades()
    {
        $conditions = array(
            'active' => true
        );
        $fetchOptions['join'] = XenForo_Model_UserUpgrade::JOIN_UPGRADE;

        $upgradeRecords = $this->getUserUpgradeRecords($conditions, $fetchOptions);

        foreach ($upgradeRecords as $upgrade) {
            if ($this->isExpiringUpgrade($upgrade)) {
                $this->notifyUserOfExpiringUpgrade($upgrade);
            }
        }
    } /* END notifyUsersOfExpiringUpgrades */

    /**
     * Checks if the given upgrade meets the expiration criteria
     *
     * @param array $upgrade
     *
     * @return boolean
     *
     */
    public function isExpiringUpgrade($upgrade)
    {
        $dayCount = 0;
        if ($upgrade['notification_day']) {
            $dayCount = $upgrade['notification_day'] - 1;
        } else
            if (XenForo_Application::get('options')->waindigo_userUpgrades_defaultNotificationDay) {
                $dayCount = XenForo_Application::get('options')->waindigo_userUpgrades_defaultNotificationDay - 1;
            }
        if (isset($dayCount) && ! $upgrade['recurring']) {
            // returns true only if within one day of alerting period - ie only alert once per upgrade
            $day = 60 * 60 * 24;
            $alertTime = $upgrade['end_date'] - ($dayCount * $day);
            $difference = $alertTime - XenForo_Application::$time;
            if ($difference >= 0 && $difference < $day) {
                return true;
            } else {
                return false;
            }
        }
        return false;
    } /* END isExpiringUpgrade */

    /**
     * Notifies user of expiring user upgrades based on user preference
     *
     * @param array $upgrade
     */
    public function notifyUserOfExpiringUpgrade(array $upgrade)
    {
        if (! $upgrade) {
            return false;
        }

        /* @var $userModel XenForo_Model_User */
        $userModel = XenForo_Model::create('XenForo_Model_User');
        $user = $userModel->getUserById($upgrade['user_id']);

        if (XenForo_Model_Alert::userReceivesAlert($user, 'user_upgrade', 'expiring')) {
            XenForo_Model_Alert::alert($user['user_id'], $user['user_id'], $user['username'], 'user_upgrade', $upgrade['user_upgrade_id'], 'expiring');
        }
    } /* END notifyUserOfExpiringUpgrade */

    public function editEndDate($upgradeRecord, $endDate)
    {
        return $this->_getDb()->query('
        UPDATE xf_user_upgrade_active
            SET
        end_date = ?
        	WHERE user_upgrade_record_id = ?
            ', array(
            $endDate,
            $upgradeRecord['user_upgrade_record_id']
        ));
    } /* END editEndDate */

    public function updateRegistrationUserStateIfRequired($userId)
    {
        /* @var $dw XenForo_DataWriter_User */
        $dw = XenForo_DataWriter::create('XenForo_DataWriter_User');
        $dw->setExistingData($userId);
        if ($dw->get('user_state') == 'email_confirm') {
            if (XenForo_Application::getOptions()->waindigo_userUpgrades_emailConfirmOverride) {
                $dw->advanceRegistrationUserState();
            }
        } elseif ($dw->get('user_state') == 'paypal_pending') {
            $request = new Zend_Controller_Request_Http();
            $input = new XenForo_Input($request);
            $payerEmail = $input->filterSingle('payer_email', XenForo_Input::STRING);
            if ($payerEmail == $dw->get('email')) {
                $allowEmailConfirm = false;
            } else {
                $allowEmailConfirm = true;
            }
            $dw->advanceRegistrationUserState($allowEmailConfirm);
            if ($dw->get('user_state') == 'email_confirm') {
                if (XenForo_Application::getOptions()->waindigo_userUpgrades_emailConfirmOverride) {
                    $dw->advanceRegistrationUserState(false);
                } else {
                    $user = $dw->getMergedData();
                    /* @var $userConfirmationModel XenForo_Model_UserConfirmation */
                    $userConfirmationModel = $this->getModelFromCache('XenForo_Model_UserConfirmation');
                    $userConfirmationModel->sendEmailConfirmation($user);
                }
            }
        }
        $dw->save();
    } /* END updateRegistrationUserStateIfRequired */

    /**
     * Gets the user group model object.
     *
     * @return XenForo_Model_UserGroup
     */
    protected function _getUserGroupModel()
    {
        return $this->getModelFromCache('XenForo_Model_UserGroup');
    } /* END _getUserGroupModel */

    /**
     * Gets the node model object.
     *
     * @return XenForo_Model_Node
     */
    protected function _getNodeModel()
    {
        return $this->getModelFromCache('XenForo_Model_Node');
    } /* END _getNodeModel */

    /**
     *
     * @see XenForo_Model_User::isMemberOfUserGroup()
     *
     */
    protected function _isMemberOfAnyOfTheGivenUserGroups($viewingUser, $userGroupIds)
    {
        /* @var $userModel XenForo_Model_User */
        $userModel = XenForo_Model::create('XenForo_Model_User');
        $response = false;
        foreach (explode(',', $userGroupIds) as $userGroupId) {
            if ($userModel->isMemberOfUserGroup($viewingUser, $userGroupId)) {
                $response = true;
            }
        }
        return $response;
    } /* END _isMemberOfAnyOfTheGivenUserGroups */

    protected function _prepareSpecificEndDatePhrase($upgrade)
    {
        $cost = "$upgrade[cost_amount] $upgrade[currency]";

        $endDate = strtotime(date($upgrade['specific_end_date']) . ' 00:00:00');

        $datePhrase = XenForo_Locale::dateTime($endDate);

        $upgrade['costPhrase'] = new XenForo_Phrase("waindigo_until_specific_end_date_userupgrades", array(
            'cost' => $cost,
            'specific_end_date' => $datePhrase
        ));

        return $upgrade;
    } /* END _prepareSpecificEndDatePhrase */

    protected function _createCongratulationsThread($upgrade, $userId)
    {
        /* @var $userModel XenForo_Model_User */
        $userModel = XenForo_Model::create('XenForo_Model_User');
        $user = $userModel->getUserById($userId);
        $username = $user['username'];
        $nodeId = XenForo_Application::get('options')->waindigo_userUpgrades_congratsMessageNodeId;
        $userGroupTitles = implode(', ', self::_getUserGroupTitlesFromIds($upgrade['extra_group_ids']));

        $writer = XenForo_DataWriter::create('XenForo_DataWriter_Discussion_Thread', 'ERROR_SILENT');
        $writer->bulkSet(array(
            'user_id' => $userId,
            'username' => $username,
            'title' => new XenForo_Phrase("waindigo_congratulations_thread_title_userupgrades", array(
                'username' => $username,
                'usergroup' => $userGroupTitles
            )),
            'node_id' => $nodeId
        ));
        $postWriter = $writer->getFirstMessageDw();
        $postWriter->set('message', new XenForo_Phrase("waindigo_congratulations_thread_message_userupgrades", array(
            'username' => $username,
            'usergroup' => $userGroupTitles,
            'boardTitle' => XenForo_Application::getOptions()->boardTitle
        )));
        $writer->save();
    } /* END _createCongratulationsThread */

    protected function _getUserGroupTitlesFromIds($userGroupIds)
    {
        $allUserGroupTitles = $this->_getUserGroupModel()->getAllUserGroupTitles();
        $userGroupTitles = array();
        if ($userGroupIds) {
            $userGroupIds = explode(',', $userGroupIds);
            foreach ($userGroupIds as $userGroupId) {
                $userGroupTitles[] = $allUserGroupTitles[$userGroupId];
            }
            $userGroupTitles = array_unique($userGroupTitles);
        }
        return $userGroupTitles;
    } /* END _getUserGroupTitlesFromIds */

    protected function _getUserGroupIdsFromTitles($userGroupTitles)
    {
        $allUserGroupTitles = $this->_getUserGroupModel()->getAllUserGroupTitles();
        $userGroupIds = array();
        foreach ($allUserGroupTitles as $userGroupId => $userGroupTitle) {
            if (in_array($userGroupTitle, $userGroupTitles)) {
                $userGroupIds[] = $userGroupId;
            }
        }
        return $userGroupIds;
    } /* END _getUserGroupIdsFromTitles */

    protected function _getTieredCost($upgrade, $userId)
    {
        $db = XenForo_Application::get('db');

        $tieredCostReduction = $db->fetchOne('
            SELECT MAX(upgrade.cost_amount)
            FROM xf_user_upgrade_active AS upgrade_active
            INNER JOIN xf_user_upgrade AS upgrade ON (upgrade_active.user_upgrade_id=upgrade.user_upgrade_id)
            WHERE user_id = ? AND tiered_upgrade = 1 AND cost_currency = ? AND upgrade.user_upgrade_id != ?
            ', array(
            $userId,
            $upgrade['cost_currency'],
            $upgrade['user_upgrade_id']
        ));

        if ($tieredCostReduction) {
            $upgrade['cost_amount'] = number_format($upgrade['cost_amount'] - $tieredCostReduction, 2);
            if ($upgrade['cost_amount'] < 0) {
                $upgrade['cost_amount'] = number_format(0, 2);
            }
        }

        return $upgrade;
    } /* END _getTieredCost */

    protected function _checkCongratulationsNodeExists()
    {
        $nodeModel = $this->_getNodeModel();

        $node = $nodeModel->getNodeById(XenForo_Application::get('options')->waindigo_userUpgrades_congratsMessageNodeId);

        return (! empty($node) && is_array($node) && array_key_exists('node_id', $node) && array_key_exists('parent_node_id', $node));
    } /* END _checkCongratulationsNodeExists */

    protected function _sendConfirmationEmail($upgrade, $userId)
    {
        /* @var $userModel XenForo_Model_User */
        $userModel = XenForo_Model::create('XenForo_Model_User');
        $user = $userModel->getUserById($userId);

        $params = array(
            'username' => $user['username'],
            'upgrade' => $upgrade,
            'boardTitle' => XenForo_Application::get('options')->boardTitle,
            'boardUrl' => XenForo_Application::get('options')->boardUrl
        );

        $mail = XenForo_Mail::create('waindigo_user_upgrade_confirmation_userupgrades', $params, $user['language_id']);

        return $mail->send($user['email'], $user['username']);
    } /* END _sendConfirmationEmail */

    protected function _applyPermanentUserGroupChange($upgrade, $userId)
    {
        /* @var $writer XenForo_DataWriter_User */
        $writer = XenForo_DataWriter::create('XenForo_DataWriter_User');
        $writer->setExistingData($userId);
        $writer->setOption(XenForo_DataWriter_User::OPTION_ADMIN_EDIT, true);
        $secondaryGroupIds = $writer->get('secondary_group_ids');
        if ($secondaryGroupIds) {
            $secondaryGroupIds = explode(',', $secondaryGroupIds);
        } else {
            $secondaryGroupIds = array();
        }
        $extraUserGroupIds = explode(',', $upgrade['permanent_group_ids']);
        $secondaryGroupIds = array_merge($secondaryGroupIds, $extraUserGroupIds);

        $writer->setSecondaryGroups($secondaryGroupIds);

        $writer->save();
    } /* END _applyPermanentUserGroupChange */

    /**
     * Maybe introduce this to check for any records matching current criteria
     * and removing them
     *
     * This could cause some adverse issues along the way...
     */
    protected function _removeMatchingUserGroupChangeRecords($upgrade, $userId)
    {
        // Something like this, could use XF functions also see XenForo_Model_User
        $this->_getDb()->query('
            DELETE *
            FROM xf_user_group_change
            WHERE user_id = ?
            AND group_ids = ?', array(
            $userId,
            $upgrade['permanent_group_ids']
        ));
    } /* END _removeMatchingUserGroupChangeRecords */
}