<?php

/**
 * Model for upgrade categories.
 */
class Waindigo_UserUpgrades_Model_UpgradeCategory extends XenForo_Model
{

    /**
     * Gets upgrade categories that match the specified criteria.
     *
     * @param array $conditions List of conditions.
     * @param array $fetchOptions
     *
     * @return array [upgrade category id] => info.
     */
    public function getUpgradeCategories(array $conditions = array(), array $fetchOptions = array())
    {
        $whereClause = $this->prepareUpgradeCategoryConditions($conditions, $fetchOptions);
        
        $sqlClauses = $this->prepareUpgradeCategoryFetchOptions($fetchOptions);
        $limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
        
        return $this->fetchAllKeyed(
            $this->limitQueryResults(
                '
                SELECT upgrade_category.*
                    ' . $sqlClauses['selectFields'] . '
                FROM xf_upgrade_category AS upgrade_category
                ' . $sqlClauses['joinTables'] . '
                WHERE ' . $whereClause . '
                ' . $sqlClauses['orderClause'] . '
            ', $limitOptions['limit'], $limitOptions['offset']), 
            'upgrade_category_id');
    } /* END getUpgradeCategories */

    /**
     * Gets the upgrade category that matches the specified criteria.
     *
     * @param array $conditions List of conditions.
     * @param array $fetchOptions Options that affect what is fetched.
     *
     * @return array false
     */
    public function getUpgradeCategory(array $conditions = array(), array $fetchOptions = array())
    {
        $upgradeCategories = $this->getUpgradeCategories($conditions, $fetchOptions);
        
        return reset($upgradeCategories);
    } /* END getUpgradeCategory */

    /**
     * Gets a upgrade category by ID.
     *
     * @param integer $upgradeCategoryId
     * @param array $fetchOptions Options that affect what is fetched.
     *
     * @return array false
     */
    public function getUpgradeCategoryById($upgradeCategoryId, array $fetchOptions = array())
    {
        $conditions = array(
            'upgrade_category_id' => $upgradeCategoryId
        );
        
        return $this->getUpgradeCategory($conditions, $fetchOptions);
    } /* END getUpgradeCategoryById */

    /**
     * Gets the total number of a upgrade category that match the specified
     * criteria.
     *
     * @param array $conditions List of conditions.
     *
     * @return integer
     */
    public function countUpgradeCategories(array $conditions = array())
    {
        $fetchOptions = array();
        
        $whereClause = $this->prepareUpgradeCategoryConditions($conditions, $fetchOptions);
        $joinOptions = $this->prepareUpgradeCategoryFetchOptions($fetchOptions);
        
        $limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
        
        return $this->_getDb()->fetchOne(
            '
            SELECT COUNT(*)
            FROM xf_upgrade_category AS upgrade_category
            ' . $joinOptions['joinTables'] . '
            WHERE ' . $whereClause . '
        ');
    } /* END countUpgradeCategories */

    /**
     * Gets all upgrade categories titles.
     *
     * @return array [upgrade category id] => title.
     */
    public static function getUpgradeCategoryTitles()
    {
        $upgradeCategories = XenForo_Model::create(__CLASS__)->getUpgradeCategories();
        $titles = array();
        foreach ($upgradeCategories as $upgradeCategoryId => $upgradeCategory) {
            $titles[$upgradeCategoryId] = $upgradeCategory['title'];
        }
        return $titles;
    } /* END getUpgradeCategoryTitles */

    /**
     * Gets the default upgrade category record.
     *
     * @return array
     */
    public function getDefaultUpgradeCategory()
    {
        return array(
            'upgrade_category_id' => '', /* END 'upgrade_category_id' */
            'display_order' => 1, /* 'display_order' */
        );
    } /* END getDefaultUpgradeCategory */
    
    public function groupUpgradeCategoriesByParent(array $upgradeCategories)
    {
        $parentUpgradeCategories = array();
        foreach ($upgradeCategories as $upgradeCategoryId => $upgradeCategory) {
            if ($upgradeCategory['parent_category_id'] == 0) {
                $parentUpgradeCategories[$upgradeCategoryId] = $upgradeCategory;
            } else {
                $parentUpgradeCategories[$upgradeCategory['parent_category_id']]['upgrade_categories'][$upgradeCategoryId] = $upgradeCategory;
            }
        }
        
        return $parentUpgradeCategories;
    } /* END groupUpgradeCategoriesByParent */

    /**
     * Prepares a set of conditions to select upgrade categories against.
     *
     * @param array $conditions List of conditions.
     * @param array $fetchOptions The fetch options that have been provided. May
     * be edited if criteria requires.
     *
     * @return string Criteria as SQL for where clause
     */
    public function prepareUpgradeCategoryConditions(array $conditions, array &$fetchOptions)
    {
        $db = $this->_getDb();
        $sqlConditions = array();
        
        if (isset($conditions['upgrade_category_ids']) && !empty($conditions['upgrade_category_ids'])) {
            $sqlConditions[] = 'upgrade_category.upgrade_category_id IN (' . $db->quote(
                $conditions['upgrade_category_ids']) . ')';
        } else 
            if (isset($conditions['upgrade_category_id'])) {
                $sqlConditions[] = 'upgrade_category.upgrade_category_id = ' .
                     $db->quote($conditions['upgrade_category_id']);
            }
        
        if (isset($conditions['parent_category_ids']) && !empty($conditions['parent_category_ids'])) {
            $sqlConditions[] = 'upgrade_category.parent_category_id IN (' . $db->quote(
                $conditions['parent_category_ids']) . ')';
        } else 
            if (isset($conditions['parent_category_id'])) {
                $sqlConditions[] = 'upgrade_category.parent_category_id = ' .
                     $db->quote($conditions['parent_category_id']);
            }
        
        $this->_prepareUpgradeCategoryConditions($conditions, $fetchOptions, $sqlConditions);
        
        return $this->getConditionsForClause($sqlConditions);
    } /* END prepareUpgradeCategoryConditions */

    /**
     * Method designed to be overridden by child classes to add to set of
     * conditions.
     *
     * @param array $conditions List of conditions.
     * @param array $fetchOptions The fetch options that have been provided. May
     * be edited if criteria requires.
     * @param array $sqlConditions List of conditions as SQL snippets. May be
     * edited if criteria requires.
     */
    protected function _prepareUpgradeCategoryConditions(array $conditions, array &$fetchOptions, array &$sqlConditions)
    {
    } /* END _prepareUpgradeCategoryConditions */

    /**
     * Checks the 'join' key of the incoming array for the presence of the
     * FETCH_x bitfields in this class
     * and returns SQL snippets to join the specified tables if required.
     *
     * @param array $fetchOptions containing a 'join' integer key built from
     * this class's FETCH_x bitfields.
     *
     * @return string containing selectFields, joinTables, orderClause keys.
     * Example: selectFields = ', user.*, foo.title'; joinTables = ' INNER JOIN
     * foo ON (foo.id = other.id) '; orderClause = 'ORDER BY x.y'
     */
    public function prepareUpgradeCategoryFetchOptions(array &$fetchOptions)
    {
        $selectFields = '';
        $joinTables = '';
        $orderBy = '';
        
        if (!empty($fetchOptions['order'])) {
            switch ($fetchOptions['order']) {
                case 'default':
                    $orderBy = 'upgrade_category.parent_category_id, upgrade_category.display_order';
                    break;
                
                case 'parent_category_id':
                    $orderBy = 'upgrade_category.' . $fetchOptions['order'];
                    break;
                
                case 'title':
                default:
                    $orderBy = 'upgrade_category.title';
            }
            if (!isset($fetchOptions['orderDirection']) || $fetchOptions['orderDirection'] == 'asc') {
                $orderBy .= ' ASC';
            } else {
                $orderBy .= ' DESC';
            }
        }
        
        $this->_prepareUpgradeCategoryFetchOptions($fetchOptions, $selectFields, $joinTables, $orderBy);
        
        return array(
            'selectFields' => $selectFields,
            'joinTables' => $joinTables,
            'orderClause' => ($orderBy ? "ORDER BY $orderBy" : '')
        );
    } /* END prepareUpgradeCategoryFetchOptions */

    /**
     * Method designed to be overridden by child classes to add to SQL snippets.
     *
     * @param array $fetchOptions containing a 'join' integer key built from
     * this class's FETCH_x bitfields.
     * @param string $selectFields = ', user.*, foo.title'
     * @param string $joinTables = ' INNER JOIN foo ON (foo.id = other.id) '
     * @param string $orderBy = 'x.y ASC, x.z DESC'
     */
    protected function _prepareUpgradeCategoryFetchOptions(array &$fetchOptions, &$selectFields, &$joinTables, &$orderBy)
    {
    } /* END _prepareUpgradeCategoryFetchOptions */
}