<?php

class XfAddOns_LinkEssentials_Util_UrlParser
{
	
	/**
	 * In a string, look for a [URL] tag, if present, fetch the title for the page, and
	 * change the URL to reflect the title
	 */
	public function substituteUrls($message)
	{
		$options = XenForo_Application::getOptions();
		$matches = array();
		preg_match_all('#\[URL([^]]*)\](.*)\[/URL\]#isU', $message, $matches);

		$ret = var_export($matches, true);
		XfAddOns_LinkEssentials_Log::trace($ret);
		
		if (empty($matches[1]))
		{
			XfAddOns_LinkEssentials_Log::log("No urls found on the post");
			return;
		}
		
		// get a copy of the message
		$newMessage = $message;
		
		// remove all the favicon that are already present
		$newMessage = preg_replace("#\[LINKFAVICON\].*\[/LINKFAVICON\]#iU", "", $newMessage);
		
		// start parsing all the matches
		for ($i = 0; $i < count($matches[0]); $i++)
		{
			if (!empty($matches[1][$i]))
			{
				$url = $matches[1][$i];
				if ($this->normalizeUrl($matches[1][$i]) != $this->normalizeUrl($matches[2][$i]))
				{
					XfAddOns_LinkEssentials_Log::debug("URL ignored because a title is specified " . $matches[1][$i]);
					continue;
				}
			}
			else
			{
				$url = $matches[2][$i];
			}
			
			// we have a match
			$url = $this->normalizeUrl($url);
			XfAddOns_LinkEssentials_Log::log("Found the url " . $url);
			if (!$this->isValidUrl($url))
			{
				continue;
			}
			
			// Fetch the data for the external resource
			$externalResource = $this->getMainResource($url);
			if (!$externalResource)
			{
				continue;		// an error
			}
			
			if ($externalResource->isPage === true)
			{
				$title = $externalResource->getTitle();
				if (empty($title))
				{
					// We could not get the external title
					continue;
				}
				
				XfAddOns_LinkEssentials_Log::log("Title for the " . $url . " is " . $title);
				$replacement = "[URL=" . $url . "]" . $title . "[/URL]";
				$newMessage = preg_replace("#" . preg_quote($matches[0][$i], '#') . "#", $replacement, $newMessage);
				continue;
			}
			if ($externalResource->isImage === true)
			{
				XfAddOns_LinkEssentials_Log::log("Found an image at: " . $url);
				$replacement = "[IMG]" . $url . "[/IMG]";
				$newMessage = preg_replace("#" . preg_quote($matches[0][$i], '#') . "#", $replacement, $newMessage);
			}
		}
		
		XfAddOns_LinkEssentials_Log::debug("Replaced message: " . $newMessage);
		return $newMessage;
	}
	
	/**
	 * Normalize the Url by removing quotes and the leading equals sign if present
	 * @param string $url	The URL embedded in the bb tag
	 */
	protected function normalizeUrl($url)
	{
		// remove the leading equals sign if present
		if (substr($url, 0, 1) == '=')
		{
			$url = substr($url, 1);
		}
		
		// remove quotes if added
		$matches = array();
		if (preg_match("#^'(.*)'$#", $url, $matches))
		{
			$url = $matches[1];
		}
		if (preg_match('#^"(.*)"$#', $url, $matches))
		{
			$url = $matches[1];
		}

		// Add http to the beginning if it is missing
		if (!preg_match('#^http#', $url))
		{
			$url = 'http://' . $url;
		}		
		return $url;
	}

	/**
	 * Check if a URL is valid. At least we need a scheme and host to be able to query it
	 * @param string $url	The URL to validate
	 * @return boolean		true, if this is a valid URL
	 */
	protected function isValidUrl($url)
	{
		if (empty($url))
		{
			return false;
		}
		$urlData = parse_url($url);
		return !empty($urlData['scheme']) && !empty($urlData['host']);
	}

	/**
	 * Returns the domain for a given URL
	 * @param string $url
	 */
	public function getDomain($url)
	{
		$urlData = parse_url($url);
		if (empty($urlData) || empty($urlData['host']))
		{
			return '';
		}
		
		XfAddOns_LinkEssentials_Log::trace($urlData);
		$domainUrl = '';
		$domainUrl .= isset($urlData['scheme']) ? $urlData['scheme'] : 'http';
		$domainUrl .= "://";
		$domainUrl .= $urlData['host'];
		$domainUrl .= isset($urlData['port']) ? (':' . $urlData['port']) : '';
		return $domainUrl; 
	}
	
	/**
	 * Parses a URL and figures out which kind of content we are pointing to. If the content is an
	 * HTML page, then the page is retrieve, or if the content points to an image, an special flag
	 * is set to indicate that an external image is being referenced
	 *
	 * @param string $url		The URL of the page that we will fetch
	 * @return		XfAddOns_LinkEssentials_ExternalResource
	 */
	protected function getMainResource($url)
	{
		try
		{
			// The response, a wrapper for all the metadata
			$ret = new XfAddOns_LinkEssentials_ExternalResource();
			$ret->url = $url;
			
			/* @var $response Zend_Http_Response */
			$response = XenForo_Helper_Http::getClient($url, array( 'timeout' => 10))->setHeaders('Accept-encoding', 'identity')->request('GET');
			
			// Log headers
			$headers = $response->getHeaders();
			$headersStr = var_export($headers, true);
			XfAddOns_LinkEssentials_Log::trace($headersStr);
			
			// If response is not 200, do not proceed
			if ($response->getStatus() !== 200)
			{
				XfAddOns_LinkEssentials_Log::log("Page ignored, got response code: " . $response->getStatus());
				return $ret;
			}
			
			// Options
			$options = XenForo_Application::getOptions();
			
			$contentType = $response->getHeader('Content-type');
			if (preg_match('#image#i', $contentType))
			{
				$ret->isImage = true;
				XfAddOns_LinkEssentials_Log::log("The link points to an image, mime type: " . $contentType);
				return $ret;
			}
			if (!preg_match('#text/html#', $contentType))
			{
				XfAddOns_LinkEssentials_Log::log("Page ignored, mime type: " . $contentType);
				return $ret;
			}
			
			// Check the content length
			$length = $response->getHeader('Content-length');
			XfAddOns_LinkEssentials_Log::log("Content Length: " . $length);
			if ($length > $options->xfa_le_max_content_length * 1024)
			{
				return $ret;
			}
			
			// Consume the page and return the content
			XfAddOns_LinkEssentials_Log::trace("-------\r\n" . $response . "\r\n--------");
			$ret->isPage = true;
			$ret->pageContent = $response->getBody();
			return $ret;			
		}
		catch (Exception $ex)
		{
			XfAddOns_LinkEssentials_Log::warn($ex->getMessage());
			XenForo_Error::logException($ex, false);
			return null;
		}
	}	

	/**
	 * Checks if an external resource exists
	 * @param string $url		The URL for the resource
	 * @return boolean			true, if the resource exists
	 */
	protected function resourceExists($url)
	{
		XfAddOns_LinkEssentials_Log::log("Checking if resource exists: " . $url);
		$response = XenForo_Helper_Http::getClient($url, array( 'timeout' => 10))->setHeaders('Accept-encoding', 'identity')->request('GET');
		return $response->getStatus() === 200;		
	}
	
	
	
	
}