<?php

class robodn_DonationManager_Model_Goal extends XenForo_Model
{
	const FETCH_LOG_COUNT = 0x01;

	public function getGoalById($goalId, array $fetchOptions = array())
	{
		$joinOptions = $this->prepareGoalFetchOptions($fetchOptions);

		return $this->_getDb()->fetchRow('
			SELECT goal.*
				' . $joinOptions['selectFields'] . '
			FROM robodn_goal AS goal
				' . $joinOptions['joinTables'] . '
			WHERE goal.goal_id = ?
			GROUP BY (goal.goal_id)
		', $goalId);
	}

	public function getGoals(array $conditions = array(), array $fetchOptions = array())
	{
		$whereClause = $this->prepareGoalConditions($conditions, $fetchOptions);
		//$orderClause = $this->prepareGoalOrderOptions($fetchOptions, '');
		$joinOptions = $this->prepareGoalFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);

		return $this->fetchAllKeyed($this->limitQueryResults(
			'
				SELECT goal.*
					' . $joinOptions['selectFields'] . '
				FROM robodn_goal AS goal
				' . $joinOptions['joinTables'] . '
				WHERE ' . $whereClause . '
				GROUP BY (goal.goal_id)
				ORDER BY display_order ASC
			', $limitOptions['limit'], $limitOptions['offset']
		), 'goal_id');
	}

	public function prepareGoal(array $goal)
	{
		$goal['realPercentComplete'] = $goal['donated'] / $goal['goal'] * 100;
		$goal['percentComplete'] = max(0, min(100, $goal['realPercentComplete']));

		// If use this for logs too, if title is missing it is a log so ignore that stuff
		if (isset($goal['title']))
		{
			$username = XenForo_Visitor::getInstance()->username;
			$tokens = array(
				'{name}' => $username !== '' ? $username : new XenForo_Phrase('guest')
			);

			$goal['title'] = str_replace(array_keys($tokens), $tokens, $goal['title']);
			$goal['description'] = str_replace(array_keys($tokens), $tokens, $goal['description']);
		}

		return $goal;
	}

	public function prepareGoals(array $goals)
	{
		foreach ($goals AS &$goal)
		{
			$goal = $this->prepareGoal($goal);
		}

		return $goals;
	}

	public function prepareGoalConditions(array $conditions, array &$fetchOptions)
	{
		$db = $this->_getDb();
		$sqlConditions = array();

		if (isset($conditions['feature']))
		{
			$sqlConditions[] = 'goal.feature = ' . $db->quote($conditions['feature']);
		}

		if (isset($conditions['archived']))
		{
			$sqlConditions[] = 'goal.archived = ' . $db->quote($conditions['archived']);
		}

		if (!empty($conditions['hasEndDate']))
		{
			$sqlConditions[] = 'goal.end_date != 0';
		}

		if (!empty($conditions['hideFuture']))
		{
			$sqlConditions[] = 'goal.start_date < ' . $db->quote(XenForo_Application::$time);
		}

		return $this->getConditionsForClause($sqlConditions);
	}

	public function prepareGoalFetchOptions(array $fetchOptions)
	{
		$selectFields = '';
		$joinTables = '';

		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_LOG_COUNT)
			{
				$selectFields .= ',
					COUNT(log.goal_log_id) AS log_count';
				$joinTables .= '
					LEFT JOIN robodn_goal_log AS log ON (log.goal_id = goal.goal_id)';
			}
		}

		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables
		);
	}

	public function getGoalLogById($logId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM robodn_goal_log
			WHERE goal_log_id = ?
		', $logId);
	}

	public function getGoalLogs(array $conditions = array(), array $fetchOptions = array())
	{
		$whereClause = $this->prepareGoalLogConditions($conditions, $fetchOptions);
		//$orderClause = $this->prepareGoalLogOrderOptions($fetchOptions, '');
		$joinOptions = $this->prepareGoalLogFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);

		return $this->fetchAllKeyed($this->limitQueryResults(
			'
				SELECT log.*
					' . $joinOptions['selectFields'] . '
				FROM robodn_goal_log AS log
				' . $joinOptions['joinTables'] . '
				WHERE ' . $whereClause . '
				ORDER BY start_date DESC
			', $limitOptions['limit'], $limitOptions['offset']
		), 'goal_log_id');
	}

	public function prepareGoalLogConditions(array $conditions, array &$fetchOptions)
	{
		$db = $this->_getDb();
		$sqlConditions = array();

		if (isset($conditions['goal']))
		{
			$sqlConditions[] = 'log.goal_id = ' . $db->quote($conditions['goal']);
		}

		return $this->getConditionsForClause($sqlConditions);
	}

	public function prepareGoalLogFetchOptions(array $fetchOptions)
	{
		$selectFields = '';
		$joinTables = '';

		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables
		);
	}

	public function countGoalLogs(array $conditions = array())
	{
		$fetchOptions = array();
		$whereClause = $this->prepareGoalLogConditions($conditions, $fetchOptions);

		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM robodn_goal_log AS log
			WHERE ' . $whereClause . '
		');
	}

	public function removeDonationFromGoalIfChanged(array $old, array $new)
	{
		// No old goal? Return true if we have a new one
		if (empty($old['goal_id']))
		{
			return !empty($new['goal_id']) AND $new['goal_id'] != $old['goal_id'];
		}

		$goal = $this->getGoalById($old['goal_id']);
		// Old goal gone? Return true if we have a new one once again
		if (!$goal)
		{
			return !empty($new['goal_id']) AND $new['goal_id'] != $old['goal_id'];
		}

		$logs = $this->getGoalLogs(array('goal' => $goal['goal_id']));

		// If the goal is the same and the time changed but still same period then return false
		$removeDonation = false;
		if ($old['goal_id'] == $new['goal_id'])
		{
			// Get the goal/log that the old date was in...
			if ($old['donation_date'] >= $goal['start_date'] AND (empty($goal['end_date']) OR $old['donation_date'] <= $goal['end_date']))
			{
				if ($new['donation_date'] >= $goal['start_date'] AND (empty($goal['end_date']) OR $new['donation_date'] <= $goal['end_date']))
				{
					// It's in the same goal at same period
					return false;
				}
				else
				{
					$removeDonation = 'goal';
				}
			}
			else
			{
				foreach ($logs AS $log)
				{
					if ($old['donation_date'] >= $log['start_date'] AND (empty($log['end_date']) OR $old['donation_date'] <= $log['end_date']))
					{
						if ($new['donation_date'] >= $log['start_date'] AND (empty($log['end_date']) OR $new['donation_date'] <= $log['end_date']))
						{
							// It's in the same log
							return false;
						}
						else
						{
							$removeDonation = $log['goal_log_id'];
							break;
						}
					}
				}
			}
		}
		else
		{
			if ($old['donation_date'] >= $goal['start_date'] AND (empty($goal['end_date']) OR $old['donation_date'] <= $goal['end_date']))
			{
				$removeDonation = 'goal';
			}
			else
			{
				foreach ($logs AS $log)
				{
					if ($old['donation_date'] >= $log['start_date'] AND (empty($log['end_date']) OR $old['donation_date'] <= $log['end_date']))
					{
						$removeDonation = $log['goal_log_id'];
					}
				}
			}
		}

		if (!$removeDonation)
		{
			// Didn't match anything but still need to add new stuff
			return !empty($new['goal_id']);
		}

		// Remove the old amount
		if ($removeDonation == 'goal')
		{
			// We don't want to trigger any postsave stuff and I'm being lazy so straight up query
			$this->_getDb()->query('
				UPDATE robodn_goal
				SET donated = ?
				WHERE goal_id = ?
			', array(max(0, $goal['donated'] - $old['amount']), $goal['goal_id']));
		}
		else
		{
			$this->_getDb()->query('
				UPDATE robodn_goal_log
				SET donated = donated - ?
				WHERE goal_log_id = ?
			', array($old['amount'], $removeDonation));
		}

		return true;
	}

	public function addDonationToGoal(array $donation)
	{
		if (empty($donation['goal_id']))
		{
			return;
		}

		$goal = $this->getGoalById($donation['goal_id']);
		if (!$goal)
		{
			return;
		}

		// Current goal? Just add the donation and return
		if ($donation['donation_date'] >= $goal['start_date'] AND (empty($goal['end_date']) OR $donation['donation_date'] <= $goal['end_date']))
		{
			$dw = XenForo_DataWriter::create('robodn_DonationManager_DataWriter_Goal');
			$dw->setExistingData($goal['goal_id']);
			$dw->set('donated', $dw->get('donated') + $donation['amount']);
			$dw->save();

			return;
		}

		// Must be from a log, try find it and update and if not... too bad
		$logs = $this->getGoalLogs(array('goal' => $goal['goal_id']));
		foreach ($logs AS $log)
		{
			if ($donation['donation_date'] >= $log['start_date'] AND (empty($log['end_date']) OR $donation['donation_date'] <= $log['end_date']))
			{
				$this->_getDb()->query('
					UPDATE robodn_goal_log
					SET donated = donated + ?
					WHERE goal_log_id = ?
				', array($donation['amount'], $log['goal_log_id']));
				return;
			}
		}
	}

	public function archiveAndRestartEndedGoals()
	{
		$goals = $this->getGoals(array('archived' => 0, 'hasEndDate' => true));
		foreach ($goals AS $goal)
		{
			if ($goal['end_date'] <= XenForo_Application::$time)
			{
				if ($goal['recurring'])
				{
					$this->logAndRestartGoal($goal['goal_id']);
				}
				else
				{
					$this->archiveGoal($goal['goal_id']);
				}
			}
		}
	}

	public function logAndRestartGoal($goalId)
	{
		$goal = $this->getGoalById($goalId);
		if (!$goal)
		{
			return;
		}

		$this->_getDb()->insert(
			'robodn_goal_log',
			array('goal_id' => $goal['goal_id'], 'goal' => $goal['goal'], 'donated' => $goal['donated'], 'start_date' => $goal['start_date'], 'end_date' => $goal['end_date'])
		);

		$dw = XenForo_DataWriter::create('robodn_DonationManager_DataWriter_Goal');
		$dw->setExistingData($goalId);
		$dw->set('donated', 0);
		$dw->set('start_date', XenForo_Application::$time);
		if ($goal['end_date'])
		{
			$dw->set('end_date', XenForo_Application::$time + ($goal['end_date'] - $goal['start_date']));
		}
		$dw->save();
	}

	public function archiveGoal($goalId)
	{
		$dw = XenForo_DataWriter::create('robodn_DonationManager_DataWriter_Goal');
		$dw->setExistingData($goalId);
		$dw->set('archived', 1);
		$dw->save();
	}
}