<?php

/**
 * Handles user upgrade processing with PayPal.
 *
 * @package robokassa
 */
class robokassa_roboproc
{
	/**
	 * @var Zend_Controller_Request_Http
	 */
	protected $_request;

	/**
	 * @var XenForo_Input
	 */
	protected $_input;

	/**
	 * List of filtered input for handling a callback.
	 *
	 * @var array
	 */
	protected $_filtered = null;
	
	/**
	 * Info about the upgrade being processed.
	 *
	 * @var array|false
	 */
	protected $_upgrade = false;

	/**
	 * Info about the user the upgrade is for.
	 *
	 * @var array|false
	 */
	protected $_user = false;

	/**
	 * The upgrade record ID inserted/updated.
	 *
	 * @var integer|null
	 */
	protected $_upgradeRecordId = null;

	/**
	 * The upgrade record being processed.
	 *
	 * @var array|false
	 */
	protected $_upgradeRecord = false;

	/**
	 * @var XenForo_Model_UserUpgrade
	 */
	protected $_upgradeModel = null;

	/**
	 * Initializes handling for processing a request callback.
	 *
	 * @param Zend_Controller_Request_Http $request
	 */
	public function initCallbackHandling(Zend_Controller_Request_Http $request)
	{
		$this->_request = $request;
		$this->_input = new XenForo_Input($request);

		$this->_filtered = $this->_input->filter(array(
			'InvId' => XenForo_Input::UINT,
			'OutSum' => XenForo_Input::STRING,
			'SignatureValue' => XenForo_Input::STRING,
			'Shp_item' => XenForo_Input::STRING,
		));

		$this->_upgradeModel =  XenForo_Model::create('XenForo_Model_UserUpgrade');
	}

	/**
	 * Validates the callback request is valid. If failure happens, the response should
	 * tell the processor to retry.
	 *
	 * @param string $errorString Output error string
	 *
	 * @return boolean
	 */
	public function validateRequest(&$errorString)
	{
		return true;
	}

	/**
	 * Validates pre-conditions on the callback. These represent things that likely wouldn't get fixed
	 * (and generally shouldn't happen), so retries are not necessary.
	 *
	 * @param string $errorString
	 *
	 * @return boolean
	 */
	public function validatePreConditions(&$errorString, $mrh_pass2)
	{
		$itemParts = explode(',', $this->_filtered['Shp_item'], 4);
		if (count($itemParts) != 4)
		{
			$errorString = 'Invalid item (Shp_item)';
			return false;
		}

		list($userId, $userUpgradeId, $validationType, $validation) = $itemParts;
		// $validationType allows validation method changes

		$user = XenForo_Model::create('XenForo_Model_User')->getFullUserById($userId);
		if (!$user)
		{
			$errorString = 'Invalid user';
			return false;
		}
		$this->_user = $user;

		$tokenParts = explode(',', $validation);
		if (count($tokenParts) != 3 || sha1($tokenParts[1] . $user['csrf_token']) != $tokenParts[2])
		{
			$errorString = 'Invalid validation';
			return false;
		}

		$upgrade = $this->_upgradeModel->getUserUpgradeById($userUpgradeId);
		if (!$upgrade)
		{
			$errorString = 'Invalid user upgrade';
			return false;
		}
		$this->_upgrade = $upgrade;

		if (!$this->_filtered['InvId'])
		{
			$errorString = 'No InvId';
			return false;
		}

		$transaction = $this->_upgradeModel->getProcessedTransactionLog($this->_filtered['InvId']);
		if ($transaction)
		{
			$errorString = 'Transaction already processed';
			return false;
		}

		$upgradeRecord = $this->_upgradeModel->getActiveUserUpgradeRecord($this->_user['user_id'], $this->_upgrade['user_upgrade_id']);
		if ($upgradeRecord)
		{
			$this->_upgradeRecordId = $upgradeRecord['user_upgrade_record_id'];
			$this->_upgradeRecord = $upgradeRecord;
		}

    //  
    // read parameters
    $out_summ = $this->_filtered["OutSum"];
    $inv_id = $this->_filtered['InvId'];
    $shp_item = $this->_filtered["Shp_item"];
    $crc = $this->_filtered["SignatureValue"];

    $crc = strtoupper($crc);

    $my_crc = strtoupper(md5("$out_summ:$inv_id:$mrh_pass2:Shp_item=$shp_item"));

    //   
    // check signature
    if ($my_crc !=$crc)
    {
      $errorString = 'Invalid Signature';
      return false;
    }

    //    
    // success

    if ($upgradeRecord)
    {
      $extra = unserialize($upgradeRecord['extra']);
      $cost = $extra['cost_amount'];
      $currency = $extra['cost_currency'];
    }
    else
    {
      $cost = $upgrade['cost_amount'];
      $currency = $upgrade['cost_currency'];
    }
    if (round($this->_filtered['OutSum'], 2) != round($cost, 2) )
    {
      $errorString = 'Invalid payment amount';
      return false;
    }

		return true;
	}

	/**
	 * Once all conditions are validated, process the transaction.
	 *
	 * @return array [0] => log type (payment, cancel, info), [1] => log message
	 */
	public function processTransaction()
	{
    $this->_upgradeRecordId = $this->_upgradeModel->upgradeUser($this->_user['user_id'], $this->_upgrade);

    return array('payment', 'Payment received, upgraded/extended');
	}

	/**
	 * Get details for use in the log.
	 *
	 * @return array
	 */
	public function getLogDetails()
	{
		$details = $_POST;
		$details['_callbackIp'] = (isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : false);

		return $details;
	}

	/**
	 * Gets the transaction ID.
	 *
	 * @return string
	 */
	public function getTransactionId()
	{
		return $this->_filtered['InvId'];
	}

	/**
	 * Gets the ID of the processor.
	 *
	 * @return string
	 */
	public function getProcessorId()
	{
		return 'robokassa';
	}

	/**
	 * Gets the ID of the upgrade record changed.
	 *
	 * @return integer
	 */
	public function getUpgradeRecordId()
	{
		return intval($this->_upgradeRecordId);
	}

	/**
	 * Logs the request.
	 *
	 * @param string $type Log type (info, payment, cancel, error)
	 * @param string $message Log message
	 * @param array $extra Extra details to log (not including output from getLogDetails)
	 */
	public function log($type, $message, array $extra)
	{
		$upgradeRecordId = $this->getUpgradeRecordId();
		$processor = $this->getProcessorId();
		$transactionId = $this->getTransactionId();
		$details = $this->getLogDetails() + $extra;

		$this->_upgradeModel->logProcessorCallback(
			$upgradeRecordId, $processor, $transactionId, $type, $message, $details
		);
	}
}